<?php

namespace Drupal\tapis_auth\TapisProvider;

/**
 * This interface outlines how this module interacts w/ Tapis.
 */
interface TapisTokenProviderInterface {
  // Seconds.
  public const ACCESS_TOKEN_TTL = 60 * 60 * 5;
  // Seconds.
  public const REFRESH_TOKEN_TTL = 60 * 60 * 24;

  /**
   * Get the decoded payload for a JWT access token.
   *
   * @param string $jwt
   *   A JWT access token string.
   *
   * @returns array the JWT's payload, as an associative array
   */
  public static function decodeJWT($jwt);

  /**
   * Checks if a Drupal user can authenticate w/ Tapis in the Tapis tenant.
   *
   * @param int $tenantId
   *   The Tapis tenant id.
   * @param int $uid
   *   The user id.
   */
  public function canUserAccessTapisServices($tenantId, $uid);

  /**
   * Gets a new Tapis access token for Drupal's authenticator.
   *
   * @param int $tenantId
   *   The Tapis tenant id.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *    Thrown when any API call to Tapis fails.
   *
   * @returns mixed A new Tapis access token for Drupal's authenticator.
   */
  public function getNewAuthenticatorServiceToken($tenantId);

  /**
   * Validate the authentication settings for a Tapis tenant.
   *
   * @param string $tapis_tenant_id
   *   The ID of the Tapis tenant.
   * @param string $tapis_site_id
   *   The ID of the Tapis site.
   * @param string $tapis_api_endpoint
   *   The API endpoint for the Tapis api service.
   * @param string $authenticator_service
   *   The authenticator service used for authentication.
   * @param string $authenticator_service_password
   *   The password for the authenticator service.
   *
   * @return array
   *   An array of validation results.
   *
   * @throws \Drupal\tapis_auth\Exception\TapisAuthException
   *   If there is an authentication exception.
   * @throws \GuzzleHttp\Exception\GuzzleException
   *   If there is a Guzzle HTTP exception.
   */
  public function validateTenantAuthSettings(
    $tapis_tenant_id,
    $tapis_site_id,
    $tapis_api_endpoint,
    $authenticator_service,
    $authenticator_service_password);

  /**
   * Get the Tapis service access token for Drupal.
   *
   * If a valid access token already exists, that will be returned.
   * If an access token doesn't exist yet, a new one will be created & returned.
   * If an access token already exists but it has expired,
   * it will be refreshed with a new access token & returned.
   *
   * @param int $tenantId
   *   The Tapis tenant id.
   *
   * @returns mixed A valid Tapis access token for Drupal's authenticator.
   */
  public function getAuthenticatorServiceToken($tenantId);

  /**
   * A helper function for getting a Drupal user's Tapis username.
   *
   * @param int $tenantId
   *   The tenant id.
   * @param int $uid
   *   The user id.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *    Thrown when the user can't use Tapis auth.
   *
   * @returns mixed The Drupal user's Tapis username
   */
  public function getTapisUsername($tenantId, $uid);

  /**
   * Get the Tapis access token for a Drupal user.
   *
   * If a valid access token exists, that will be returned.
   * If an access token doesn't exist yet, a new one will be created & returned.
   * If an access token already exists, but it has expired,
   * it will be refreshed with a new access token & returned.
   *
   * @param int $tenantId
   *   The Tapis tenant id.
   * @param int $uid
   *   The user id.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *    Thrown for unauthorized users.
   *
   * @returns mixed A valid Tapis access token for the user.
   */
  public function getUserToken($tenantId, $uid);

  /**
   * Get the Tapis access token for an integration test user.
   *
   * (used for integration testing with Tapis)
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $tapis_username
   *   The username of the integration test user.
   *
   * @return mixed
   *   The Tapis access token for the integration test user.
   */
  public function getIntegrationTestUserToken($tenantId, $tapis_username);

  /**
   * Get the Tapis service access token for Drupal.
   *
   * This is used for interacting w/ Tapis
   * using a site-wide Tapis service account.
   *
   * NOTE: This is NOT used in our current Drupal Tapis setup.
   * This was implemented at the very early stages of this project.
   *
   * If a valid service access token already exists, that will be returned.
   * If a service access token doesn't exist yet,
   * a new one will be created & returned.
   * If a service access token already exists but it has expired,
   * it will be refreshed with a new access token & returned.
   *
   * @param int $tenantId
   *   The Tapis tenant id.
   *
   * @returns mixed A valid Tapis service access token for Drupal.
   */
  public function getDrupalServiceToken($tenantId);

}
