<?php

namespace Drupal\tapis_auth\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface;
use Drupal\tapis_tenant\DrupalIds as TenantDrupalIds;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Class TapisIntegrationTestsController.
 *
 * Drupal controller that handles requests from the Tapis integration tests.
 */
class TapisIntegrationTestsController extends ControllerBase {

  /**
   * The Tapis token provider.
   *
   * @var \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface
   */
  protected TapisTokenProviderInterface $tapisTokenProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * TapisIntegrationTestsController constructor.
   *
   * @param \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface $tapisTokenProvider
   *   The Tapis token provider.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(TapisTokenProviderInterface $tapisTokenProvider,
                              EntityTypeManagerInterface $entityTypeManager,
                              ConfigFactoryInterface $config_factory) {
    $this->tapisTokenProvider = $tapisTokenProvider;
    $this->entityTypeManager = $entityTypeManager;
    $this->config = $config_factory->get('tapis_auth.config');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tapis_auth.tapis_token_provider'),
      $container->get('entity_type.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * Validate a request from the Tapis integration tests.
   */
  private function validateRequest(Request $request) {
    // $config = \Drupal::config('tapis_auth.config');
    if (!$this->config->get("enable_integration_tests")) {
      // Throw a 404 error if integration tests are not enabled.
      throw new NotFoundHttpException();
    }

    // Get the value of the 'X-Integration-Tests-Secret' header.
    $integration_tests_secret = $request->headers->get('X-Integration-Tests-Secret');

    // Get the key value for the key
    // with id = integration_tests_secret_key_id from the config.
    $integration_tests_secret_key_id = $this->config->get('integration_tests_secret_key_id');

    // Load the key entity.
    // $key_entity = \Drupal::entityTypeManager()->getStorage('key')->load($integration_tests_secret_key_id);
    $key_entity = $this->entityTypeManager->getStorage('key')->load($integration_tests_secret_key_id);
    // Get the key value.
    /** @var \Drupal\key\Entity\Key $key_entity */
    $key_value = $key_entity->getKeyValue();

    // If the header is not equal to the value of
    // the integration_tests_secret_key_id key,
    // or there is no integration tests secret key defined, return an error.
    if (!$key_value || $integration_tests_secret !== $key_value) {
      // Throw a 404 error.
      throw new NotFoundHttpException();
    }
  }

  /**
   * Get the integration test user info for integration testing.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   HTTP request.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getIntegrationTestUserInfo(Request $request): JsonResponse {
    // Validate this request.
    $this->validateRequest($request);

    // Get the tapis site/tenant id + user id from the request.
    $tapis_tenant_id = $request->get('tapis_tenant_id');
    $tapis_site_id = $request->get('tapis_site_id');
    $tapis_username = $request->get('tapis_username');

    // If any of them are not set, return an error.
    if (!$tapis_tenant_id || !$tapis_site_id || !$tapis_username) {
      // Throw a 404 error.
      throw new NotFoundHttpException();
    }

    // Get the node (bundle: tapis_tenant)
    // where the tapis tenant id matches the tapis_tenant_id.
    // $query = \Drupal::entityQuery('node')
    $query = $this->entityTypeManager->getStorage('node')->getQuery()
      ->condition('type', TenantDrupalIds::NODE_BUNDLE_TENANT)
      ->condition(TenantDrupalIds::TENANT_TAPIS_ID, $tapis_tenant_id)
      ->condition(TenantDrupalIds::TENANT_SITE_TAPIS_ID, $tapis_site_id)
      ->range(0, 1);

    $nids = $query->accessCheck(FALSE)->execute();

    // If there is no node, return an error.
    if (empty($nids)) {
      // Throw a 404 error.
      throw new NotFoundHttpException();
    }

    // Get the drupal tenant id.
    $tenant_id = array_shift($nids);

    // Write the tenant id to Drupal logger.
    \Drupal::logger('tapis_auth')->notice('Tenant ID: ' . $tenant_id . ', Tapis Tenant ID: ' . $tapis_tenant_id . ', Tapis Site ID: ' . $tapis_site_id . ', Tapis Username: ' . $tapis_username);

    // Get the tapis_auth.tapis_token_provider service.
    // $tapis_auth_provider =
    // \Drupal::service('tapis_auth.tapis_token_provider');.
    // Get the test user access/refresh tokens
    // for integration testing in this tapis site/tenant.
    $test_user_tokens = $this->tapisTokenProvider->getIntegrationTestUserToken($tenant_id, $tapis_username);

    // Return the test user access/refresh tokens as a json response.
    return new JsonResponse($test_user_tokens);
  }

}
