<?php

namespace Drupal\tapis_auth;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a list controller for the token entity type.
 */
class TapisTokenListBuilder extends EntityListBuilder {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The Tapis Token Provider service.
   *
   * @var \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface
   */
  private TapisTokenProviderInterface $tapisTokenProvider;

  /**
   * Constructs a new TapisTokenListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface $tapisTokenProvider
   *   The Tapis Token provider service.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, DateFormatterInterface $date_formatter, TapisTokenProviderInterface $tapisTokenProvider) {
    parent::__construct($entity_type, $storage);
    $this->dateFormatter = $date_formatter;
    $this->tapisTokenProvider = $tapisTokenProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('date.formatter'),
      $container->get(Constants::SERVICE_TAPIS_TOKEN_PROVIDER)
    );
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build['table'] = parent::render();

    $total = $this->getStorage()->getQuery()->accessCheck(FALSE)->count()->execute();

    $build['summary']['#markup'] = $this->t('Total tokens: @total', ['@total' => $total]);
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['id'] = $this->t('ID');
    // $header['Access token'] = $this->t('Access token');
    $header['tenant'] = $this->t("Tenant");
    $header['subject'] = $this->t('Subject');
    $header['type'] = $this->t('Type');
    $header['expiry'] = $this->t('Expiry');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var TapisTokenInterface $entity */
    $row['id'] = $entity->id();
    // $row['access_token'] = $entity->getAccessToken();
    try {
      $tenantInfo = $entity->getTenant();
      $row['tenant'] = $tenantInfo['tenant']->label();
    }
    catch (\Throwable $e) {
      $row['tenant'] = $this->t('Unknown (deleted tenant)');
    }
    catch (\Exception $e) {
      $row['tenant'] = $this->t('Unknown  (deleted tenant)');
    }

    if ($entity->getSubjectType() === "user") {
      $row['subject']['data'] = [
        '#theme' => 'username',
        '#account' => $entity->getOwner(),
      ];
    }
    else {
      $row['subject']['data'] = $entity->getService();
    }

    $row['type'] = $entity->getSubjectType();
    $row['expiry'] = $this->getTokenExpiry($entity);

    return $row + parent::buildRow($entity);
  }

  /**
   * Return the expiry date of the tapis token entity's access token.
   *
   * @param TapisTokenInterface $token
   *   The TapiToken entity.
   *
   * @returns string the token's expiry date as a formatted string.
   */
  private function getTokenExpiry(TapisTokenInterface $token) {
    $jwt = $this->tapisTokenProvider->decodeJWT($token->getAccessToken());
    return $this->dateFormatter->format($jwt->exp, "long");
  }

}
