<?php

namespace Drupal\tapis_job\Twig\Extension;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Provides custom Twig functions for tapis Job module.
 *
 * This extension adds custom Twig functions that can be used within templates
 * to interact with file entities and perform mathematical calculations.
 */
class TapisJobTwigExtension extends AbstractExtension {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a TapisJobTwigExtension object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Returns an array of Twig functions provided by this extension.
   *
   * @return \Twig\TwigFunction[]
   *   An array of TwigFunction objects representing the custom functions.
   */
  public function getFunctions() {
    return [
      new TwigFunction('file_entity', [$this, 'getFileEntity']),
      new TwigFunction('sinDipAngle', [$this, 'getSine']),
    ];
  }

  /**
   * Retrieves a file entity based on its ID.
   *
   * @param int $fid
   *   The file entity ID.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The loaded file entity object, or null if not found.
   */
  public function getFileEntity(int $fid): EntityInterface {
    return $this->entityTypeManager->getStorage('file')->load($fid);
  }

  /**
   * Calculates the sine of a given dip angle.
   *
   * Converts the dip angle from degrees to radians before calculating the sine.
   *
   * @param float $dip_angle
   *   The dip angle in degrees.
   *
   * @return float
   *   The sine of the dip angle.
   */
  public function getSine(float $dip_angle): float {
    return sin(deg2rad($dip_angle));
  }

}
