<?php

namespace Drupal\tapis_job;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a list controller for the job entity type.
 */
class TapisJobListBuilder extends EntityListBuilder {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new TapisJobListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(EntityTypeInterface $entity_type,
                              EntityStorageInterface $storage,
                              DateFormatterInterface $date_formatter,
                              EntityTypeManagerInterface $entityTypeManager) {
    parent::__construct($entity_type, $storage);
    $this->dateFormatter = $date_formatter;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('date.formatter'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build['table'] = parent::render();

    $total = $this->getStorage()->getQuery()->accessCheck(FALSE)->count()->execute();

    $build['summary']['#markup'] = $this->t('Total jobs: @total', ['@total' => $total]);
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['id'] = $this->t('ID');
    $header['status'] = $this->t('Status');
    $header['label'] = $this->t('Label');
    $header['tenant'] = $this->t("Tenant");
    $header['app'] = $this->t('App');
    $header['system'] = $this->t('System');
    $header['uid'] = $this->t('Author');
    $header['created'] = $this->t('Created');
    $header['changed'] = $this->t('Updated');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var TapisJobInterface $entity */

    $tenant = $entity->getTenant();

    // $app = \Drupal::entityTypeManager()->getStorage("node")->load($entity->getAppId());
    $app = $this->entityTypeManager->getStorage("node")->load($entity->getAppId());

    // $system = \Drupal::entityTypeManager()->getStorage("node")->load($entity->getSystemId());
    $system = $this->entityTypeManager->getStorage("node")->load($entity->getSystemId());

    $row['id'] = $entity->id();
    $row['status'] = $entity->getStatus();
    $row['label'] = $entity->toLink();
    $row['tenant'] = $tenant->toLink();

    if ($app) {
      $row['app'] = $app->toLink();
    }
    else {
      $row['app'] = 'n/a';
    }

    if ($system) {
      $row['system'] = $system->toLink();
    }
    else {
      $row['system'] = 'n/a';
    }
    $row['uid']['data'] = [
      '#theme' => 'username',
      '#account' => $entity->getOwner(),
    ];
    $row['created'] = $this->dateFormatter->format($entity->get('created')->value);
    $row['changed'] = $this->dateFormatter->format($entity->getChangedTime());
    return $row + parent::buildRow($entity);
  }

}
