<?php

namespace Drupal\tapis_job\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\tapis_app\DrupalIds as AppDrupalIds;
use Drupal\tapis_job\TapisJobInterface;
use Drupal\tapis_job\TapisProvider\TapisJobProviderInterface;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class JobCancelForm.
 *
 * This class is used to create a form that allows a user to cancel a job.
 *
 * @package Drupal\tapis_job\Form
 */
class JobCancelForm extends FormBase {

  /**
   * The Tapis site tenant provider.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */
  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Tapis job provider.
   *
   * @var \Drupal\tapis_job\TapisProvider\TapisJobProviderInterface
   */
  protected TapisJobProviderInterface $tapisJobProvider;

  /**
   * Constructs a new JobCancelForm object.
   *
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\tapis_job\TapisProvider\TapisJobProviderInterface $tapisJobProvider
   *   The Tapis job provider.
   */
  public function __construct(TapisSiteTenantProviderInterface $tapisSiteTenantProvider,
                              EntityTypeManagerInterface $entityTypeManager,
                              TapisJobProviderInterface $tapisJobProvider) {
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
    $this->entityTypeManager = $entityTypeManager;
    $this->tapisJobProvider = $tapisJobProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get("tapis_tenant.tapis_site_tenant_provider"),
      $container->get('entity_type.manager'),
      $container->get('tapis_job.tapis_job_provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'tapis_job_cancel_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, TapisJobInterface $tapis_job = NULL) {
    $form['#tree'] = TRUE;

    if (!$tapis_job) {
      return;
    }

    $node = $tapis_job->getApp();

    // Check if the app's Tapis tenant's Tapis site is in maintenance mode.
    // $tapisSiteTenantProvider =
    // \Drupal::service("tapis_tenant.tapis_site_tenant_provider");.
    $tenantId = $node->get(AppDrupalIds::APP_TENANT)->first()->getValue()['target_id'];
    if ($this->tapisSiteTenantProvider->isTenantInMaintenanceMode($tenantId)) {
      $form['message'] = [
        '#type' => 'markup',
        '#markup' => '<p>This job cannot be cancelled because its site is in maintenance mode.</p>',
      ];
      return $form;
    }

    $form_state->set("job_id", $tapis_job->id());

    $form['cancel_button'] = [
      '#type' => 'submit',
      '#value' => t('Cancel job'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $job_id = $form_state->get("job_id");

    /** @var \Drupal\tapis_job\TapisJobInterface $job */
    // $job = \Drupal::entityTypeManager()
    // ->getStorage('tapis_job')->load($job_id);
    $job = $this->entityTypeManager->getStorage('tapis_job')->load($job_id);
    $tenantId = $job->getTenantId();

    $jobOwnerId = $job->getOwnerId();

    // /** @var Drupal\tapis_job\TapisProvider\TapisJobProviderInterface */
    // $tapisJobProvider = \Drupal::service("tapis_job.tapis_job_provider");
    $this->tapisJobProvider->cancelJob($tenantId, $job->getTapisUUID(), $jobOwnerId);

    $app = $job->getApp();
    $appType = $app->get(AppDrupalIds::APP_TYPE)->getValue()[0]['value'];
    if ($appType === "web" || $appType === "vnc") {
      $this->tapisJobProvider->deleteAllAccessLinksForJob($job->id());
    }

    $form_state->setRedirect('entity.tapis_job.canonical', [
      'tapis_job' => $job->id(),
    ]);
  }

}
