<?php

namespace Drupal\tapis_job\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\tapis_job\TapisJobInterface;
use Drupal\tapis_job\TapisProvider\TapisJobProviderInterface;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class JobOutputForm.
 *
 * This class is used to create a form that displays a job's output files.
 *
 * @package Drupal\tapis_job\Form
 */
class JobOutputForm extends FormBase {

  /**
   * The Tapis site tenant provider.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */
  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The Tapis job provider.
   *
   * @var \Drupal\tapis_job\TapisProvider\TapisJobProviderInterface
   */
  protected TapisJobProviderInterface $tapisJobProvider;

  /**
   * The HTTP client used for making requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * Constructs a new JobOutputForm object.
   *
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   * @param \Drupal\tapis_job\TapisProvider\TapisJobProviderInterface $tapisJobProvider
   *   The Tapis job provider.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The http client.
   */
  public function __construct(TapisSiteTenantProviderInterface $tapisSiteTenantProvider,
                              TapisJobProviderInterface $tapisJobProvider,
                              ClientInterface $httpClient) {
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
    $this->tapisJobProvider = $tapisJobProvider;
    $this->httpClient = $httpClient;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get("tapis_tenant.tapis_site_tenant_provider"),
      $container->get('tapis_job.tapis_job_provider'),
      $container->get('http_client'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'tapis_job_output_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, TapisJobInterface $tapis_job = NULL) {
    if (!$tapis_job) {
      return;
    }

    if (!$this->currentUser()->hasPermission("create job")) {
      $form['message'] = [
        '#type' => 'markup',
        '#markup' => '<b>You do not have permission to view the job outputs.</b>',
      ];
      return $form;
    }

    // Check if the app's Tapis tenant's Tapis site is in maintenance mode.
    $tenantId = $tapis_job->getTenantId();
    if ($this->tapisSiteTenantProvider->isTenantInMaintenanceMode($tenantId)) {
      $form['message'] = [
        '#type' => 'markup',
        '#markup' => "<p>This job's output is currently unavailable because its site is in maintenance mode.</p>",
      ];
      return $form;
    }

    $jobOwnerId = $tapis_job->getOwnerId();

    $tapisJobDefinition = $this->tapisJobProvider->getJob($tenantId, $tapis_job->getTapisUUID(), $jobOwnerId);
    $execSystemId = $tapisJobDefinition['execSystemId'];
    $execSystemExecDir = $tapisJobDefinition['execSystemExecDir'];

    // $page = $form_state->get('page') ?: 0;
    // Number of items to display per page.
    // Set up max number as default: 1000.
    $limit = 1;
    $skip = 0;

    // Get the current page of output files for the job.
    $rescure = "false";
    $outputList = $this->tapisJobProvider->getJobOutputs($tenantId, $execSystemId, $execSystemExecDir, $rescure, $jobOwnerId, $skip, $limit);

    $form['job_output'] = [];

    if (!empty($outputList['result']) && count($outputList['result']) > 0) {
      $jobProxyURL = Url::fromRoute('entity.tapis_job.output_file',
        [
          'tapis_job' => $tapis_job->id(),
          'outputPath' => $execSystemExecDir,
          'mimeType' => "",
          'type' => "dir",
          'execSystemId' => $execSystemId,
        ]);

      // Define a hidden form element that holds the file URL.
      $form['job_output']['file_url'] = [
        '#type' => 'hidden',
        // This should be dynamically generated.
        '#value' => $jobProxyURL->toString(),
      ];

      // Define the download button.
      $form['job_output']['download_button'] = [
        '#type' => 'button',
        '#name' => 'download_file_btn',
        '#value' => t('Download All'),
        '#attributes' => [
          // Ensure the button has a unique ID.
          'id' => 'downloadButton',
        ],
      ];

      $form['job_output']['file_tree'] = [
        '#type' => 'markup',
        '#markup' => '<div id="osp_filetree"></div>',
      ];

      // Add the js library.
      $libraries = [
        'tapis_job/jstree',
        'tapis_job/tapis_job.file_tree',
        'tapis_job/tapis_job.font_awesome',
      ];
      $form['#attached']['library'] = $libraries;
      $form['#attached']['drupalSettings']['tapis_job']['tapis_job_id'] = $tapis_job->id();
      $form['#attached']['drupalSettings']['tapis_job']['exec_system_exec_dir'] = $execSystemExecDir;
      $config = $this->configFactory()->get('tapis_job.config');
      $supported_extensions = explode(" ", $config->get('supported_extensions'));
      $supported_image_extensions = explode(" ", $config->get('supported_image_extensions'));
      $form['#attached']['drupalSettings']['tapis_job']['supported_extensions'] = $supported_extensions;
      $form['#attached']['drupalSettings']['tapis_job']['supported_image_extensions'] = $supported_image_extensions;
    }
    else {
      $form['job_output']['job_output_message'] = [
        '#markup' => "<div>No output files were found for this job.</div>",
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function previousPageSubmit(array &$form, FormStateInterface $form_state): void {
    $page = $form_state->get('page') ?: 0;
    if ($page > 0) {
      $form_state->set('page', $page - 1);
    }
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function nextPageSubmit(array &$form, FormStateInterface $form_state): void {
    $page = $form_state->get('page') ?: 0;
    $form_state->set('page', $page + 1);
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  private function humanizeNumBytes($bytes) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB', 'PB'];
    for ($i = 0; $bytes > 1024; $i++) {
      $bytes /= 1024;
    }
    return round($bytes, 2) . ' ' . $units[$i];
  }

  /**
   * {@inheritdoc}
   */
  public function refreshCallback(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function refreshFormCallback(array &$form, FormStateInterface $form_state) {
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
  }

}
