<?php

namespace Drupal\tapis_job\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\tapis_job\JobAccessLinkInterface;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the job access link entity class.
 *
 * @ContentEntityType(
 *   id = "job_access_link",
 *   label = @Translation("Job access link"),
 *   label_collection = @Translation("Job access links"),
 *   label_singular = @Translation("job access link"),
 *   label_plural = @Translation("job access links"),
 *   label_count = @PluralTranslation(
 *     singular = "@count job access links",
 *     plural = "@count job access links",
 *   ),
 *   base_table = "job_access_link",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 * )
 */
class JobAccessLink extends ContentEntityBase implements JobAccessLinkInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * Tapis definition.
   *
   * @var string
   */
  protected string $tapisDefinition;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['job'] = BaseFieldDefinition::create('entity_reference')->setLabel(t('Job'))->setRequired(TRUE)->setSetting('target_type', 'tapis_job');

    $fields['proxyId'] = BaseFieldDefinition::create('string')->setLabel(t('Proxy ID'))->setRequired(FALSE)->setSetting('max_length', 255)->setDisplayConfigurable('form', FALSE)->setDisplayConfigurable('view', FALSE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Owner'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(static::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form',
        [
          'type' => 'entity_reference_autocomplete',
          'settings' =>
            [
              'match_operator' => 'CONTAINS',
              'size' => 60,
              'placeholder' => '',
            ],
          'weight' => 15,
        ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view',
        [
          'label' => 'above',
          'type' => 'author',
          'weight' => 15,
        ]
      )
      ->setDisplayConfigurable('view', TRUE);
    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the job was created.'))
      ->setDisplayOptions('view',
        [
          'label' => 'above',
          'type' => 'timestamp',
          'weight' => 20,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form',
        [
          'type' => 'datetime_timestamp',
          'weight' => 20,
        ]
      )
      ->setDisplayConfigurable('view', TRUE);
    $fields['changed'] = BaseFieldDefinition::create('changed')->setLabel(t('Changed'))->setDescription(t('The time that the job was last edited.'));
    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getProxyURL(): ?string {
    $proxyHost = $this->getProxyHost();
    if (!$proxyHost) {
      return NULL;
    }
    return "https://$proxyHost";
  }

  /**
   * {@inheritdoc}
   */
  public function getProxyHost(): ?string {
    $jobProxyID = $this->get("proxyId")->getValue()[0]['value'];
    if (!$jobProxyID) {
      return NULL;
    }
    $tenantId = $this->getJob()->getTenantId();
    $tenantInfo = \Drupal::service("tapis_tenant.tapis_site_tenant_provider")->getTenantInfo($tenantId);
    $site_apps_subdomain = $tenantInfo['apps_domain'];
    return "$jobProxyID.$site_apps_subdomain";
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getJobId() {
    return $this->get('job')->first()->getValue()['target_id'];
  }

  /**
   * {@inheritdoc}
   */
  public function getJob() {
    $field = "job";

    // Return $this->get($field)
    // ->first()->get('entity')->getTarget()->getValue();
    /** @var \Drupal\Core\Entity\Plugin\DataType\EntityReference $jobEntity */
    $jobEntity = $this->get($field)->first()->get('entity');
    return $jobEntity->getTarget()->getValue();
  }

  /**
   * {@inheritdoc}
   */
  public function setProxyId($proxyId): void {
    $this->set("proxyId", $proxyId);
  }

  /**
   * {@inheritdoc}
   */
  public function getProxyId() {
    return $this->get("proxyId")->value;
  }

}
