<?php

namespace Drupal\tapis_job\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\tapis_app\DrupalIds as AppDrupalIds;
use Drupal\tapis_job\TapisJobInterface;
use Drupal\tapis_job\TapisProvider\TapisJobProviderInterface;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class JobAccessLinksForm.
 *
 * This class is used to create a form that displays a job's access links.
 *
 * @package Drupal\tapis_job\Form
 */
class JobAccessLinksForm extends FormBase {

  /**
   * The Tapis site tenant provider.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */
  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Tapis job provider.
   *
   * @var \Drupal\tapis_job\TapisProvider\TapisJobProviderInterface
   */
  protected TapisJobProviderInterface $tapisJobProvider;

  /**
   * Constructs a new JobAccessLinksForm object.
   *
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\tapis_job\TapisProvider\TapisJobProviderInterface $tapisJobProvider
   *   The Tapis job provider.
   */
  public function __construct(TapisSiteTenantProviderInterface $tapisSiteTenantProvider,
                              EntityTypeManagerInterface $entityTypeManager,
                              TapisJobProviderInterface $tapisJobProvider) {
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
    $this->entityTypeManager = $entityTypeManager;
    $this->tapisJobProvider = $tapisJobProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get("tapis_tenant.tapis_site_tenant_provider"),
      $container->get('entity_type.manager'),
      $container->get('tapis_job.tapis_job_provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'tapis_job_access_links_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, TapisJobInterface $tapis_job = NULL) {
    if (!$tapis_job) {
      return;
    }

    $app = $tapis_job->getApp();
    $appType = $app->get(AppDrupalIds::APP_TYPE)->getValue()[0]['value'];

    if ($appType !== "web" && $appType !== "vnc") {
      $form['job_access_links']['message'] = [
        '#type' => 'markup',
        '#markup' => $this->t('Access links are only available for Web & VNC apps.'),
      ];
      return $form;
    }

    // Check if the app's Tapis tenant's Tapis site is in maintenance mode.
    // $tapisSiteTenantProvider =
    // \Drupal::service("tapis_tenant.tapis_site_tenant_provider");.
    $tenantId = $app->get(AppDrupalIds::APP_TENANT)->first()->getValue()['target_id'];
    if ($this->tapisSiteTenantProvider->isTenantInMaintenanceMode($tenantId)) {
      $form['message'] = [
        '#type' => 'markup',
        '#markup' => "<p>This job's access links are currently unavailable, because its site is in maintenance mode.</p>",
      ];
      return $form;
    }

    // /** @var Drupal\tapis_job\TapisProvider\TapisJobProviderInterface */
    // $tapisJobProvider =
    // \Drupal::service("tapis_job.tapis_job_provider");
    $tenantId = $tapis_job->getTenantId();
    $jobOwnerId = $tapis_job->getOwnerId();

    // Only show a button for generating a new access link
    // if the job is running.
    if ($this->tapisJobProvider->getJobStatus($tenantId, $tapis_job->getTapisUUID(), $jobOwnerId) !== "RUNNING") {
      $form['job_access_links']['message'] = [
        '#type' => 'markup',
        '#markup' => $this->t('Access links are only available for jobs that are currently running.'),
      ];
      return $form;
    }

    $form_state->set("job_id", $tapis_job->id());

    $form['#tree'] = TRUE;
    $form['job_access_links']['#type'] = 'container';
    $form['job_access_links']['#prefix'] = '<div id="job-access-links-table-wrapper">';
    $form['job_access_links']['#suffix'] = '</div>';

    /** @var \Drupal\tapis_job\Entity\TapisJob $tapis_job */
    $jobAccessLinks = $this->tapisJobProvider->getJobAccessLinksForJob($tapis_job);

    $rows = [];

    $form['job_access_links']['generate_access_link'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add app session link'),
      '#submit' => ['::generateNewAccessLink'],
      '#ajax' => [
        'wrapper' => "job-access-links-table-wrapper",
        'callback' => '::generateNewAccessLink_AjaxCallback',
      ],
    ];

    $form['job_access_links']['table'] = [
      '#type' => 'table',
      '#title' => 'Access links',
      '#header' => ['link' => 'App session link', 'delete' => 'Operations'],
    ];

    if ($jobAccessLinks) {

      foreach ($jobAccessLinks as $jobAccessLink) {
        $jobAccessLinkURLString = $jobAccessLink->getProxyURL();
        $jobAccessURL = Url::fromUri($jobAccessLinkURLString);

        $link = new Link($jobAccessLinkURLString, $jobAccessURL);

        $row_no = $jobAccessLink->id();

        $form['job_access_links']['table'][$row_no]['link'] = [
          '#markup' => $link->toString(),
        ];

        $form['job_access_links']['table'][$row_no]['delete'] = [
          '#type' => 'submit',
          '#name' => $row_no,
          '#value' => $this->t('Delete'),
          '#submit' => ['::removeJobAccessLink'],
          '#ajax' => [
            'wrapper' => 'job-access-links-table-wrapper',
            'callback' => '::removeJobAccessLink_AjaxCallback',
          ],
        ];
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function generateNewAccessLink_AjaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['job_access_links'];
  }

  /**
   * {@inheritdoc}
   */
  public function removeJobAccessLink_AjaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['job_access_links'];
  }

  /**
   * Generate a new access link for the job.
   */
  public function generateNewAccessLink(array &$form, FormStateInterface $form_state) {
    $job_id = $form_state->get("job_id");

    /** @var \Drupal\tapis_job\Entity\TapisJob $job */
    // $job = \Drupal::entityTypeManager()
    // ->getStorage('tapis_job')->load($job_id);
    $job = $this->entityTypeManager->getStorage('tapis_job')->load($job_id);
    $tenantId = $job->getTenantId();

    // /** @var Drupal\tapis_job\TapisProvider\TapisJobProviderInterface */
    // $tapisJobProvider = \Drupal::service("tapis_job.tapis_job_provider");
    $this->tapisJobProvider->createNewAccessLink($tenantId, $job);

    $form_state->setRebuild();
  }

  /**
   * Remove an access link for the job.
   */
  public function removeJobAccessLink(array &$form, FormStateInterface $form_state) {
    $job_id = $form_state->get("job_id");

    /** @var \Drupal\tapis_job\Entity\TapisJob $job */
    // $job = \Drupal::entityTypeManager()
    // ->getStorage('tapis_job')->load($job_id);
    $job = $this->entityTypeManager->getStorage('tapis_job')->load($job_id);

    $job_access_link_id_to_remove = $form_state->getTriggeringElement()['#name'];

    // /** @var Drupal\tapis_job\TapisProvider\TapisJobProviderInterface */
    // $tapisJobProvider = \Drupal::service("tapis_job.tapis_job_provider");
    $this->tapisJobProvider->deleteAccessLink($job_access_link_id_to_remove);

    // If the access link that was deleted was the last one,
    // then create a new one.
    $jobAccessLinks = $this->tapisJobProvider->getJobAccessLinksForJob($job);
    if (!$jobAccessLinks) {
      $tenantId = $job->getTenantId();
      $this->tapisJobProvider->createNewAccessLink($tenantId, $job);
    }

    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
  }

}
