<?php

namespace Drupal\tapis_job\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class SettingsForm.
 *
 * Configure the Tapis Job module's settings.
 *
 * @package Drupal\tapis_job\Form
 */
class SettingsForm extends ConfigFormBase {
  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'tapis_job.config';

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'tapis_job_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::SETTINGS);
    $enabled_access_links = $config->get("enabled_access_links");
    $webhook_jwt_secret_key_id = $config->get("webhook_jwt_secret_key_id");
    $upload_files_to_tapis = $config->get("upload_files_to_tapis");

    $form['enabled_access_links'] = [
      '#type' => 'checkbox',
      '#default_value' => $enabled_access_links,
      '#title' => $this->t('Share app sessions'),
      '#description' => $this->t("When enabled, users will be able to share their app sessions for VNC and web apps."),
    ];

    // Entity reference to drupal key.
    // '#default_value' => $webhook_jwt_secret_key_id ?
    // \Drupal::entityTypeManager()
    // ->getStorage('key')->load($webhook_jwt_secret_key_id) : NULL,.
    $form['webhook_jwt_secret_key_id'] = [
      '#type' => 'entity_autocomplete',
      '#target_type' => 'key',
      '#title' => $this->t('Webhook JWT Secret Key'),
      '#description' => $this->t("The key used to sign the JWT for all Tapis Webhook requests."),
      '#default_value' => $webhook_jwt_secret_key_id ? $this->entityTypeManager->getStorage('key')->load($webhook_jwt_secret_key_id) : NULL,
    ];

    $form['upload_files_to_tapis'] = [
      '#type' => 'checkbox',
      '#default_value' => $upload_files_to_tapis,
      '#title' => $this->t('Upload files to Tapis'),
      '#description' => $this->t("When enabled, job input files will be uploaded to the execution system via Tapis before submitting the job. If false, Tapis will download the input files from this site to the execution host on-demand, once the job is scheduled on the system. For optimal performance, this should be unchecked."),
    ];

    $supported_extensions = $config->get('supported_extensions');
    $form['supported_extensions'] = [
      '#title' => 'Supported Extensions',
      '#type' => 'textarea',
      '#default_value' => $supported_extensions,
    ];

    $supported_image_extensions = $config->get('supported_image_extensions');
    $form['supported_image_extensions'] = [
      '#title' => 'Supported Image Extensions',
      '#type' => 'textarea',
      '#default_value' => $supported_image_extensions,
    ];

    $supported_mimes = $config->get('supported_mimes');
    $form['supported_mimes'] = [
      '#title' => 'Supported MIME Types',
      '#type' => 'textarea',
      '#default_value' => $supported_mimes,
    ];

    $supported_image_mimes = $config->get('supported_image_mimes');
    $form['supported_image_mimes'] = [
      '#title' => 'Supported Image MIME Types',
      '#type' => 'textarea',
      '#default_value' => $supported_image_mimes,
    ];

    // Ensure prism_css is an array, defaulting to an empty array if not.
    $prism_css = $config->get('prism_css') ?? [];
    $form['prism_css'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Prism CSS URLs'),
      '#description' => $this->t('Enter one URL per line.'),
      '#default_value' => implode("\n", $prism_css),
    ];

    // Ensure prism_js is an array, defaulting to an empty array if not.
    $prism_js = $config->get('prism_js') ?? [];
    $form['prism_js'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Prism JS URLs'),
      '#description' => $this->t('Enter one URL per line.'),
      '#default_value' => implode("\n", $prism_js),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(
    array &$form,
    FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $webhook_jwt_secret_key_id = $form_state->getValue("webhook_jwt_secret_key_id");
    if ($webhook_jwt_secret_key_id) {
      // $jwt_secret_key = \Drupal::entityTypeManager()->getStorage('key')->load($webhook_jwt_secret_key_id);
      /** @var \Drupal\key\Entity\Key $jwt_secret_key */
      $jwt_secret_key = $this->entityTypeManager->getStorage('key')->load($webhook_jwt_secret_key_id);
      /** @var \Drupal\key\Plugin\KeyPluginInterface $jwt_keytype */
      $jwt_keytype = $jwt_secret_key->getKeyType();
      $jwt_plugin_id = $jwt_keytype->getPluginId();
      // Ensure that $jwt_plugin_id starts with 'jwt_'.
      if (strpos($jwt_plugin_id, 'jwt_') !== 0) {
        $form_state->setErrorByName("webhook_jwt_secret_key_id", $this->t("The selected key's type is not a JWT key."));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->getEditable(static::SETTINGS);
    $config->set("enabled_access_links", $form_state->getValue("enabled_access_links"));
    $config->set("webhook_jwt_secret_key_id", $form_state->getValue("webhook_jwt_secret_key_id"));
    $config->set("upload_files_to_tapis", $form_state->getValue("upload_files_to_tapis"));
    $config->set('supported_extensions', $form_state->getValue('supported_extensions'));
    $config->set('supported_image_extensions', $form_state->getValue('supported_image_extensions'));
    $config->set('supported_mimes', $form_state->getValue('supported_mimes'));
    $config->set('supported_image_mimes', $form_state->getValue('supported_image_mimes'));
    $config->set('prism_css', explode("\n", $form_state->getValue('prism_css')));
    $config->set('prism_js', explode("\n", $form_state->getValue('prism_js')));

    $config->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::SETTINGS,
    ];
  }

}
