<?php

namespace Drupal\tapis_job;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Defines the access control handler for the job entity type.
 */
class TapisJobAccessControlHandler extends EntityAccessControlHandler {

  // @todo Edit the permission names here if you ever change the permission machine names in permissions.yml.

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    /** @var \Drupal\tapis_job\TapisJobInterface $entity */
    switch ($operation) {
      case 'view':
        // True if admin or (you are job owner
        // and you have the "view own job" permission)
        if ($entity->getOwnerId() === $account->id() && $account->hasPermission('view own job')) {
          return AccessResult::allowed()->cachePerUser();
        }
        return AccessResult::allowedIfHasPermissions(
          $account,
          ['administer job', 'view any job'],
          'OR',
        )->cachePerUser();

      case 'update':
        // True if admin or (you are job owner
        // and you have the "edit own job" permission)
        if ($entity->getOwnerId() === $account->id() && $account->hasPermission('edit own job')) {
          return AccessResult::allowed()->cachePerUser();
        }
        return AccessResult::allowedIfHasPermissions(
          $account,
          ['administer job', 'edit any job'],
          'OR',
        )->cachePerUser();

      case 'delete':
        // True if admin or (you are job owner
        // and you have the "delete own job" permission)
        if ($entity->getOwnerId() === $account->id() && $account->hasPermission('delete own job')) {
          return AccessResult::allowed()->cachePerUser();
        }
        return AccessResult::allowedIfHasPermissions(
          $account,
          ['administer job', 'delete any job'],
          'OR',
        )->cachePerUser();

      case 'cancel':
        // True if admin or (you are job owner
        // and you have the "cancel own job" permission)
        if ($entity->getOwnerId() === $account->id() && $account->hasPermission('cancel own job')) {
          return AccessResult::allowed()->cachePerUser();
        }
        return AccessResult::allowedIfHasPermissions(
          $account,
          ['administer job', 'cancel any job'],
          'OR',
        )->cachePerUser();

      default:
        // No opinion.
        return AccessResult::forbidden();
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    return AccessResult::allowedIfHasPermissions(
      $account,
      ['create job', 'administer job'],
      'OR',
    );
  }

}
