<?php

namespace Drupal\tapis_job\TapisProvider;

use Drupal\tapis_job\Entity\TapisJob;

/**
 * Interface TapisJobProviderInterface.
 *
 * Provides methods for interacting with Tapis jobs.
 */
interface TapisJobProviderInterface {

  /**
   * Backup a job's files before restarting it.
   *
   * Such as the outputs/ospjob.log, outputs/tapisjob.out,
   * tapisjob.sh, ospjob_proxy_info.txt, tapisjob.env.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $systemTapisId
   *   The Tapis ID of the system.
   * @param string $originalJobUUID
   *   The UUID of the original job.
   * @param string $originalJobExecDir
   *   The execution directory of the original job.
   * @param int $uid
   *   The user ID.
   */
  public function backupJobFilesBeforeRestart($tenantId, $systemTapisId, $originalJobUUID, $originalJobExecDir, $uid = -1);

  /**
   * Generate a JWT for a job.
   *
   * This is used to verify if certain requests are sent from Tapis,
   *  such as for:
   *  - job webhook updates,
   *  - requests to download input files for a job, etc.)
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param array $claims
   *   to include in the JWT.
   */
  public function generateJWTForJob($tenantId, array $claims);

  /**
   * Subscribe to job event webhooks from Tapis.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job.
   * @param int $uid
   *   The user ID.
   */
  public function subscribeToJobEvents($tenantId, $jobUuid, $uid = -1);

  /**
   * Upload a file to a system.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $systemTapisId
   *   The Tapis ID of the system.
   * @param string $filepath
   *   The path of the file to upload.
   * @param string $file_content
   *   The content of the file.
   * @param int $uid
   *   The user ID.
   * @param int $systemOwnerUid
   *   The owner UID of the system.
   */
  public function uploadFileToSystem($tenantId, $systemTapisId, $filepath, $file_content, int $uid = -1, int $systemOwnerUid = -1);

  /**
   * Submit a job to Tapis.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param array $tapisDefinition
   *   The definition of the job.
   * @param int $uid
   *   The user ID.
   */
  public function submitJob($tenantId, array $tapisDefinition, $uid = -1);

  /**
   * Get a job from Tapis.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job.
   * @param int $uid
   *   The user ID.
   */
  public function getJob($tenantId, $jobUuid, $uid = -1);

  /**
   * Get a job's history from Tapis.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job.
   * @param int $uid
   *   The user ID.
   */
  public function getJobHistory($tenantId, $jobUuid, $uid = -1);

  /**
   * Get a job's status from Tapis.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job.
   * @param int $uid
   *   The user ID.
   */
  public function getJobStatus($tenantId, $jobUuid, $uid = -1);

  /**
   * Cancel a job in Tapis.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job to cancel.
   * @param int $uid
   *   The user ID.
   */
  public function cancelJob($tenantId, $jobUuid, $uid = -1);

  /**
   * Get a job's output file list from Tapis via the Files API.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job.
   * @param string $outputPath
   *   The path to the output files.
   * @param int $jobOwnerUid
   *   The user ID of the job owner.
   * @param int $systemOwnerUid
   *   The user ID of the system owner.
   */
  public function getJobSystemOutputList($tenantId, $jobUuid, $outputPath = "", $jobOwnerUid = -1, $systemOwnerUid = -1);

  /**
   * Get a job's output file list from Tapis via the Jobs API.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job.
   * @param string $outputPath
   *   The path to the output files.
   * @param int $uid
   *   The user ID.
   * @param int $skip
   *   The skip.
   * @param int $limit
   *   The limit.
   */
  public function getJobOutputList($tenantId, $jobUuid, $outputPath = "", $uid = -1, $skip = -1, $limit = -1);

  /**
   * Get a job's output file list from Tapis via the Jobs API.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $execSystemId
   *   The system ID of the job.
   * @param string $outputPath
   *   The path to the output files.
   * @param string $recurse
   *   The recursive status.
   * @param int $uid
   *   The user ID.
   * @param int $skip
   *   The skip.
   * @param int $limit
   *   The limit.
   */
  public function getJobOutputs($tenantId, $execSystemId, $outputPath = "", $recurse = "true", $uid = -1, $skip = -1, $limit = -1);

  /**
   * Get a job's output file from Tapis via the Jobs API.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job.
   * @param string $execSystemId
   *   The system ID of the job.
   * @param string $outputPath
   *   The path to the output file.
   * @param bool $zip
   *   Whether to compress the output file.
   * @param int $uid
   *   The user ID.
   */
  public function getJobOutputFileDownload($tenantId, $jobUuid, $execSystemId, $outputPath = "", $zip = FALSE, $uid = -1);

  /**
   * Get a job's output file from Tapis via the Jobs API.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job.
   * @param string $outputPath
   *   The path to the output file.
   * @param bool $compress
   *   Whether to compress the output file.
   * @param int $uid
   *   The user ID.
   */
  public function getJobOutputDownload($tenantId, $jobUuid, $outputPath = "", $compress = FALSE, $uid = -1);

  /**
   * Gets a new token from Satellite.
   *
   * @param int $tenantId
   *   The ID of the Tapis tenant.
   *
   * @return mixed
   *   The result of creating the token.
   */
  public function createNewSatelliteToken($tenantId);

  /**
   * Redeems a token given by satellite with the job's revssh container.
   *
   * This method redeems a token obtained from Satellite
   * for use with the job's revssh container.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $jobUuid
   *   The UUID of the job.
   * @param string $token
   *   The token to redeem.
   * @param int $jobOwnerUid
   *   (Optional) The user ID of the job owner. Defaults to -1.
   * @param int $systemOwnerUid
   *   (Optional) The user ID of the system owner. Defaults to -1.
   *
   * @return mixed
   *   The result of redeeming the token.
   */
  public function redeemSatelliteToken($tenantId, $jobUuid, $token, int $jobOwnerUid = -1, int $systemOwnerUid = -1);

  /**
   * Deletes a token given by satellite.
   *
   * @param $tenantId
   *   The ID of the Tapis tenant.
   * @param $token
   *   The token to delete.
   * @param $jobUuid
   *   The UUID of the job.
   * @param int $uid
   *   The user ID.
   *
   * @return mixed
   */

  /**
   * Deletes a token given by satellite.
   *
   * This method deletes a token given by satellite,
   * both from the satellite proxy and the job's revssh container.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $token
   *   The token to delete.
   * @param string|null $jobUuid
   *   (Optional) The UUID of the job. Defaults to NULL.
   * @param int $jobOwnerUid
   *   (Optional) The user ID of the job owner. Defaults to -1.
   * @param int $systemOwnerUid
   *   (Optional) The user ID of the system owner. Defaults to -1.
   *
   * @return mixed
   *   The result of deleting the token.
   */
  public function deleteSatelliteToken($tenantId, $token, $jobUuid = NULL, $jobOwnerUid = -1, $systemOwnerUid = -1);

  /**
   * Create a new access link for a job.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param \Drupal\tapis_job\Entity\TapisJob $job
   *   The Tapis job entity.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function createNewAccessLink($tenantId, TapisJob $job, $uid = -1);

  /**
   * Create an access link for a job for an existing proxy id.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param \Drupal\tapis_job\Entity\TapisJob $job
   *   The Tapis job entity.
   * @param int $proxyId
   *   The proxy ID.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function createAccessLink($tenantId, TapisJob $job, $proxyId, $uid = -1);

  /**
   * Get all access links for a job.
   *
   * @param \Drupal\tapis_job\Entity\TapisJob $job
   *   The Tapis job entity.
   */
  public function getJobAccessLinksForJob(TapisJob $job);

  /**
   * Get an access link for a job.
   *
   * @param \Drupal\tapis_job\Entity\TapisJob $job
   *   The Tapis job entity.
   */
  public function getAJobAccessLinkForJob(TapisJob $job);

  /**
   * Delete an access link for a job.
   *
   * @param int $jobAccessLinkId
   *   The access link ID.
   */
  public function deleteAccessLink($jobAccessLinkId);

  /**
   * Delete all access links for a job.
   *
   * @param int $jobId
   *   The ID of the job.
   */
  public function deleteAllAccessLinksForJob($jobId);

  /**
   * Get all job names.
   *
   * @param int $uid
   *   The user ID.
   */
  public function getAllJobNames($uid = -1);

  /**
   * Generate a unique job name for a user.
   *
   * @param string $userJobName
   *   The user-provided job name.
   */
  public function generateUniqueJobName($userJobName);

  /**
   * Check the access permission for tapis system and app.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The app content interface.
   */
  public function checkAccessForAppAndSystem($node);

}
