<?php

namespace Drupal\tapis_system\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\tapis_system\DrupalIds;
use Drupal\tapis_system\TapisSystemCredentialInterface;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the system credential entity class.
 *
 * @ContentEntityType(
 *   id = "tapis_system_credential",
 *   label = @Translation("Compute/Storage System Credential"),
 *   label_collection = @Translation("Compute/Storage System Credentials"),
 *   label_singular = @Translation("compute/storage system credential"),
 *   label_plural = @Translation("compute/storage system credentials"),
 *   label_count = @PluralTranslation(
 *     singular = "@count system credential",
 *     plural = "@count system credentials",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\tapis_system\TapisSystemCredentialListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "access" = "Drupal\tapis_system\TapisSystemCredentialAccessControlHandler",
 *     "form" = {
 *       "add" = "Drupal\tapis_system\Form\TapisSystemCredentialForm",
 *       "edit" = "Drupal\tapis_system\Form\TapisSystemCredentialForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     }
 *   },
 *   base_table = "tapis_system_credential",
 *   admin_permission = "administer tapis system",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 *   links = {
 *     "collection" = "/admin/content/tapis-system-credential",
 *     "add-form" = "/tapis/system-credential/add",
 *     "canonical" = "/tapis/system-credential/{tapis_system_credential}",
 *     "edit-form" = "/tapis/system-credential/{tapis_system_credential}/edit",
 *     "delete-form" = "/tapis/system-credential/{tapis_system_credential}/delete",
 *   },
 * )
 */
class TapisSystemCredential extends ContentEntityBase implements TapisSystemCredentialInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  protected $tapisDefinition;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['system'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('System'))
      ->setRequired(TRUE)
      ->setSetting('target_type', 'node')
      ->setSetting('handler', 'default')
      ->setSetting('handler_settings',
        [
          'target_bundles' => [DrupalIds::NODE_BUNDLE_SYSTEM => DrupalIds::NODE_BUNDLE_SYSTEM],
        ]
      )
      ->setDisplayOptions('form',
        [
          'type' => 'options_select',
          'weight' => 0,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view',
        [
          'label' => 'inline',
          'type' => 'entity_reference_label',
          'weight' => 0,
        ]
      )
      ->setDisplayConfigurable('view', TRUE);
    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Nickname'))
      ->setDescription(t("Set a label for this system credential"))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form',
        [
          'label' => 'inline',
          'type' => 'string',
          'weight' => 1,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view',
        [
          'label' => 'hidden',
          'type' => 'string',
          'weight' => 1,
        ]
      )
      ->setDisplayConfigurable('view', TRUE);
    $fields['loginUser'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Username'))
      ->setDescription(t("Set the username for this system credential"))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form',
        [
          'label' => 'inline',
          'type' => 'string',
          'weight' => 2,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view',
        [
          'label' => 'inline',
          'type' => 'string',
          'weight' => 2,
        ]
      )
      ->setDisplayConfigurable('view', TRUE);
    $fields['publicKey'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Public key'))
      ->setDescription("Enter the corresponding public key for this system credential.")
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayOptions('view',
        [
          'type' => 'text_default',
          'label' => 'above',
          'weight' => 3,
        ]
      )
      ->setDisplayConfigurable('view', TRUE);
    $fields['description'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Description'))
      ->setDisplayOptions('form',
        [
          'type' => 'text_textarea',
          'weight' => 6,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view',
        [
          'type' => 'text_default',
          'label' => 'above',
          'weight' => 4,
        ]
      )
      ->setDisplayConfigurable('view', TRUE);
    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('User'))
      ->setRequired(TRUE)
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(static::class . '::getDefaultEntityOwner');
    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the system credential was created.'))
      ->setDisplayOptions('view',
        [
          'label' => 'above',
          'type' => 'timestamp',
          'weight' => 5,
        ]
      )
      ->setDisplayConfigurable('view', TRUE);
    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the system credential was last edited.'))
      ->setDisplayOptions('view',
        [
          'label' => 'above',
          'type' => 'timestamp',
          'weight' => 6,
        ]
      )
      ->setDisplayConfigurable('view', TRUE);
    return $fields;
  }

  /**
   * Set the public key for this system credential.
   *
   * @param string $publicKey
   *   The public key.
   */
  public function setPublicKey($publicKey) {
    $this->set("publicKey", $publicKey);
  }

  /**
   * Get the Tapis definition for this system credential.
   */
  public function getTapisDefinition() {
    return $this->tapisDefinition;
  }

  /**
   * Set the Tapis definition for this system credential.
   *
   * @param array $definition
   *   The Tapis definition.
   */
  public function setTapisDefinition(array $definition) {
    $this->tapisDefinition = $definition;
  }

  /**
   * Get the system id for this system credential.
   *
   * @return int
   *   The system ID.
   */
  public function getSystemId() {
    return $this->get('system')->first()->getValue()['target_id'];
  }

  /**
   * Get the public key for this system credential.
   *
   * @return string
   *   The public key.
   */
  public function getPublicKey() {
    return $this->get('publicKey')->getValue()[0]['value'];
  }

}
