<?php

namespace Drupal\tapis_system\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\node\NodeInterface;
use Drupal\tapis_system\DrupalIds;
use Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface;
use Drupal\tapis_tenant\Exception\TapisException;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Class TapisSystemChangeOwnerForm.
 *
 * Form for changing the owner of a Tapis System.
 */
class TapisSystemChangeOwnerForm extends FormBase {

  /**
   * The Tapis site tenant provider.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */
  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The Tapis system provider.
   *
   * @var \Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface
   */
  protected TapisSystemProviderInterface $tapisSystemProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new AppLaunchForm object.
   *
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   * @param \Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface $tapisSystemProvider
   *   The Tapis system provider.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(TapisSiteTenantProviderInterface $tapisSiteTenantProvider,
                              TapisSystemProviderInterface $tapisSystemProvider,
                              EntityTypeManagerInterface $entityTypeManager) {
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
    $this->entityTypeManager = $entityTypeManager;
    $this->tapisSystemProvider = $tapisSystemProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get("tapis_tenant.tapis_site_tenant_provider"),
      $container->get('tapis_system.tapis_system_provider'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'tapis_system_changeowner_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL) {
    $form['#tree'] = TRUE;

    if (!$node) {
      return;
    }

    if ($node->bundle() !== DrupalIds::NODE_BUNDLE_SYSTEM) {
      throw new NotFoundHttpException();
    }

    // Throw a not found exception if the system's owner
    // is NOT the current user or a user with the admin permission.
    // $currentUser = \Drupal::currentUser();
    $currentUser = $this->currentUser();
    $userPermissionForSystemAdmin = $currentUser->hasPermission('administer tapis system');
    $userPermissionForEditAnyContent = $currentUser->hasPermission('edit any tapis_system content');
    if (!$userPermissionForSystemAdmin && !$userPermissionForEditAnyContent &&
      $currentUser->id() !== $node->getOwnerId()) {
      throw new NotFoundHttpException();
    }

    // Check if the system's Tapis tenant's Tapis site
    // is in maintenance mode.
    // $tapisSiteTenantProvider =
    // \Drupal::service("tapis_tenant.tapis_site_tenant_provider");.
    $tenantId = $node->get(DrupalIds::SYSTEM_TENANT)->first()->getValue()['target_id'];
    if ($this->tapisSiteTenantProvider->isTenantInMaintenanceMode($tenantId)) {
      $form['message'] = [
        '#type' => 'markup',
        '#markup' => "<p>This system's owner cannot be changed because its tenant is in maintenance mode.</p>",
      ];
      return $form;
    }

    $form_state->set("system_id", $node->id());

    $system_label = $node->label();

    $form['message'] = [
      '#type' => 'markup',
      '#markup' => "<p>Note: This operation will change $system_label's owner to be the selected user below. Proceed with caution.</p>",
    ];

    // A field to select a Drupal user to be the new System owner.
    $form['new_owner'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('New system owner'),
      '#target_type' => 'user',
      '#required' => TRUE,
      '#description' => $this->t('Select the new owner for this system.'),
      '#default_value' => $node->getOwner(),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => t('Change owner'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Make sure that the selected user is not the same
    // as the system's current owner id.
    $systemId = $form_state->get("system_id");
    // $system = \Drupal::entityTypeManager()
    // ->getStorage('node')->load($systemId);
    $system = $this->entityTypeManager->getStorage('node')->load($systemId);
    $currentOwnerId = $system->getOwnerId();
    $newOwnerUid = $form_state->getValue('new_owner');
    if ($currentOwnerId === $newOwnerUid) {
      $form_state->setErrorByName('new_owner', $this->t('The new owner cannot be the same as the current owner.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $system_id = $form_state->get("system_id");

    // /** @var \Drupal\node\NodeInterface $job */
    // $system = \Drupal::entityTypeManager()
    // ->getStorage('node')->load($system_id);
    $system = $this->entityTypeManager->getStorage('node')->load($system_id);
    $systemTapisId = $system->get(DrupalIds::SYSTEM_TAPIS_ID)->first()->getValue()['value'];
    $systemOwnerUid = $system->getOwnerId();
    $newOwnerUid = $form_state->getValue('new_owner');
    // $newOwnerDisplayName = \Drupal::entityTypeManager()->getStorage('user')->load($newOwnerUid)->getDisplayName();
    $newOwnerDisplayName = $this->entityTypeManager->getStorage('user')->load($newOwnerUid)->getDisplayName();

    $tenantId = $system->get(DrupalIds::SYSTEM_TENANT)->first()->getValue()['target_id'];

    // /** @var Drupal\tapis_job\TapisProvider\TapisJobProviderInterface */
    // $tapisSystemProvider =
    // \Drupal::service("tapis_system.tapis_system_provider");
    try {
      $this->tapisSystemProvider->changeSystemOwner($tenantId, $systemTapisId, $systemOwnerUid, $newOwnerUid);
      $system->setOwnerId($newOwnerUid);
      $system->set(DrupalIds::SYSTEM_OWNER, ["target_id" => $newOwnerUid]);
      $system->save();
      // \Drupal::messenger()
      // ->addMessage(Markup::create("<b>" . $system->label() .
      // "</b>'s owner has been changed to $newOwnerDisplayName."));
      $this->messenger()->addMessage(Markup::create("<b>" . $system->label() . "</b>'s owner has been changed to $newOwnerDisplayName."));
    }
    catch (\Exception | \Throwable $e) {
      if ($e instanceof TapisException || strpos(strtolower($e->getMessage()), 'tapis') !== FALSE) {
        if ($e instanceof TapisException) {
          $exception_message = "The system's owner could not be changed: " . $e->getPrimaryMessage() ?: "The system's owner could not be changed.";
        }
        else {
          $exception_message = "The system's owner could not be changed.";
        }
        $details = [
          '#type' => 'details',
          '#title' => 'Additional Details',
          '#open' => FALSE,
          '#children' => $e->getMessage(),
        ];
        $container = [
          '#type' => 'container',
          '#attributes' => ['class' => ['my-message-container']],
          '#children' => [
            'message' => [
              '#markup' => $exception_message,
              // Add a <p> tag before the message text.
              '#prefix' => '<p>',
              // Add a </p> tag after the message text.
              '#suffix' => '</p>',
            ],
            'details' => $details,
          ],
        ];
        $this
          ->messenger()
          ->addError($container);
      }
      else {
        $this
          ->messenger()
          ->addError($this
            ->t("The system's owner could not be changed. Please try again."));
      }
      $form_state
        ->setRebuild();
    }
  }

}
