<?php

namespace Drupal\tapis_system\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\tapis_system\DrupalIds;
use Drupal\tapis_tenant\Exception\TapisException;

/**
 * Form controller for the system credential entity edit forms.
 */
class TapisSystemCredentialForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    // Get the 'systemId' and 'loginUser' query parameters from the URL.
    // $systemId = \Drupal::request()->query->get('systemId');
    // $loginUser = \Drupal::request()->query->get('loginUser');
    $systemId = $this->getRequest()->query->get('systemId');
    $loginUser = $this->getRequest()->query->get('loginUser');

    if ($systemId) {
      $form['system']['widget']['#default_value'] = $systemId;
    }

    if ($loginUser) {
      $form['loginUser']['widget'][0]['value']['#default_value'] = $loginUser;
    }

    /** @var \Drupal\tapis_system\TapisSystemCredentialInterface $entity */
    $entity = $this->getEntity();
    $keyPairType = $form_state->getValue("key_pair_type");

    // $canUseGeneratedKeypair = \Drupal::currentUser()
    // ->hasPermission('use auto generated ssh key pair');
    // $canUseCustomKeypair = \Drupal::currentUser()
    // ->hasPermission('allow uploading own ssh key pair');
    $canUseGeneratedKeypair = $this->currentUser()->hasPermission('use auto generated ssh key pair');
    $canUseCustomKeypair = $this->currentUser()->hasPermission('allow uploading own ssh key pair');

    if (!$canUseCustomKeypair && !$canUseGeneratedKeypair) {
      // If the user does not have permission to use either type of keypair,
      // then show a message and disable the entire form.
      $form['#disabled'] = TRUE;
      // \Drupal::messenger()->addWarning
      // ("You do not have permission to auto-generate
      // or use your own keypair.");
      $this->messenger()->addWarning("You do not have permission to auto-generate or use your own keypair.");
      return $form;
    }

    if ($entity->isNew()) {
      // An AJAX-powered radio that will dynamically control
      // the display of the public key and private key fields.
      // The radio options will be: "Generate new SSH key pair"
      // and "Use existing SSH key pair".
      // When the value is "Generate new SSH key pair",
      // the public key and private key fields will be hidden and not required.
      // When the value is "Use existing SSH key pair",
      // the public key and private key fields will be displayed and required.
      $options = [];
      $defaultOption = NULL;

      // Check if the user has the permission
      // to generate a new key pair.
      if ($canUseGeneratedKeypair) {
        $options['generate'] = $this->t('Generate new SSH key pair');
        $defaultOption = 'generate';
      }

      // Check if the user has the permission
      // to use an existing ssh key pair.
      if ($canUseCustomKeypair) {
        $options['use_existing'] = $this->t('Use existing SSH key pair');
        if (!$defaultOption) {
          $defaultOption = 'use_existing';
        }
      }

      $form['key_pair_type'] = [
        '#type' => 'radios',
        '#title' => $this->t('SSH key pair'),
        '#required' => TRUE,
        '#options' => $options,
        '#default_value' => $defaultOption,
        '#ajax' => [
          'callback' => '::keyPairTypeCallback',
          'wrapper' => 'key-pair-type-wrapper',
        ],
      ];

      $form['key_pair_type_wrapper'] = [
        '#type' => 'container',
        '#attributes' => ['id' => 'key-pair-type-wrapper'],
      ];

      // Add fields for private key and public key **ONLY**
      // if this is a new System Credential entity.
      if ($keyPairType === "use_existing" && $canUseCustomKeypair) {
        $form['key_pair_type_wrapper']["tapis_system_credential_publicKey"] = [
          '#type' => 'textarea',
          '#title' => t('Public key'),
          '#description' => t('Enter the corresponding public key for this system credential.'),
          '#required' => TRUE,
          '#weight' => 4,
          '#states' => [
            'visible' => [
              ':input[name="key_pair_type"]' => ['value' => 'use_existing'],
            ],
            'invisible' => [
              ':input[name="key_pair_type"]' => ['value' => 'generate'],
            ],
            'required' => [
              ':input[name="key_pair_type"]' => ['value' => 'use_existing'],
            ],
            'optional' => [
              ':input[name="key_pair_type"]' => ['value' => 'generate'],
            ],
          ],
        ];
        $form['key_pair_type_wrapper']["tapis_system_credential_privateKey"] = [
          '#type' => 'textarea',
          '#title' => t('Private key'),
          '#description' => t('Enter the private key for this system credential. The private key is stored in a remote secure vault only.'),
          '#required' => TRUE,
          '#weight' => 5,
          '#states' => [
            'visible' => [
              ':input[name="key_pair_type"]' => ['value' => 'use_existing'],
            ],
            'invisible' => [
              ':input[name="key_pair_type"]' => ['value' => 'generate'],
            ],
            'required' => [
              ':input[name="key_pair_type"]' => ['value' => 'use_existing'],
            ],
            'optional' => [
              ':input[name="key_pair_type"]' => ['value' => 'generate'],
            ],
          ],
        ];
      }
    }
    else {
      // Hide the following fields: system, loginUser.
      $form['system']['#access'] = FALSE;
      $form['loginUser']['#access'] = FALSE;
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function keyPairTypeCallback(array &$form, FormStateInterface $form_state) {
    return $form['key_pair_type_wrapper'];
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $key_pair_type = $form_state->getValue("key_pair_type");
    $public_key = $form_state->getValue("tapis_system_credential_publicKey");
    $private_key = $form_state->getValue("tapis_system_credential_privateKey");

    /** @var \Drupal\tapis_system\TapisSystemCredentialInterface $entity */
    $entity = $this->getEntity();
    $is_entity_new = $entity->isNew();

    if ($is_entity_new) {
      $systemId = $entity->getSystemId();
      // $system = \Drupal::entityTypeManager()
      // ->getStorage("node")->load($systemId);
      $system = $this->entityTypeManager->getStorage("node")->load($systemId);

      if ($key_pair_type === "generate") {
        // Comment <system host>-nickname.
        $systemHost = $system->get(DrupalIds::SYSTEM_HOST)->getValue()[0]["value"];
        $nickname = $entity->label();
        $credentialComment = $systemHost . " - " . $nickname;

        // Generate a new SSH key pair.
        $keyPair = tapis_system_generate_ssh_keypair($credentialComment);
        $public_key = $keyPair["public_key"];
        $private_key = $keyPair["private_key"];
      }

      // Create the Tapis definition for this system credential
      // using the following fields:
      // $systemId, $systemOwnerUid, $accessorUserId,
      // $accessorLoginUser, $privateKey, $publicKey.
      $tapisDefinition = [];
      $tapisDefinition['systemId'] = $system->get(DrupalIds::SYSTEM_TAPIS_ID)->getValue()[0]['value'];
      $tapisDefinition['systemOwnerUid'] = $system->get(DrupalIds::SYSTEM_OWNER)->getValue()[0]['target_id'];

      // $tapisDefinition['systemEffectiveUserId'] =
      // $system->get(DrupalIds::SYSTEM_EFFECTIVE_USER)->getValue()[0]['value'];
      if (isset($system->get(DrupalIds::SYSTEM_EFFECTIVE_USER)->getValue()[0]['value'])) {
        $tapisDefinition['systemEffectiveUserId'] = $system->get(DrupalIds::SYSTEM_EFFECTIVE_USER)->getValue()[0]['value'];
      }
      else {
        $tapisDefinition['systemEffectiveUserId'] = NULL;
      }
      // $tapisDefinition['accessorUserId'] = \Drupal::currentUser()->id();
      $tapisDefinition['accessorUserId'] = $this->currentUser()->id();
      $tapisDefinition['accessorLoginUser'] = $form_state->getValue("loginUser")[0]['value'];

      $tapisDefinition['privateKey'] = $private_key;
      $tapisDefinition['publicKey'] = $public_key;

      // NOTE: These 2 lines below are NECESSARY for Tapis
      // to process the credentials correctly!
      $tapisDefinition['privateKey'] = str_replace("\r", '', $tapisDefinition['privateKey']);
      $tapisDefinition['publicKey'] = str_replace("\r", '', $tapisDefinition['publicKey']);

      if ($key_pair_type === "generate") {
        $tapisDefinition['skipCredentialCheck'] = TRUE;
      }
      else {
        $tapisDefinition['skipCredentialCheck'] = FALSE;
      }

      $entity->setPublicKey($tapisDefinition['publicKey']);
      $entity->setTapisDefinition($tapisDefinition);
    }

    try {
      $result = parent::save($form, $form_state);
    }
    catch (\Exception $e) {
      if ($e instanceof TapisException || strpos(strtolower($e->getMessage()), 'tapis') !== FALSE) {
        if ($e instanceof TapisException) {
          $exception_message = "The system credential could not be saved: " . $e->getPrimaryMessage() ?: "The system credential could not be saved.";
        }
        else {
          $exception_message = "The system credential could not be saved.";
        }
        $details = [
          '#type' => 'details',
          '#title' => 'Additional Details',
          '#open' => FALSE,
          '#children' => $e->getMessage(),
        ];
        $container = [
          '#type' => 'container',
          '#attributes' => ['class' => ['my-message-container']],
          '#children' => [
            'message' => [
              '#markup' => $exception_message,
        // Add a <p> tag before the message text.
              '#prefix' => '<p>',
        // Add a </p> tag after the message text.
              '#suffix' => '</p>',
            ],
            'details' => $details,
          ],
        ];
        $this
          ->messenger()
          ->addError($container);
      }
      else {
        $this
          ->messenger()
          ->addError($this
            ->t("The system credential could not be saved. Please try again."));
      }

      $form_state
        ->setRebuild();
      return;
    }

    $message_arguments = ['%label' => $entity->toLink()->toString()];
    $logger_arguments = [
      '%label' => $entity->label(),
      'link' => $entity->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New system credential %label has been created.', $message_arguments));
        $this->logger('tapis_system')->notice('Created new system credential %label', $logger_arguments);
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The system credential %label has been updated.', $message_arguments));
        $this->logger('tapis_system')->notice('Updated system credential %label.', $logger_arguments);
        break;
    }

    $form_state->setRedirect('entity.tapis_system_credential.canonical', ['tapis_system_credential' => $entity->id()]);
    return $result;
  }

}
