<?php

namespace Drupal\tapis_system\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\node\NodeInterface;
use Drupal\tapis_system\DrupalIds;
use Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Class TapisSystemTenantSharingForm.
 *
 * Form for sharing/unsharing a Tapis System with a Tapis tenant.
 */
class TapisSystemTenantSharingForm extends FormBase {

  /**
   * The Tapis site tenant provider.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */
  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The Tapis system provider.
   *
   * @var \Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface
   */
  protected TapisSystemProviderInterface $tapisSystemProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new AppLaunchForm object.
   *
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   * @param \Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface $tapisSystemProvider
   *   The Tapis system provider.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(TapisSiteTenantProviderInterface $tapisSiteTenantProvider,
                              TapisSystemProviderInterface $tapisSystemProvider,
                              EntityTypeManagerInterface $entityTypeManager) {
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
    $this->entityTypeManager = $entityTypeManager;
    $this->tapisSystemProvider = $tapisSystemProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get("tapis_tenant.tapis_site_tenant_provider"),
      $container->get('tapis_system.tapis_system_provider'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'tapis_system_tenantsharing_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL) {
    $form['#tree'] = TRUE;

    if (!$node) {
      return;
    }

    if ($node->bundle() !== DrupalIds::NODE_BUNDLE_SYSTEM) {
      throw new NotFoundHttpException();
    }

    // Check if the system's Tapis tenant's Tapis site
    // is in maintenance mode.
    // $tapisSiteTenantProvider =
    // \Drupal::service("tapis_tenant.tapis_site_tenant_provider");.
    $tenantId = $node->get(DrupalIds::SYSTEM_TENANT)->first()->getValue()['target_id'];
    if ($this->tapisSiteTenantProvider->isTenantInMaintenanceMode($tenantId)) {
      $form['message'] = [
        '#type' => 'markup',
        '#markup' => '<p>This system cannot be shared/unshared because its site is in maintenance mode.</p>',
      ];
      return $form;
    }

    $form_state->set("system_id", $node->id());

    $system_label = $node->label();

    $form['message_share'] = [
      '#type' => 'markup',
      '#markup' => "<p>Sharing the system will enable site users other than <b>$system_label</b>'s owner to also use the system (<b><i>only</i></b> if they satisfy the system's access control policies).</p>",
    ];
    $form['message_unshare'] = [
      '#type' => 'markup',
      '#markup' => "<p>Unsharing the system will prevent site users other than <b>$system_label</b>'s owner from using the system (even if they satisfy the system's access control policies).</p>",
    ];

    $form['share_button'] = [
      '#type' => 'submit',
      '#value' => t('Share'),
    ];

    $form['unshare_button'] = [
      '#type' => 'submit',
      '#value' => t('Unshare'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $system_id = $form_state->get("system_id");

    // /** @var \Drupal\node\NodeInterface $job */
    // $system = \Drupal::entityTypeManager()
    // ->getStorage('node')->load($system_id);
    $system = $this->entityTypeManager->getStorage('node')->load($system_id);
    $systemOwnerId = $system->getOwnerId();

    $tenantId = $system->get(DrupalIds::SYSTEM_TENANT)->first()->getValue()['target_id'];

    // /** @var Drupal\tapis_job\TapisProvider\TapisJobProviderInterface */
    // $tapisSystemProvider =
    // \Drupal::service("tapis_system.tapis_system_provider");
    // Get which button was clicked.
    $clicked_button = $form_state->getTriggeringElement()['#value']->render();
    if ($clicked_button === "Share") {
      $this->tapisSystemProvider->shareSystemWithTenant($tenantId, $system->get(DrupalIds::SYSTEM_TAPIS_ID)->getValue()[0]['value'], $systemOwnerId);
      // \Drupal::messenger()
      // ->addMessage(Markup::create("<b>" . $system->label() . "</b> has
      // been shared with the tenant."));
      $this->messenger()->addMessage(Markup::create("<b>" . $system->label() . "</b> has been shared with the tenant."));
    }
    else {
      $this->tapisSystemProvider->unshareSystemWithTenant($tenantId, $system->get(DrupalIds::SYSTEM_TAPIS_ID)->getValue()[0]['value'], $systemOwnerId);
      // \Drupal::messenger()
      // ->addMessage(Markup::create("<b>" . $system->label() . "</b> has been
      // unshared with the tenant."));
      $this->messenger()->addMessage(Markup::create("<b>" . $system->label() . "</b> has been unshared with the tenant."));
    }
  }

}
