<?php

namespace Drupal\tapis_system\TapisProvider;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\node\NodeInterface;
use Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface;
use Drupal\tapis_system\DrupalIds;
use Drupal\tapis_system\Exception\TapisSystemException;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use Drupal\user\Entity\User;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a service for interacting with Tapis systems.
 */
class TapisSystemProvider implements TapisSystemProviderInterface {

  const TAPIS_API_VERSION = "v3";

  /**
   * The HTTP client for making requests to Tapis systems.
   *
   * @var \GuzzleHttp\ClientInterface
   */

  protected ClientInterface $httpClient;

  /**
   * The provider for retrieving Tapis authentication tokens.
   *
   * @var \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface
   */

  protected TapisTokenProviderInterface $tapisTokenProvider;

  /**
   * The current Drupal user account.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */

  protected AccountProxyInterface $currentUser;

  /**
   * The provider for retrieving Tapis site tenant information.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */

  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new TapisSystemProvider object.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface $tapisTokenProvider
   *   The Tapis token provider.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   */
  public function __construct(
    ClientInterface $httpClient,
    TapisTokenProviderInterface $tapisTokenProvider,
    AccountProxyInterface $currentUser,
    TapisSiteTenantProviderInterface $tapisSiteTenantProvider,
    EntityTypeManagerInterface $entityTypeManager,
    LoggerChannelFactoryInterface $loggerFactory) {
    $this->httpClient = $httpClient;
    $this->tapisTokenProvider = $tapisTokenProvider;
    $this->currentUser = $currentUser;
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
    $this->entityTypeManager = $entityTypeManager;
    $this->logger = $loggerFactory->get('tapis_system');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client'),
      $container->get('tapis_auth.tapis_token_provider'),
      $container->get("current_user"),
      $container->get("tapis_tenant.tapis_site_tenant_provider"),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function createSystem($tenantId, $tapisDefinition, $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tenant_tapis_id = $tenantInfo['tapis_id'];
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems";

    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      'json' => $tapisDefinition,
      "http_errors" => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Failed to create a Tapis system in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function doesSystemIdExist($tenantId, $systemId, $uid = -1): bool {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/$systemId";

    $response = $this->httpClient->request('GET', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      "http_errors" => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() === 200) {
      return TRUE;
    }
    elseif ($response->getStatusCode() === 404) {
      return FALSE;
    }
    else {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not check if the Tapis system '$systemId' exists in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function getSystem($tenantId, $systemId, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/$systemId";

    $response = $this->httpClient->request('GET', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      "http_errors" => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not fetch the Tapis system '$systemId' from the '$tenant_tapis_id' tenant: $message");
    }
    else {
      return $response_body['result'];
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function updateSystem($tenantId, $systemId, $tapisDefinition, $uid = -1) {
    // Remove any non-update-able keys from $tapisDefinition
    // Note: the "FALSE" values are added so that array_diff_key would work.
    $non_updateable_keys = [
      "id" => FALSE,
      "systemType" => FALSE,
      "owner" => FALSE,
      "enabled" => FALSE,
      "bucketName" => FALSE,
      "rootDir" => FALSE,
      "isDtn" => FALSE,
      "canExec" => FALSE,
    ];
    $tapisDefinition = array_diff_key($tapisDefinition, $non_updateable_keys);

    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/$systemId";

    $response = $this->httpClient->request('PATCH', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      'json' => $tapisDefinition,
      "http_errors" => FALSE,
    ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not update the Tapis system '$systemId' in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function enableSystem($tenantId, $systemId, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/$systemId/enable";

    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      "http_errors" => FALSE,
    ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not enable the Tapis system '$systemId' in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function disableSystem($tenantId, $systemId, $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/$systemId/disable";

    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      "http_errors" => FALSE,
    ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not disable the Tapis system '$systemId' in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function createSchedulerProfile($tenantId, $tapisDefinition, $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/schedulerProfile";
    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      'json' => $tapisDefinition,
      "http_errors" => FALSE,
    ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not create a Tapis scheduler profile in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritDoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function getSchedulerProfile($tenantId, $schedulerProfileId, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/schedulerProfile/$schedulerProfileId";
    $response = $this->httpClient->request('GET', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]
    );
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("An error occurred when fetching a Tapis scheduler profile from the '$tenant_tapis_id' tenant: $message");
    }

    return $response_body['result'];
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function deleteSchedulerProfile($tenantId, $schedulerProfileId, $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/schedulerProfile/$schedulerProfileId";
    $response = $this->httpClient->request('DELETE', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]
    );
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("An error occurred when deleting a Tapis scheduler profile from the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function createUserCredential(
    $tenantId,
    $systemId,
    $systemOwnerUid,
    $systemEffectiveUserId,
    $accessorUserId,
    $accessorLoginUser,
    $privateKey,
    $publicKey,
    $skipCredentialCheck = FALSE,
  ): void {
    $userName = $this->tapisTokenProvider->getTapisUsername($tenantId, $accessorUserId);
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $systemOwnerUid);
    $loginUser = $accessorLoginUser;

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $json = [
      "privateKey" => $privateKey,
      "publicKey" => $publicKey,
    ];

    $skipCredentialCheckArg = $skipCredentialCheck ? "true" : "false";

    if ($systemEffectiveUserId === '${apiUserId}') {
      // The system is configured to use a dynamic effective user
      // based on the current api user id.
      $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/credential/$systemId/user/$userName?skipCredentialCheck=$skipCredentialCheckArg";
      $json['loginUser'] = $loginUser;
    }
    else {
      // The system is configured to use a static effective user.
      $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/credential/$systemId/user/$loginUser?skipCredentialCheck=$skipCredentialCheckArg";
    }

    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        'json' => $json,
        "http_errors" => FALSE,
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not create a Tapis system credential for the '$systemId' system in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function checkUserCredential($tenantId, $systemId, $systemOwnerUid, $accessorUserId): bool {
    $userName = $this->tapisTokenProvider->getTapisUsername($tenantId, $accessorUserId);
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $systemOwnerUid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];

    $currentUser = User::load($accessorUserId);
    // Get the system.
    $system = $this->getSystem($tenantId, $systemId, $systemOwnerUid);

    $userPermissionForSystemAdmin = $currentUser->hasPermission('administer tapis system');
    $userPermissionForEditAnyContent = $currentUser->hasPermission('edit any tapis_system content');

    $isPublic = $system['isPublic'];
    $sharedWithUsers = $system['sharedWithUsers'];
    $tapisUserId = $this->tapisTokenProvider->getTapisUsername($tenantId, $accessorUserId);
    if ($isPublic || in_array($tapisUserId, $sharedWithUsers) ||
      $userPermissionForSystemAdmin || $userPermissionForEditAnyContent ||
      $currentUser->id() === $systemOwnerUid) {
      $isDynamicEffectiveUser = $system['isDynamicEffectiveUser'];
      if (!$isDynamicEffectiveUser) {
        $userName = $system['effectiveUserId'];
      }
    }

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/credential/$systemId/user/$userName/check";
    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]);

    // Log the response array as a string.
    $response_body = json_decode($response->getBody(), TRUE);
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() === 200) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function removeUserCredential($tenantId, $systemId, $systemOwnerUid, $systemEffectiveUserId, $accessorUserId): void {
    $userName = $this->tapisTokenProvider->getTapisUsername($tenantId, $accessorUserId);
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $systemOwnerUid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    if ($systemEffectiveUserId === '${apiUserId}') {
      // The system is configured to use a dynamic effective user
      // based on the current api user id.
      $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/credential/$systemId/user/$userName";
    }
    else {
      // The system is configured to use a static effective user.
      $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/credential/$systemId/user/$systemEffectiveUserId";
    }

    $response = $this->httpClient->request('DELETE', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]
    );
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not delete a Tapis system credential for the '$systemId' system in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function convertSystemToJSON(NodeInterface $system): array {
    $json_doc = [];
    $json_doc['id'] = trim($system->get(DrupalIds::SYSTEM_TAPIS_ID)->getValue()[0]['value']);

    // Enable cmdPrefix for this system
    // so that we can run batch and vnc jobs on it.
    $json_doc['enableCmdPrefix'] = TRUE;

    // @todo send description field to Tapis? note that description is a *formatted* text field!
    $json_doc['systemType'] = $system->get(DrupalIds::SYSTEM_TYPE)->getValue()[0]['value'];

    $tenantId = $system->get(DrupalIds::SYSTEM_TENANT)->getValue()[0]['target_id'];

    $owner_userid = $system->get(DrupalIds::SYSTEM_OWNER)->getValue()[0]['target_id'];
    $json_doc['owner'] = $this->tapisTokenProvider->getTapisUsername($tenantId, $owner_userid);

    $json_doc['host'] = $system->get(DrupalIds::SYSTEM_HOST)->getValue()[0]['value'];
    $json_doc['enabled'] = boolval($system->get(DrupalIds::SYSTEM_ENABLED)->getValue()[0]['value']);
    $json_doc['effectiveUserId'] = $this->getEffectiveUserForSystem($system);

    $json_doc['defaultAuthnMethod'] = "PKI_KEYS";
    // $json_doc['bucketName'] =
    // $system->get(DrupalIds::SYSTEM_BUCKET_NAME)->getValue()[0]['value'];
    if (isset($system->get(DrupalIds::SYSTEM_BUCKET_NAME)->getValue()[0]['value'])) {
      $json_doc['bucketName'] = $system->get(DrupalIds::SYSTEM_BUCKET_NAME)->getValue()[0]['value'];
    }
    else {
      $json_doc['bucketName'] = NULL;
    }
    $json_doc['rootDir'] = trim($system->get(DrupalIds::SYSTEM_ROOT_DIRECTORY)->getValue()[0]['value']);

    $json_doc['port'] = intval($system->get(DrupalIds::SYSTEM_PORT)->getValue()[0]['value']);

    $json_doc['useProxy'] = boolval($system->get(DrupalIds::SYSTEM_USE_PROXY)->getValue()[0]['value']);

    if ($json_doc['useProxy']) {
      $json_doc['proxyHost'] = $system->get(DrupalIds::SYSTEM_PROXY_HOST)->getValue()[0]['value'];
      $json_doc['proxyPort'] = intval($system->get(DrupalIds::SYSTEM_PROXY_PORT)->getValue()[0]['value']);
    }

    $isDtn = boolval($system->get(DrupalIds::SYSTEM_IS_DTN)->getValue()[0]['value']);

    $dtnSystem = $system->get(DrupalIds::SYSTEM_DTN_SYSTEM)->entity;
    if ($isDtn && $dtnSystem) {
      /** @var \Drupal\node\NodeInterface $dtnSystem */
      $json_doc['dtnSystemId'] = $dtnSystem->get(DrupalIds::SYSTEM_TAPIS_ID)->getValue()[0]['value'];
      // $json_doc['dtnMountPoint'] = $system->get(DrupalIds::SYSTEM_DTN_MOUNT_POINT)->getValue()[0]['value'];
      // $json_doc['dtnMountSourcePath'] = trim($system->get(DrupalIds::SYSTEM_DTN_MOUNT_SOURCE_PATH)->getValue()[0]['value']);
    }

    $json_doc['canExec'] = boolval($system->get(DrupalIds::SYSTEM_CAN_EXECUTE_JOBS)->getValue()[0]['value']);
    $json_doc['canRunBatch'] = boolval($system->get(DrupalIds::SYSTEM_CAN_RUN_BATCH)->getValue()[0]['value']);

    $jobRuntimes = [];
    foreach ($system->get(DrupalIds::SYSTEM_JOB_RUNTIMES)->getValue() ?? [] as $runtime) {
      $jobRuntimes[] = ['runtimeType' => $runtime['value']];
    }
    $json_doc['jobRuntimes'] = $jobRuntimes;

    $json_doc['jobWorkingDir'] = trim($system->get(DrupalIds::SYSTEM_JOB_WORKING_DIR)->getValue()[0]['value']);

    $job_env_vars = [];
    if (isset($system->get(DrupalIds::SYSTEM_JOB_ENV_VARS)->getValue()[0]['value'])) {
      foreach (explode("\n", $system->get(DrupalIds::SYSTEM_JOB_ENV_VARS)
        ->getValue()[0]['value']) as $env_var_line) {
        $env_var_line = str_replace(["\n", "\r"], '', $env_var_line);
        if ($env_var_line) {
          $env_var_parts = explode("=", $env_var_line);
          $env_var = [];
          if ($env_var_parts) {
            $env_var['key'] = $env_var_parts[0];
            $env_var['value'] = $env_var_parts[1] ?? '';
          }
          $job_env_vars[] = $env_var;
        }
      }
    }
    $json_doc['jobEnvVariables'] = $job_env_vars;

    $json_doc['jobMaxJobs'] = intval($system->get(DrupalIds::SYSTEM_MAX_NUM_OF_JOBS)->getValue()[0]['value']);
    $json_doc['jobMaxJobsPerUser'] = intval($system->get(DrupalIds::SYSTEM_MAX_JOBS_PER_USER)->getValue()[0]['value']);

    if ($json_doc['canRunBatch']) {
      // $mpiCmd = $system->get(DrupalIds::SYSTEM_MPI_COMMAND)
      // ->getValue()[0]['value'];
      if (isset($system->get(DrupalIds::SYSTEM_MPI_COMMAND)->getValue()[0]['value'])) {
        $mpiCmd = $system->get(DrupalIds::SYSTEM_MPI_COMMAND)->getValue()[0]['value'];
      }
      else {
        $mpiCmd = NULL;
      }
      if ($mpiCmd) {
        $json_doc['mpiCmd'] = $mpiCmd;
      }

      $json_doc['batchScheduler'] = $system->get(DrupalIds::SYSTEM_BATCH_SCHEDULER)->getValue()[0]['value'];

      $batch_logical_queues = [];
      $batch_logical_queue_entities = $system->get(DrupalIds::SYSTEM_BATCH_LOGICAL_QUEUES)->referencedEntities() ?? [];
      foreach ($batch_logical_queue_entities as $queue) {
        $batch_logical_queues[] = $this->convertBatchLogicalQueueToJSON($queue);
      }
      $json_doc['batchLogicalQueues'] = $batch_logical_queues;

      $batch_default_logical_queue_name = $system->get(DrupalIds::SYSTEM_DEFAULT_BATCH_LOGICAL_QUEUE)->getValue()[0]['value'];

      if ($batch_default_logical_queue_name) {
        // $batch_default_logical_queue_entity->getName();
        $json_doc['batchDefaultLogicalQueue'] = $batch_default_logical_queue_name;
      }

      $batchSchedulerProfileEntity = $system->get(DrupalIds::SYSTEM_BATCH_SCHEDULER_PROFILE)->first();

      if ($batchSchedulerProfileEntity) {
        /** @var \Drupal\Core\Entity\Plugin\DataType\EntityReference $batchSchedulerProfileEntity */
        $batchSchedulerProfileEntity = $batchSchedulerProfileEntity->get("entity");
        $batchSchedulerProfileEntity = $batchSchedulerProfileEntity->getTarget()->getValue();
        $json_doc['batchSchedulerProfile'] = $batchSchedulerProfileEntity->get(DrupalIds::SYSTEM_SCHEDULER_PROFILE_TAPIS_ID)->getValue()[0]['value'];
      }
    }

    $job_capabilities = [];
    $job_capability_entities = $system->get(DrupalIds::SYSTEM_JOB_CAPABILITIES)->referencedEntities() ?? [];
    foreach ($job_capability_entities as $job_capability) {
      $job_capabilities[] = $this->convertJobCapabilityToJSON($job_capability);
    }
    $json_doc['jobCapabilities'] = $job_capabilities;

    if (isset($system->get(DrupalIds::SYSTEM_NOTES)->getValue()[0]['value']) &&
    !empty($system->get(DrupalIds::SYSTEM_NOTES)->getValue()[0]['value'])) {
      $systemNotes = trim($system->get(DrupalIds::SYSTEM_NOTES)
        ->getValue()[0]['value']);
      $result = json_decode($systemNotes);
      if (json_last_error() === JSON_ERROR_NONE) {
        // JSON is valid.
        $json_doc['notes'] = $result;
      }
    }
    // @todo tags
    return $json_doc;
  }

  /**
   * {@inheritdoc}
   */
  public function getEffectiveUserForSystem($system) {
    /** @var \Drupal\tapis_system\Entity\TapisSystemCredential $system */
    $use_static_system_user = boolval($system->get(DrupalIds::SYSTEM_USE_STATIC_SYSTEM_USER)->getValue()[0]['value']);
    if (!$use_static_system_user) {
      // If we're not using a static system username
      // for this system, return ${apiUserId}.
      return '${apiUserId}';
    }
    else {
      // we're using a static system user for accessing this system.
      return $system->get(DrupalIds::SYSTEM_EFFECTIVE_USER)->getValue()[0]['value'];
    }
  }

  /**
   * {@inheritdoc}
   */
  private function convertBatchLogicalQueueToJSON($queue): array {
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MAX_JOBS)->getValue()[0]['value'])) {
      $maxJobs = intval($queue->get(DrupalIds::SYSTEM_BLQ_MAX_JOBS)
        ->getValue()[0]['value']);
    }
    else {
      $maxJobs = 0;
    }
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MAX_JOBS_PER_USER)->getValue()[0]['value'])) {
      $maxJobsPerUser = intval($queue->get(DrupalIds::SYSTEM_BLQ_MAX_JOBS_PER_USER)
        ->getValue()[0]['value']);
    }
    else {
      $maxJobsPerUser = 0;
    }
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MIN_NODES)->getValue()[0]['value'])) {
      $minNodeCount = intval($queue->get(DrupalIds::SYSTEM_BLQ_MIN_NODES)
        ->getValue()[0]['value']);
    }
    else {
      $minNodeCount = 0;
    }
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MAX_NODES)->getValue()[0]['value'])) {
      $maxNodeCount = intval($queue->get(DrupalIds::SYSTEM_BLQ_MAX_NODES)
        ->getValue()[0]['value']);
    }
    else {
      $maxNodeCount = 0;
    }
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MIN_CORES_PER_NODE)->getValue()[0]['value'])) {
      $minCoresPerNode = intval($queue->get(DrupalIds::SYSTEM_BLQ_MIN_CORES_PER_NODE)
        ->getValue()[0]['value']);
    }
    else {
      $minCoresPerNode = 0;
    }
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MAX_CORES_PER_NODE)->getValue()[0]['value'])) {
      $maxCoresPerNode = intval($queue->get(DrupalIds::SYSTEM_BLQ_MAX_CORES_PER_NODE)
        ->getValue()[0]['value']);
    }
    else {
      $maxCoresPerNode = 0;
    }
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MIN_MEMORY)->getValue()[0]['value'])) {
      $minMemoryMB = intval($queue->get(DrupalIds::SYSTEM_BLQ_MIN_MEMORY)
        ->getValue()[0]['value']);
    }
    else {
      $minMemoryMB = 0;
    }
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MAX_MEMORY)->getValue()[0]['value'])) {
      $maxMemoryMB = intval($queue->get(DrupalIds::SYSTEM_BLQ_MAX_MEMORY)
        ->getValue()[0]['value']);
    }
    else {
      $maxMemoryMB = 0;
    }
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MIN_TIME)->getValue()[0]['value'])) {
      $minMinutes = intval($queue->get(DrupalIds::SYSTEM_BLQ_MIN_TIME)
        ->getValue()[0]['value']);
    }
    else {
      $minMinutes = 0;
    }
    if (isset($queue->get(DrupalIds::SYSTEM_BLQ_MAX_TIME)->getValue()[0]['value'])) {
      $maxMinutes = intval($queue->get(DrupalIds::SYSTEM_BLQ_MAX_TIME)
        ->getValue()[0]['value']);
    }
    else {
      $maxMinutes = 0;
    }
    return [
      'name' => $queue->getName(),
      'hpcQueueName' => $queue->get(DrupalIds::SYSTEM_BLQ_HPC_QUEUE_NAME)->getValue()[0]['value'],
      'maxJobs' => $maxJobs,
      'maxJobsPerUser' => $maxJobsPerUser,
      'minNodeCount' => $minNodeCount,
      'maxNodeCount' => $maxNodeCount,
      'minCoresPerNode' => $minCoresPerNode,
      'maxCoresPerNode' => $maxCoresPerNode,
      'minMemoryMB' => $minMemoryMB,
      'maxMemoryMB' => $maxMemoryMB,
      'minMinutes' => $minMinutes,
      'maxMinutes' => $maxMinutes,
    ];
  }

  /**
   * {@inheritdoc}
   */
  private function convertJobCapabilityToJSON($job_capability): array {
    return [
      'name' => $job_capability->getName(),
      'category' => $job_capability->get(DrupalIds::SYSTEM_JOB_CAPABILITY_CATEGORY)->getValue()[0]['value'],
      'datatype' => $job_capability->get(DrupalIds::SYSTEM_JOB_CAPABILITY_DATA_TYPE)->getValue()[0]['value'],
      'precedence' => $job_capability->get(DrupalIds::SYSTEM_JOB_CAPABILITY_PRECEDENCE)->getValue()[0]['value'],
      'value' => $job_capability->get(DrupalIds::SYSTEM_JOB_CAPABILITY_VALUE)->getValue()[0]['value'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function convertSchedulerProfileToJSON(NodeInterface $schedulerProfile, $uid = -1): array {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $tenant_id = $schedulerProfile->get(DrupalIds::SYSTEM_SCHEDULER_PROFILE_TENANT)->getValue()[0]['target_id'];

    $json_doc = [];
    $json_doc['name'] = $schedulerProfile->get(DrupalIds::SYSTEM_SCHEDULER_PROFILE_TAPIS_ID)->getValue()[0]['value'];
    $json_doc['description'] = $schedulerProfile->get(DrupalIds::SYSTEM_SCHEDULER_PROFILE_SUMMARY)->getValue()[0]['value'] ?? "";
    $json_doc['owner'] = $this->tapisTokenProvider->getTapisUsername($tenant_id, $uid);

    $json_doc['moduleLoads'] = [];

    $moduleCommands = $schedulerProfile->get(DrupalIds::SYSTEM_SCHEDULER_PROFILE_MODULE_COMMANDS)->getValue();
    foreach ($moduleCommands as $moduleCommand) {
      $moduleCommandId = $moduleCommand['target_id'];
      // Load the "storage" content entity with this id.
      // $moduleCommand = \Drupal::entityTypeManager()->getStorage('storage')->load($moduleCommandId);
      /** @var \Drupal\storage\Entity\Storage $moduleCommand */
      $moduleCommand = $this->entityTypeManager->getStorage('storage')->load($moduleCommandId);
      $moduleLoadCommand = $moduleCommand->get(DrupalIds::MODULE_COMMAND)->getValue()[0]['value'];
      // $modulesToLoad = trim($moduleCommand->get(DrupalIds::MODULE_COMMAND_MODULES)->getValue()[0]['value']);
      if (isset($moduleCommand->get(DrupalIds::MODULE_COMMAND_MODULES)->getValue()[0]['value'])) {
        $modulesToLoad = trim($moduleCommand->get(DrupalIds::MODULE_COMMAND_MODULES)->getValue()[0]['value']);
      }
      else {
        $modulesToLoad = NULL;
      }
      $modulesToLoadArray = [];

      // Split the modulesToLoad string into
      // an array of modules, space delimited.
      if (!empty($modulesToLoad)) {
        $modulesToLoadArray = explode(" ", $modulesToLoad);
      }

      $json_doc['moduleLoads'][] = [
        "moduleLoadCommand" => $moduleLoadCommand,
        "modulesToLoad" => $modulesToLoadArray,
      ];
    }

    $json_doc['hiddenOptions'] = [];
    $hiddenOptions = $schedulerProfile->get(DrupalIds::SYSTEM_SCHEDULER_PROFILE_HIDDEN_OPTIONS)->getValue();
    foreach ($hiddenOptions as $hiddenOption) {
      $json_doc['hiddenOptions'][] = $hiddenOption['value'] ?? "";
    }
    return $json_doc;
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function changeSystemOwner($tenantId, $systemId, $systemOwnerUid, $newOwnerUid): void {
    $userName = $this->tapisTokenProvider->getTapisUsername($tenantId, $newOwnerUid);
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $systemOwnerUid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/$systemId/changeOwner/$userName";
    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]);

    // Log the response array as a string.
    $response_body = json_decode($response->getBody(), TRUE);
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() === 200) {
      return;
    }

    if (is_array($response_body) && isset($response_body['message'])) {
      $message = $response_body['message'];
    } else {
      // Optionally, include the raw body or a default message
      $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
    }
    throw new TapisSystemException("Failed to change the owner for system '$systemId': $message");
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function shareSystemWithTenant($tenantId, $systemId, $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/share_public/$systemId/";

    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not share the Tapis '$systemId' system with the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function grantSystemPermission($tenantId, $systemId, $username, $permission, $path = "/", $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/files/permissions/$systemId/$path";

    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
        'json' => [
          "username" => $username,
          "permission" => $permission,
        ],
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not grant the Tapis '$systemId' system permission to the user '$username' in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function revokeSystemPermission($tenantId, $systemId, $username, $path = "/", $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/files/permissions/$systemId/$path";

    $response = $this->httpClient->request('DELETE', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
        'json' => [
          "username" => $username,
        ],
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not revoke the Tapis '$systemId' system permission from the user '$username' in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function unshareSystemWithTenant($tenantId, $systemId, $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/unshare_public/$systemId/";

    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not unshare the Tapis '$systemId' system from the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getSystemStatInfo($tenantId, $systemId, $path, $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/files/utils/linux/$systemId/$path";

    $response = $this->httpClient->request('GET', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function getShareSystemWithTenant($tenantId, $systemId, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/share/$systemId";

    $response = $this->httpClient->request('GET', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      throw new TapisSystemException("Could not get the share list for the Tapis '$systemId' system with the '$tenant_tapis_id' tenant: " . $response_body['message']);
    }
    else {
      return $response_body['result'];
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function shareSystemWithUser($tenantId, $systemId, $shareUserId, $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);
    $username = $this->tapisTokenProvider->getTapisUsername($tenantId, $shareUserId);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/share/$systemId/";

    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
        'json' => [
          "users" => [$username],
        ],
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug("StatusCode = " . $response->getStatusCode());
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not share the Tapis '$systemId' system permission to the user '$username' in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_system\Exception\TapisSystemException|\GuzzleHttp\Exception\GuzzleException
   */
  public function unshareSystemWithUser($tenantId, $systemId, $shareUserId, $uid = -1): void {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);
    $username = $this->tapisTokenProvider->getTapisUsername($tenantId, $shareUserId);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/systems/unshare/$systemId/";

    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
        'json' => [
          "users" => [$username],
        ],
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug("StatusCode = " . $response->getStatusCode());
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisSystemException("Could not unshare the Tapis '$systemId' system permission to the user '$username' in the '$tenant_tapis_id' tenant: $message");
    }
  }

}
