<?php

namespace Drupal\tapis_system\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\tapis_system\DrupalIds;
use Drupal\tapis_tenant\Exception\TapisException;

/**
 * Form controller for the system credential entity edit forms.
 */
class TapisSystemCredentialForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    // Get the 'systemId' and 'loginUser' query parameters from the URL.
    // $systemId = \Drupal::request()->query->get('systemId');
    // $loginUser = \Drupal::request()->query->get('loginUser');
    $systemId = $this->getRequest()->query->get('systemId');
    $loginUser = $this->getRequest()->query->get('loginUser');

    if ($systemId) {
      $form['system']['widget']['#default_value'] = $systemId;
    }

    if ($loginUser) {
      $form['loginUser']['widget'][0]['value']['#default_value'] = $loginUser;
    }

    /** @var \Drupal\tapis_system\TapisSystemCredentialInterface $entity */
    $entity = $this->getEntity();
    $keyPairType = $form_state->getValue("key_pair_type");

    // $canUseGeneratedKeypair = \Drupal::currentUser()
    // ->hasPermission('use auto generated ssh key pair');
    // $canUseCustomKeypair = \Drupal::currentUser()
    // ->hasPermission('allow uploading own ssh key pair');
    $canUseGeneratedKeypair = $this->currentUser()->hasPermission('use auto generated ssh key pair');
    $canUseCustomKeypair = $this->currentUser()->hasPermission('allow uploading own ssh key pair');

    if (!$canUseCustomKeypair && !$canUseGeneratedKeypair) {
      // If the user does not have permission to use either type of keypair,
      // then show a message and disable the entire form.
      $form['#disabled'] = TRUE;
      // \Drupal::messenger()->addWarning
      // ("You do not have permission to auto-generate
      // or use your own keypair.");
      $this->messenger()->addWarning("You do not have permission to auto-generate or use your own keypair.");
      return $form;
    }

    if ($entity->isNew()) {
      // Populate the 'system' dropdown with systems owned by the current user
      // and include status (static/dynamic) and existing usernames in the option label.
      try {
        $current_uid = $this->currentUser()->id();
        $is_admin = $this->currentUser()->hasPermission('administer tapis system');
        $node_storage = $this->entityTypeManager->getStorage('node');
        // Base: published systems.
        $base_query = $node_storage->getQuery()
          ->accessCheck(FALSE)
          ->condition('type', DrupalIds::NODE_BUNDLE_SYSTEM)
          ->condition('status', 1);
        if ($is_admin) {
          // Admin sees all published systems.
          $system_nids = $base_query->execute();
        }
        else {
          // Non-admin: owned systems OR dynamic+public systems (use stored field).
          $owned_nids = (clone $base_query)
            ->condition('uid', $current_uid)
            ->execute();

          $public_dynamic_nids = (clone $base_query)
            ->condition(DrupalIds::SYSTEM_USE_STATIC_SYSTEM_USER, 0)
            ->condition('tapis_sys_restrictions', 'public')
            ->execute();

          $system_nids = array_values(array_unique(array_merge($owned_nids, $public_dynamic_nids)));
        }

        $options = [];
        if (!empty($system_nids)) {
          // Deduplicate nids to avoid accidental duplicates from merges.
          $system_nids = array_values(array_unique($system_nids));
          $systems = $node_storage->loadMultiple($system_nids);

          // Load all existing credentials for all users, grouped by system.
          $cred_storage = $this->entityTypeManager->getStorage('tapis_system_credential');
          $cred_ids = $cred_storage->getQuery()
            ->accessCheck(FALSE)
            ->condition('system', array_keys($systems), 'IN')
            ->execute();
          $login_by_system = [];
          if (!empty($cred_ids)) {
            $creds = $cred_storage->loadMultiple($cred_ids);
            foreach ($creds as $cred) {
              $sid = $cred->get('system')->first()->getValue()['target_id'] ?? NULL;
              $login = $cred->get('loginUser')->getValue()[0]['value'] ?? NULL;
              if ($sid && $login) {
                $login_by_system[$sid][$login] = TRUE; // use keys to ensure uniqueness
              }
            }
          }

          foreach ($systems as $system) {
            $sid = $system->id();
            $label = $system->label();
            $use_static = FALSE;
            $field = $system->get(DrupalIds::SYSTEM_USE_STATIC_SYSTEM_USER);
            if ($field && $field->first()) {
              $use_static = (bool) ($field->first()->getValue()['value'] ?? FALSE);
            }
            $status = $use_static ? $this->t('static') : $this->t('dynamic');

            // When include_shared is off, systems are already filtered to owned only.
            // Only owned systems are listed; no network calls are made here.

            $logins = [];
            if (!empty($login_by_system[$sid])) {
              $logins = array_keys($login_by_system[$sid]);
            }
            $login_text = !empty($logins)
              ? implode(', ', $logins)
              : $this->t('none');
            $options[$sid] = $this->t('@title (@status, Username: @logins)', [
              '@title' => $label,
              '@status' => $status,
              '@logins' => $login_text,
            ]);
          }
        }

        // Apply the options to the widget. Support common structures.
        if (isset($form['system']['widget'][0]['target_id'])) {
          $form['system']['widget'][0]['target_id']['#options'] = $options;
          $form['system']['widget'][0]['target_id']['#empty_option'] = $this->t('- Select a system -');
        }
        elseif (isset($form['system']['widget'])) {
          // Fallback in case the widget consolidates the select at this level.
          $form['system']['widget']['#options'] = $options;
          $form['system']['widget']['#empty_option'] = $this->t('- Select a system -');
        }

        if (empty($options)) {
          $this->messenger()->addWarning($this->t('No eligible systems found. You may create credentials on systems you own, or public dynamic systems.'));
        }
      }
      catch (\Throwable $e) {
        // Non-fatal if filtering fails; keep default behavior.
      }
      // Add AJAX on system select to toggle Username field for static systems,
      // and wrap the Username widget for partial replacement.
      if (isset($form['system']['widget'][0]['target_id'])) {
        $form['system']['widget'][0]['target_id']['#ajax'] = [
          'callback' => '::systemSelectCallback',
          'wrapper' => 'login-user-wrapper',
          'event' => 'change',
        ];
      }
      elseif (isset($form['system']['widget'])) {
        // Fallback in case the select is rendered directly on widget.
        $form['system']['widget']['#ajax'] = [
          'callback' => '::systemSelectCallback',
          'wrapper' => 'login-user-wrapper',
          'event' => 'change',
        ];
      }
      // Wrap loginUser element for AJAX replace.
      if (isset($form['loginUser'])) {
        $form['loginUser']['#prefix'] = '<div id="login-user-wrapper">';
        $form['loginUser']['#suffix'] = '</div>';
      }

      // Initialize Username based on selected system (static/dynamic).
      $selected_system_id = $form_state->getValue(['system', 0, 'target_id']) ?? $systemId;
      if (!$selected_system_id) {
        $input = $form_state->getUserInput();
        if (!empty($input['system'][0]['target_id'])) {
          $selected_system_id = $input['system'][0]['target_id'];
        }
        elseif (!empty($input['system'])) {
          // Some widgets flatten the value under 'system'.
          $selected_system_id = is_array($input['system']) ? reset($input['system']) : $input['system'];
        }
      }
      if ($selected_system_id) {
        $system = $this->entityTypeManager->getStorage('node')->load($selected_system_id);
        if ($system) {
          $use_static = FALSE;
          $field = $system->get(DrupalIds::SYSTEM_USE_STATIC_SYSTEM_USER);
          if ($field && $field->first()) {
            $use_static = (bool) ($field->first()->getValue()['value'] ?? FALSE);
          }
          if ($use_static) {
            $effective = $system->get(DrupalIds::SYSTEM_EFFECTIVE_USER)->getValue()[0]['value'] ?? '';
            if (isset($form['loginUser']['widget'][0]['value'])) {
              // Force the value both in the form element and form state.
              $form['loginUser']['widget'][0]['value']['#value'] = $effective;
              $form['loginUser']['widget'][0]['value']['#default_value'] = $effective;
              // Also set user input so FAPI uses it for this AJAX rebuild.
              $input = $form_state->getUserInput();
              if (!isset($input['loginUser'][0])) {
                $input['loginUser'][0] = [];
              }
              $input['loginUser'][0]['value'] = $effective;
              $form_state->setUserInput($input);
              $form_state->setValue(['loginUser', 0, 'value'], $effective);
              $form['loginUser']['widget'][0]['value']['#disabled'] = TRUE;
              $form['loginUser']['widget'][0]['value']['#attributes']['readonly'] = 'readonly';
              $form['loginUser']['widget'][0]['value']['#description'] = $this->t('Username is fixed for static systems.');
            }
          }
          else {
            if (isset($form['loginUser']['widget'][0]['value'])) {
              // Ensure field is enabled for dynamic systems; do not override user input.
              $form['loginUser']['widget'][0]['value']['#disabled'] = FALSE;
              unset($form['loginUser']['widget'][0]['value']['#attributes']['readonly']);
            }
          }
        }
      }

      // An AJAX-powered radio that will dynamically control
      // the display of the public key and private key fields.
      // The radio options will be: "Generate new SSH key pair"
      // and "Use existing SSH key pair".
      // When the value is "Generate new SSH key pair",
      // the public key and private key fields will be hidden and not required.
      // When the value is "Use existing SSH key pair",
      // the public key and private key fields will be displayed and required.
      $options = [];
      $defaultOption = NULL;

      // Check if the user has the permission
      // to generate a new key pair.
      if ($canUseGeneratedKeypair) {
        $options['generate'] = $this->t('Generate new SSH key pair');
        $defaultOption = 'generate';
      }

      // Check if the user has the permission
      // to use an existing ssh key pair.
      if ($canUseCustomKeypair) {
        $options['use_existing'] = $this->t('Use existing SSH key pair');
        if (!$defaultOption) {
          $defaultOption = 'use_existing';
        }
      }

      $form['key_pair_type'] = [
        '#type' => 'radios',
        '#title' => $this->t('SSH key pair'),
        '#required' => TRUE,
        '#options' => $options,
        '#default_value' => $defaultOption,
        '#ajax' => [
          'callback' => '::keyPairTypeCallback',
          'wrapper' => 'key-pair-type-wrapper',
        ],
      ];

      $form['key_pair_type_wrapper'] = [
        '#type' => 'container',
        '#attributes' => ['id' => 'key-pair-type-wrapper'],
      ];

      // Add fields for private key and public key **ONLY**
      // if this is a new System Credential entity.
      if ($keyPairType === "use_existing" && $canUseCustomKeypair) {
        $form['key_pair_type_wrapper']["tapis_system_credential_publicKey"] = [
          '#type' => 'textarea',
          '#title' => t('Public key'),
          '#description' => t('Enter the corresponding public key for this system credential.'),
          '#required' => TRUE,
          '#weight' => 4,
          '#states' => [
            'visible' => [
              ':input[name="key_pair_type"]' => ['value' => 'use_existing'],
            ],
            'invisible' => [
              ':input[name="key_pair_type"]' => ['value' => 'generate'],
            ],
            'required' => [
              ':input[name="key_pair_type"]' => ['value' => 'use_existing'],
            ],
            'optional' => [
              ':input[name="key_pair_type"]' => ['value' => 'generate'],
            ],
          ],
        ];
        $form['key_pair_type_wrapper']["tapis_system_credential_privateKey"] = [
          '#type' => 'textarea',
          '#title' => t('Private key'),
          '#description' => t('Enter the private key for this system credential. The private key is stored in a remote secure vault only.'),
          '#required' => TRUE,
          '#weight' => 5,
          '#states' => [
            'visible' => [
              ':input[name="key_pair_type"]' => ['value' => 'use_existing'],
            ],
            'invisible' => [
              ':input[name="key_pair_type"]' => ['value' => 'generate'],
            ],
            'required' => [
              ':input[name="key_pair_type"]' => ['value' => 'use_existing'],
            ],
            'optional' => [
              ':input[name="key_pair_type"]' => ['value' => 'generate'],
            ],
          ],
        ];
      }
    }
    else {
      // Hide the following fields: system, loginUser.
      $form['system']['#access'] = FALSE;
      $form['loginUser']['#access'] = FALSE;
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function keyPairTypeCallback(array &$form, FormStateInterface $form_state) {
    return $form['key_pair_type_wrapper'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Enforce rule: system owner, Drupal admin, or public system with
    // dynamic account may add/update credentials.
    $selected_system_id = $form_state->getValue(['system', 0, 'target_id']) ?? NULL;
    if (!$selected_system_id) {
      return;
    }

    $system = $this->entityTypeManager->getStorage('node')->load($selected_system_id);
    if (!$system) {
      return;
    }

    $current_account = $this->currentUser();
    $is_admin = $current_account->hasPermission('administer tapis system');
    if ($is_admin || ((string) $system->getOwnerId() === (string) $current_account->id())) {
      return; // Allowed.
    }

    // Must be dynamic account.
    $use_static = FALSE;
    $f = $system->get(DrupalIds::SYSTEM_USE_STATIC_SYSTEM_USER);
    if ($f && $f->first()) {
      $use_static = (bool) ($f->first()->getValue()['value'] ?? FALSE);
    }
    if ($use_static) {
      $form_state->setError($form, $this->t('Only systems that use dynamic accounts allow adding credentials by non-owners.'));
      return;
    }

    // Must be public in Tapis.
    $tenant_ref = $system->get(DrupalIds::SYSTEM_TENANT)->first()->getValue()['target_id'] ?? NULL;
    $tapis_id = $system->get(DrupalIds::SYSTEM_TAPIS_ID)->first()->getValue()['value'] ?? NULL;
    if (!$tenant_ref || !$tapis_id) {
      $form_state->setError($form, $this->t('This system is not ready to accept credentials.'));
      return;
    }

    try {
      /** @var \Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface $tapisSystemProvider */
      $tapisSystemProvider = \Drupal::service('tapis_system.tapis_system_provider');
      $owner_uid = $system->getOwnerId();
      $state = $tapisSystemProvider->getSystem($tenant_ref, $tapis_id, $owner_uid);
      $is_public = !empty($state['isPublic']);
      if (!$is_public) {
        $form_state->setError($form, $this->t('Only public systems allow adding credentials by non-owners.'));
      }
    }
    catch (\Throwable $e) {
      $form_state->setError($form, $this->t('Unable to verify public status for the selected system. Please try again.'));
    }
  }

  /**
   * AJAX callback to refresh the Username element when system changes.
   */
  public function systemSelectCallback(array &$form, FormStateInterface $form_state) {
    return $form['loginUser'];
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $key_pair_type = $form_state->getValue("key_pair_type");
    $public_key = $form_state->getValue("tapis_system_credential_publicKey");
    $private_key = $form_state->getValue("tapis_system_credential_privateKey");

    /** @var \Drupal\tapis_system\TapisSystemCredentialInterface $entity */
    $entity = $this->getEntity();
    $is_entity_new = $entity->isNew();

    if ($is_entity_new) {
      $systemId = $entity->getSystemId();
      // $system = \Drupal::entityTypeManager()
      // ->getStorage("node")->load($systemId);
      $system = $this->entityTypeManager->getStorage("node")->load($systemId);

      if ($key_pair_type === "generate") {
        // Comment <system host>-nickname.
        $systemHost = $system->get(DrupalIds::SYSTEM_HOST)->getValue()[0]["value"];
        $nickname = $entity->label();
        $credentialComment = $systemHost . " - " . $nickname;

        // Generate a new SSH key pair.
        $keyPair = tapis_system_generate_ssh_keypair($credentialComment);
        $public_key = $keyPair["public_key"];
        $private_key = $keyPair["private_key"];
      }

      // Create the Tapis definition for this system credential
      // using the following fields:
      // $systemId, $systemOwnerUid, $accessorUserId,
      // $accessorLoginUser, $privateKey, $publicKey.
      $tapisDefinition = [];
      $tapisDefinition['systemId'] = $system->get(DrupalIds::SYSTEM_TAPIS_ID)->getValue()[0]['value'];
      $tapisDefinition['systemOwnerUid'] = $system->get(DrupalIds::SYSTEM_OWNER)->getValue()[0]['target_id'];

      // $tapisDefinition['systemEffectiveUserId'] =
      // $system->get(DrupalIds::SYSTEM_EFFECTIVE_USER)->getValue()[0]['value'];
      if (isset($system->get(DrupalIds::SYSTEM_EFFECTIVE_USER)->getValue()[0]['value'])) {
        $tapisDefinition['systemEffectiveUserId'] = $system->get(DrupalIds::SYSTEM_EFFECTIVE_USER)->getValue()[0]['value'];
      }
      else {
        $tapisDefinition['systemEffectiveUserId'] = NULL;
      }
      // $tapisDefinition['accessorUserId'] = \Drupal::currentUser()->id();
      $tapisDefinition['accessorUserId'] = $this->currentUser()->id();
      // Determine accessor login user: for static systems, use effective user
      // and enforce on entity; for dynamic, take value from form.
      $selected_system_id = $entity->getSystemId();
      $selected_system = $this->entityTypeManager->getStorage('node')->load($selected_system_id);
      $use_static = FALSE;
      if ($selected_system) {
        $f = $selected_system->get(DrupalIds::SYSTEM_USE_STATIC_SYSTEM_USER);
        if ($f && $f->first()) {
          $use_static = (bool) ($f->first()->getValue()['value'] ?? FALSE);
        }
      }
      if ($use_static) {
        $effective = $selected_system->get(DrupalIds::SYSTEM_EFFECTIVE_USER)->getValue()[0]['value'] ?? '';
        $tapisDefinition['accessorLoginUser'] = $effective;
        // Ensure entity value is set even if field is disabled.
        $entity->set('loginUser', $effective);
      }
      else {
        $tapisDefinition['accessorLoginUser'] = $form_state->getValue('loginUser')[0]['value'] ?? '';
      }

      $tapisDefinition['privateKey'] = $private_key;
      $tapisDefinition['publicKey'] = $public_key;

      // NOTE: These 2 lines below are NECESSARY for Tapis
      // to process the credentials correctly!
      $tapisDefinition['privateKey'] = str_replace("\r", '', $tapisDefinition['privateKey']);
      $tapisDefinition['publicKey'] = str_replace("\r", '', $tapisDefinition['publicKey']);

      if ($key_pair_type === "generate") {
        $tapisDefinition['skipCredentialCheck'] = TRUE;
      }
      else {
        $tapisDefinition['skipCredentialCheck'] = FALSE;
      }

      $entity->setPublicKey($tapisDefinition['publicKey']);
      $entity->setTapisDefinition($tapisDefinition);
    }

    try {
      $result = parent::save($form, $form_state);
    }
    catch (\Exception $e) {
      if ($e instanceof TapisException || strpos(strtolower($e->getMessage()), 'tapis') !== FALSE) {
        if ($e instanceof TapisException) {
          $exception_message = "The system credential could not be saved: " . $e->getPrimaryMessage() ?: "The system credential could not be saved.";
        }
        else {
          $exception_message = "The system credential could not be saved.";
        }
        $details = [
          '#type' => 'details',
          '#title' => 'Additional Details',
          '#open' => FALSE,
          '#children' => $e->getMessage(),
        ];
        $container = [
          '#type' => 'container',
          '#attributes' => ['class' => ['my-message-container']],
          '#children' => [
            'message' => [
              '#markup' => $exception_message,
        // Add a <p> tag before the message text.
              '#prefix' => '<p>',
        // Add a </p> tag after the message text.
              '#suffix' => '</p>',
            ],
            'details' => $details,
          ],
        ];
        $this
          ->messenger()
          ->addError($container);
      }
      else {
        $this
          ->messenger()
          ->addError($this
            ->t("The system credential could not be saved. Please try again."));
      }

      $form_state
        ->setRebuild();
      return;
    }

    $message_arguments = ['%label' => $entity->toLink()->toString()];
    $logger_arguments = [
      '%label' => $entity->label(),
      'link' => $entity->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New system credential %label has been created.', $message_arguments));
        $this->logger('tapis_system')->notice('Created new system credential %label', $logger_arguments);
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The system credential %label has been updated.', $message_arguments));
        $this->logger('tapis_system')->notice('Updated system credential %label.', $logger_arguments);
        break;
    }

    $form_state->setRedirect('entity.tapis_system_credential.canonical', ['tapis_system_credential' => $entity->id()]);
    return $result;
  }

}
