<?php

namespace Drupal\tapis_system\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\NodeInterface;
use Drupal\tapis_system\DrupalIds;
use Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class TestTapisSystemCredentialForm.
 *
 * Form for checking the credentials of a Tapis system.
 */
class TestTapisSystemCredentialForm extends FormBase {
  /**
   * Node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected NodeInterface $system;

  /**
   * The Tapis system provider.
   *
   * @var \Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface
   */
  protected TapisSystemProviderInterface $tapisSystemProvider;

  /**
   * Constructs a new AppLaunchForm object.
   *
   * @param \Drupal\tapis_system\TapisProvider\TapisSystemProviderInterface $tapisSystemProvider
   *   The Tapis system provider.
   */
  public function __construct(TapisSystemProviderInterface $tapisSystemProvider) {
    $this->tapisSystemProvider = $tapisSystemProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tapis_system.tapis_system_provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'test_tapis_system_credential_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL) {
    $this->system = $node;
    $form['test_button'] = [
      '#type' => 'button',
      '#button_type' => 'primary',
      '#value' => t('Check system access'),
      '#prefix' => t('</br>'),
      '#suffix' => t('</br>'),
      '#ajax' => [
        'callback' => '::ajax_handler',
        'event' => 'click',
      ],
    ];

    // Attach the library for pop-up dialogs/modals.
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
    return $form;
  }

  /**
   * Ajax handler for the "Check system access" button.
   *
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   */
  public function ajax_handler(array $form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    // $tapisSystemProvider =
    // \Drupal::service("tapis_system.tapis_system_provider");
    $tenantId = $this->system->get(DrupalIds::SYSTEM_TENANT)->first()->getValue()['target_id'];
    $systemId = $this->system->get(DrupalIds::SYSTEM_TAPIS_ID)->first()->getValue()['value'];
    $systemOwnerUid = $this->system->getOwnerId();
    // $accessorUserId = \Drupal::currentUser()->id();
    $accessorUserId = $this->currentUser()->id();

    $systemConnectionStatus = $this->tapisSystemProvider->checkUserCredential($tenantId, $systemId, $systemOwnerUid, $accessorUserId);
    $tapisSystemHost = $this->system->get(DrupalIds::SYSTEM_HOST)->first()->getValue()['value'];

    if ($systemConnectionStatus) {
      $title = 'Success';
      // Set $content['#markup'] to be: "Succeeded in accessing
      // the <system> system.", where <system> is
      // the name of the system and links to the system.
      $content['#markup'] = t('Succeeded in accessing the @system system.', [
        '@system' => $this->system->label(),
      ]);
    }
    else {
      $title = 'Error';
      $content['#markup'] = t('Failed to access the @system system.<br/><br/>Follow the steps below to complete the setup:<br/><br/>1. Manually login to this system: @host<br/>2. Append the public key displayed to your ~/.ssh/authorized_keys file<br/>3. Verify system access by clicking Check system access button.', [
        '@system' => $this->system->label(),
        '@host' => $tapisSystemHost,
      ]);
    }
    $response->addCommand(new OpenModalDialogCommand($title, $content,
      [
        'width' => '400',
        'height' => '500',
      ]
    ));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
  }

}
