<?php

namespace Drupal\tapis_system\TapisProvider;

use Drupal\node\NodeInterface;

/**
 * Provides a service for interacting with Tapis systems.
 */
interface TapisSystemProviderInterface {

  /**
   * Get a system's details from a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function getSystem($tenantId, $systemId, $uid = -1);

  /**
   * Create a new system in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param array $tapisDefinition
   *   The Tapis definition array.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function createSystem($tenantId, array $tapisDefinition, $uid = -1);

  /**
   * Update a system in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param array $tapisDefinition
   *   The Tapis definition array.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function updateSystem($tenantId, $systemId, array $tapisDefinition, $uid = -1);

  /**
   * Enable a system in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function enableSystem($tenantId, $systemId, $uid = -1);

  /**
   * Disable a system in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function disableSystem($tenantId, $systemId, $uid = -1);

  /**
   * Check if a system ID exists in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function doesSystemIdExist($tenantId, $systemId, $uid = -1);

  /**
   * Create a scheduler profile in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param array $tapisDefinition
   *   The Tapis definition array.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function createSchedulerProfile($tenantId, array $tapisDefinition, $uid = -1);

  /**
   * Get a scheduler profile from Tapis.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $schedulerProfileId
   *   The scheduler profile ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function getSchedulerProfile($tenantId, $schedulerProfileId, $uid = -1);

  /**
   * Delete a scheduler profile in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $schedulerProfileId
   *   The scheduler profile ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function deleteSchedulerProfile($tenantId, $schedulerProfileId, $uid = -1);

  /**
   * Create a user system credential in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param string $systemOwnerUid
   *   The system owner UID.
   * @param string $systemEffectiveUserId
   *   The system effective user ID.
   * @param string $accessorUserId
   *   The accessor user ID.
   * @param string $accessorLoginUser
   *   The accessor login user.
   * @param string $privateKey
   *   The private key.
   * @param string $publicKey
   *   The public key.
   * @param bool $skipCredentialCheck
   *   Skip credential check flag (optional, defaults to FALSE).
   */
  public function createUserCredential($tenantId, $systemId, $systemOwnerUid, $systemEffectiveUserId, $accessorUserId, $accessorLoginUser, $privateKey, $publicKey, $skipCredentialCheck = FALSE);

  /**
   * Check a user's system credentials in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param string $systemOwnerUid
   *   The system owner UID.
   * @param string $accessorUserId
   *   The accessor user ID.
   */
  public function checkUserCredential($tenantId, $systemId, $systemOwnerUid, $accessorUserId);

  /**
   * Remove a user system credential in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param string $systemOwnerUid
   *   The system owner UID.
   * @param string $systemEffectiveUserId
   *   The system effective user ID.
   * @param string $accessorUserId
   *   The accessor user ID.
   */
  public function removeUserCredential($tenantId, $systemId, $systemOwnerUid, $systemEffectiveUserId, $accessorUserId);

  /**
   * Convert a System Drupal node to a Tapis definition array.
   *
   * @param \Drupal\node\NodeInterface $system
   *   The Drupal node representing the system.
   */
  public function convertSystemToJSON(NodeInterface $system);

  /**
   * Convert a System scheduler profile Drupal node to a Tapis definition array.
   *
   * @param \Drupal\node\NodeInterface $schedulerProfile
   *   The Drupal node representing the scheduler profile.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function convertSchedulerProfileToJSON(NodeInterface $schedulerProfile, $uid = -1);

  /**
   * Change a system's owner in a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param string $systemOwnerUid
   *   The current system owner UID.
   * @param string $newOwnerUid
   *   The new system owner UID.
   */
  public function changeSystemOwner($tenantId, $systemId, $systemOwnerUid, $newOwnerUid);

  /**
   * Share a system with a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function shareSystemWithTenant($tenantId, $systemId, $uid = -1);

  /**
   * Unshare a system with a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function unshareSystemWithTenant($tenantId, $systemId, $uid = -1);

  /**
   * Get the effective user for a system.
   *
   * ("${apiUser}" or a static system user, etc.).
   *
   * @param string $system
   *   The system ID.
   */
  public function getEffectiveUserForSystem($system);

  /**
   * Get filesystem stat info for a path on a system.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param string $path
   *   The path.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function getSystemStatInfo($tenantId, $systemId, $path, $uid = -1);

  /**
   * Grants a file system permission to a user on a system.
   *
   * (this is for the Tapis Files API, NOT the posix file system).
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param string $username
   *   The username.
   * @param string $permission
   *   The permission.
   * @param string $path
   *   The path (optional, defaults to "/").
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function grantSystemPermission($tenantId, $systemId, $username, $permission, $path = "/", $uid = -1);

  /**
   * Revokes a file system permission from a user on a system.
   *
   * (this is for the Tapis Files API, NOT the posix file system).
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param string $username
   *   The username.
   * @param string $path
   *   The path (optional, defaults to "/").
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function revokeSystemPermission($tenantId, $systemId, $username, $path = "/", $uid = -1);

  /**
   * Share a system with a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function getShareSystemWithTenant($tenantId, $systemId, $uid = -1);

  /**
   * Share a system with a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param int $shareUserId
   *   The share user ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function shareSystemWithUser($tenantId, $systemId, $shareUserId, $uid = -1);

  /**
   * Share a system with a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $systemId
   *   The system ID.
   * @param int $shareUserId
   *   The share user ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function unshareSystemWithUser($tenantId, $systemId, $shareUserId, $uid = -1);

}
