<?php

namespace Drupal\tapis_tenant\EventSubscriber;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\tapis_tenant\Exception\TapisException;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber for handling exceptions of type TapisTenantException.
 */
class TapisTenantEventSubscriber implements EventSubscriberInterface {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The messenger interface.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a new MyClass object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   */
  public function __construct(
        RequestStack $request_stack,
        MessengerInterface $messenger,
        LoggerChannelFactoryInterface $loggerFactory
    ) {
    $this->requestStack = $request_stack;
    $this->messenger = $messenger;
    $this->logger = $loggerFactory->get('tapis_tenant');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
          $container->get('request_stack'),
          $container->get('messenger'),
          $container->get('logger.factory')
      );
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[KernelEvents::EXCEPTION][] = ['onKernelException'];
    return $events;
  }

  /**
   * Overrides the response object with the exception's message.
   *
   * @param \Symfony\Component\HttpKernel\Event\ExceptionEvent $event
   *   The exception event.
   */
  public function onKernelException(ExceptionEvent $event): void {
    $exception = $event->getThrowable();

    // If the exception is of type TapisTenantException or
    // the exception message's lower case contains the word "tapis",
    // override the response
    // We need to consider the second case,
    // because sometimes this could be a Drupal exception
    // (EntityStorageException) that contains the word "tapis" in the message.
    if ($exception instanceof TapisException || strpos(strtolower($exception->getMessage()), 'tapis') !== FALSE) {
      // Build a details element with proper child content. Avoid setting
      // '#children' directly (it is populated by the renderer) and instead
      // add nested render arrays.
      $details = [
        '#type' => 'details',
        '#title' => 'Additional Details',
        '#open' => FALSE,
        'content' => [
          // Allow a safe subset of admin HTML in the details message.
          '#markup' => Xss::filterAdmin($exception->getMessage()),
        ],
      ];

      $exception_message = "Could not complete your request.";
      if ($exception instanceof TapisException) {
        $exception_message = $exception->getPrimaryMessage() ?: $exception_message;
      }

      // Build the container correctly by adding children as keys on the
      // render array (rather than using '#children'). Avoid disallowed tags
      // like <p> or <br> which are stripped from status messages.
      $container = [
        '#type' => 'container',
        '#attributes' => ['class' => ['my-message-container']],
        'message' => [
          // Allow a safe subset of admin HTML in the primary message.
          '#markup' => Xss::filterAdmin($exception_message),
        ],
        'details' => $details,
      ];

      /** @var \Drupal\Core\Render\RendererInterface $renderer */
      $renderer = \Drupal::service('renderer');
      $rendered_html = $renderer->renderRoot($container);

      // Pass the rendered HTML to the messenger service
      $this->messenger->addError($rendered_html);

      // Log the exception's message.
      // \Drupal::logger('tapis_tenant')->error($exception->getMessage());
      $this->logger->error($exception->getMessage());

      $currentRequest = $this->requestStack->getCurrentRequest();
      if ($exception instanceof TapisException) {
        // Check if we should reload the current page.
        if ($exception->shouldReloadSamePage()) {
          // Redirect to the current page url
          // Reload the current page URL using a RedirectResponse.
          // $current_url = \Drupal::request()->getUri();
          $current_url = $currentRequest ? $currentRequest->getUri() : '';
          $response = new RedirectResponse($current_url, Response::HTTP_FOUND);
        }
        else {
          // Redirect to the error page with a URL-encoded message.
          $response = new RedirectResponse('/tapis/error?msg=' . rawurlencode($exception->getMessage()), 301);
        }
      }
      else {
        // Reload the current page URL using a RedirectResponse.
        // $current_url = \Drupal::request()->getUri();
        $current_url = $currentRequest ? $currentRequest->getUri() : '';
        $response = new RedirectResponse($current_url, Response::HTTP_FOUND);
      }
      $event->setResponse($response);
    }
  }

}
