<?php

namespace Drupal\tapis_tenant\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\node\NodeForm;
use Drupal\tapis_tenant\Exception\TapisException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class TapisNodeForm.
 *
 * This class extends the NodeForm class, and overrides
 * the save() method to handle Tapis exceptions.
 *
 * @package Drupal\tapis_tenant\Form
 * @phpstan-ignore-next-line
 */
class TapisNodeForm extends NodeForm {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a TapisNodeForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The temp store factory.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface|null $entity_type_bundle_info
   *   The entity type bundle info.
   * @param \Drupal\Component\Datetime\TimeInterface|null $time
   *   The time service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter.
   */
  public function __construct(
        EntityTypeManagerInterface $entity_type_manager,
        EntityRepositoryInterface $entity_repository,
        PrivateTempStoreFactory $temp_store_factory,
        EntityTypeBundleInfoInterface $entity_type_bundle_info = NULL,
        TimeInterface $time = NULL,
        AccountInterface $current_user,
        DateFormatterInterface $date_formatter
    ) {
    $this->entityTypeManager = $entity_type_manager;
    // Call the parent constructor.
    parent::__construct(
          $entity_repository,
          $temp_store_factory,
          $entity_type_bundle_info,
          $time,
          $current_user,
          $date_formatter
      );
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('entity_type.manager'),
          $container->get('entity.repository'),
          $container->get('tempstore.private'),
          $container->get('entity_type.bundle.info'),
          $container->get('datetime.time'),
          $container->get('current_user'),
          $container->get('date.formatter')
      );
  }

  /**
   * Overrides the save() method from the NodeForm class.
   *
   * This method overrides the save() method from the NodeForm class,
   * in order to handle Tapis exceptions.
   *
   * @param array $form
   *   From array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function save(array $form, FormStateInterface $form_state): void {
    $node = $this->entity;
    // Get the bundle id.
    $bundle = $node->bundle();
    $bundle_entity_type = $this->entityTypeManager->getStorage('node_type')->load($bundle);
    $bundle_label = strtolower($bundle_entity_type->label());
    // Check if the bundle starts with 'tapis_'.
    if (strpos($bundle, 'tapis_') !== 0) {
      // If not, call the parent save method.
      parent::save($form, $form_state);
      return;
    }

    $success = FALSE;

    try {
      parent::save($form, $form_state);
      $success = TRUE;
    }
    catch (\Exception | \Throwable $e) {
      if ($e instanceof TapisException || strpos(strtolower($e->getMessage()), 'tapis') !== FALSE) {
        if ($e instanceof TapisException) {
          $exception_message = "The $bundle_label could not be saved: " .
                    $e->getPrimaryMessage() ?: "The $bundle_label could not be saved.";
        }
        else {
          $exception_message = "The $bundle_label could not be saved.";
        }
        $details = [
          '#type' => 'details',
          '#title' => 'Additional Details',
          '#open' => FALSE,
          '#children' => $e->getMessage(),
        ];
        $container = [
          '#type' => 'container',
          '#attributes' => ['class' => ['my-message-container']],
          '#children' => [
            'message' => [
              '#markup' => $exception_message,
          // Add a <p> tag before the message text.
              '#prefix' => '<p>',
          // Add a </p> tag after the message text.
              '#suffix' => '</p>',
            ],
            'details' => $details,
          ],
        ];
        $this
          ->messenger()
          ->addError($container);
      }
      else {
        $this
          ->messenger()
          ->addError($this
            ->t('The @label could not be saved. Please try again.', ['@label' => $bundle_label]));
      }

      $form_state
        ->setRebuild();
    }

    if ($success && $bundle === "tapis_system") {
      // Once a Tapis system has been created,
      // redirect the user to the system credential form,
      // with a query parameter '?systemId=<system id>'.
      // Check if the system uses a static system user.
      $usesStaticSystemUser = boolval($node->get("tapis_sys_use_static_system_user")->getValue()[0]['value']);

      // If the system uses a static system user,
      // redirect to the system credential form.
      if ($usesStaticSystemUser) {
        $systemEffectiveUser = $node->get("tapis_sys_effective_user")->getValue()[0]['value'];
        $system_id = $node->id();
        $url = Url::fromRoute(
              'entity.tapis_system_credential.add_form',
              [
                'systemId' => $system_id,
                'loginUser' => $systemEffectiveUser,
              ]
          );
      }
      else {
        $system_id = $node->id();
        $url = Url::fromRoute(
              'entity.tapis_system_credential.add_form',
              [
                'systemId' => $system_id,
              ]
                );
      }
      $form_state->setRedirectUrl($url);
    }
  }

}
