<?php

namespace Drupal\tapis_tenant\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\tapis_tenant\DrupalIds;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'MaintenanceModeBlock' Block.
 *
 * @Block(
 *   id = "tapis_tenant_maintenance_mode_block",
 *   admin_label = @Translation("Tapis Maintenance Mode"),
 *   category = @Translation("Tapis Maintenance Mode"),
 * )
 */
class MaintenanceModeBlock extends BlockBase implements ContainerFactoryPluginInterface  {

  /**
   * The provider for retrieving Tapis site tenant information.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */

  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new MaintenanceModeBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        TapisSiteTenantProviderInterface $tapisSiteTenantProvider,
        EntityTypeManagerInterface $entityTypeManager
    ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition
    ): static {
    return new static(
          $configuration,
          $plugin_id,
          $plugin_definition,
          $container->get("tapis_tenant.tapis_site_tenant_provider"),
          $container->get('entity_type.manager')
      );
  }

  /**
   * {@inheritdoc}
   *
   * This block will display a list of Tapis tenants
   * that are in maintenance mode.
   */
  public function build(): array {
    // Load all Tapis tenants.
    // $tapisSiteTenantProvider =
    // \Drupal::service("tapis_tenant.tapis_site_tenant_provider");
    // $tapisTenantNodeIds = \Drupal::entityQuery('node')->accessCheck(FALSE)
    // ->condition('type', DrupalIds::NODE_BUNDLE_TENANT)->execute();
    $tapisTenantNodeIds = $this->entityTypeManager->getStorage('node')->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', DrupalIds::NODE_BUNDLE_TENANT)
      ->execute();

    $maintenance_mode_tenants = [];
    foreach ($tapisTenantNodeIds as $tapisTenantNodeId) {
      $maintenance_mode = $this->tapisSiteTenantProvider->isTenantInMaintenanceMode($tapisTenantNodeId);
      if ($maintenance_mode) {
        // $tenantNode = \Drupal::entityTypeManager()->getStorage('node')->load($tapisTenantNodeId);
        $tenantNode = $this->entityTypeManager->getStorage('node')->load($tapisTenantNodeId);
        $label = $tenantNode->label();
        $link = $tenantNode->toUrl()->toString();
        $maintenance_mode_tenants[] = "<a href='$link'>$label</a>";
      }
    }

    if (count($maintenance_mode_tenants) > 0) {
      return [
        '#markup' => $this->t(
            'The following Tapis tenants are in maintenance mode: @mode',
            ['@mode' => implode(", ", $maintenance_mode_tenants)]
        ),
      ];
    }
    else {
      return [
        '#markup' => $this->t('None of your Tapis tenants are in maintenance mode.'),
      ];
    }
  }

  /**
   * {@inheritdoc}
   *
   * This block will only be visible
   * if there is at least one Tapis tenant in maintenance mode.
   */
  protected function blockAccess(AccountInterface $account) {
    // Load all Tapis sites.
    // $tapisSiteTenantProvider = \
    // \Drupal::service("tapis_tenant.tapis_site_tenant_provider");
    // $tapisTenantNodeIds = \Drupal::entityQuery('node')->accessCheck(FALSE)
    // ->condition('type', DrupalIds::NODE_BUNDLE_TENANT)->execute();
    $tapisTenantNodeIds = $this->entityTypeManager->getStorage('node')->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', DrupalIds::NODE_BUNDLE_TENANT)
      ->execute();

    $maintenance_mode_tenants = [];
    foreach ($tapisTenantNodeIds as $tapisTenantNodeId) {
      $maintenance_mode = $this->tapisSiteTenantProvider->isTenantInMaintenanceMode($tapisTenantNodeId);
      if ($maintenance_mode) {
        // $tenantNode = \Drupal::entityTypeManager()->getStorage('node')->load($tapisTenantNodeId);
        $tenantNode = $this->entityTypeManager->getStorage('node')->load($tapisTenantNodeId);
        $label = $tenantNode->label();
        $maintenance_mode_tenants[] = $label;
      }
    }

    // If there are no tenants in maintenance mode, then don't show the block.
    return AccessResult::allowedIf(count($maintenance_mode_tenants) > 0);
  }

}
