<?php

namespace Drupal\tapis_tenant\TapisProvider;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\node\NodeInterface;
use Drupal\node\NodeStorageInterface;
use Drupal\tapis_tenant\DrupalIds;
use Drupal\tapis_tenant\Exception\TapisTenantException;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * This class is used by the tapis_tenant module for making API calls to Tapis.
 */
class TapisSiteTenantProvider implements TapisSiteTenantProviderInterface {

  const TAPIS_API_VERSION = "v3";

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * Drupal's entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */

  protected EntityTypeManagerInterface $entityTypeManager;
  /**
   * The entity storage for Nodes.
   *
   * @var \Drupal\node\NodeStorageInterface
   */

  protected NodeStorageInterface $nodeStorage;
  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */

  protected KeyRepositoryInterface $keyRepository;
  /**
   * If TRUE, ignore maintenance mode and don't throw any errors.
   *
   * @var bool
   */
  private bool $ignoreMaintenanceMode;

  /**
   * {@inheritdoc}
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __construct(
    ClientInterface $httpClient,
    EntityTypeManagerInterface $entityTypeManager,
    KeyRepositoryInterface $keyRepository
  ) {
    $this->httpClient = $httpClient;
    $this->entityTypeManager = $entityTypeManager;
    $this->nodeStorage = $this->entityTypeManager->getStorage("node");
    $this->keyRepository = $keyRepository;
    $this->ignoreMaintenanceMode = FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client'),
      $container->get('entity_type.manager'),
      $container->get("key.repository")
    );
  }

  /**
   * {@inheritdoc}
   */
  public function setIgnoreMaintenanceMode($ignore): void {
    $this->ignoreMaintenanceMode = $ignore;
  }

  /**
   * {@inheritdoc}
   */
  public function doesTapisSiteExist($tapisSiteId, $siteAdminTenantBaseUrl): bool {
    $tapis_api_url = "$siteAdminTenantBaseUrl/" . self::TAPIS_API_VERSION . "/sites/$tapisSiteId";

    $response = $this->httpClient->request('GET', $tapis_api_url, ['http_errors' => FALSE]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    \Drupal::logger('tapis_tenant')->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200) {
      $e = new TapisTenantException("Could not verify if the Tapis site '$tapisSiteId' exists");
      $e->setReloadSamePage(TRUE);
      throw $e;
    }

    if ($response_body['status'] !== "success") {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function doesTapisTenantExist($tapisTenantId, $siteAdminTenantBaseUrl): bool {
    $tapis_api_url = "$siteAdminTenantBaseUrl/" . self::TAPIS_API_VERSION . "/tenants/$tapisTenantId";

    $response = $this->httpClient->request('GET', $tapis_api_url, ['http_errors' => FALSE]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    \Drupal::logger('tapis_tenant')->debug(print_r($response_body, TRUE));

    // @todo Figure out how to handle Tapis api call errors
    if ($response->getStatusCode() !== 200) {
      $e = new TapisTenantException("Could not verify if the Tapis tenant '$tapisTenantId' exists.");
      $e->setReloadSamePage(TRUE);
      throw $e;
    }

    if ($response_body['status'] !== "success") {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_tenant\Exception\TapisTenantException|\GuzzleHttp\Exception\GuzzleException
   */
  public function getTapisSiteServices($tapisSiteId, $siteAdminTenantBaseUrl) {
    $tapis_api_url = "$siteAdminTenantBaseUrl/" . self::TAPIS_API_VERSION . "/sites/$tapisSiteId";
    $response = $this->httpClient->request('GET', $tapis_api_url, ['http_errors' => FALSE]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    \Drupal::logger('tapis_tenant')->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: '<br>Please try again later.';
      }
      throw new TapisTenantException("Could not fetch the services for the Tapis site '$tapisSiteId'. $message");
    }

    if ($response_body['status'] !== "success") {
      return [];
    }
    return $response_body['result']['services'];
  }

  /**
   * {@inheritdoc}
   */
  public function getTapisServiceHealth($siteAdminTenantBaseUrl, $service): bool {
    $services_with_no_healthcheck_endpoint = [
      'actors',
      'pgrest',
      'pods',
      'tenants',
      'authenticator',
    ];
    // @todo How do we determine health of services that don't have an official health check endpoint?
    if (in_array($service, $services_with_no_healthcheck_endpoint)) {
      return TRUE;
    }

    $tapis_api_url = "$siteAdminTenantBaseUrl/" . self::TAPIS_API_VERSION . "/$service/healthcheck";
    if ($service === "tokens") {
      $tapis_api_url = "$siteAdminTenantBaseUrl/" . self::TAPIS_API_VERSION . "/$service/hello";
    }
    $response = $this->httpClient->request('GET', $tapis_api_url, ['http_errors' => FALSE]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    \Drupal::logger('tapis_tenant')->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200) {
      return FALSE;
    }

    return $response_body['status'] === "success";
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_tenant\Exception\TapisTenantException
   */
  public function getTenantInfo($tenantId): array {
    $tenant = $this->nodeStorage->load($tenantId);

    // Throw an exception if the tenant is in maintenance mode.
    if (!$this->ignoreMaintenanceMode && $this->isTenantInMaintenanceMode($tenantId)) {
      throw new TapisTenantException("This Tapis site is in maintenance mode, please try again later.");
    }
    return $this->getTenantInfoFromTenant($tenant);
  }

  /**
   * {@inheritdoc}
   */
  public function isTenantInMaintenanceMode($tenantId) {
    $tenant = $this->nodeStorage->load($tenantId);
    if (isset($tenant->get(DrupalIds::MAINTENANCE_MODE_TENANT)->getValue()[0])) {
      return boolval($tenant->get(DrupalIds::MAINTENANCE_MODE_TENANT)->getValue()[0]['value']);
    } else {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_tenant\Exception\TapisTenantException
   */
  public function getTenantInfoFromTenant(NodeInterface $tenant): array {
    // Throw an exception if the tenant's site is in maintenance mode.
    if (!$this->ignoreMaintenanceMode && $this->isTenantInMaintenanceMode($tenant->id())) {
      throw new TapisTenantException("This Tapis site is in maintenance mode, please try again later.");
    }

    $authenticator_service_password_key_id = $tenant->get(DrupalIds::TENANT_AUTHENTICATOR_SERVICE_PASSWORD)->getValue()[0]['target_id'];

    return [
      'id' => $tenant->id(),

      'tapis_id' => $tenant->get(DrupalIds::TENANT_TAPIS_ID)->getValue()[0]['value'],
      'tapis_api_endpoint' => $tenant->get(DrupalIds::TENANT_API_ENDPOINT)->getValue()[0]['uri'],
      'maintenance_mode' => $this->isTenantInMaintenanceMode($tenant->id()),

      'tenant' => $tenant,

      'tapis_site_id' => $tenant->get(DrupalIds::TENANT_SITE_TAPIS_ID)->getValue()[0]['value'],
      'tapis_site_admin_tenant_id' => $tenant->get(DrupalIds::TENANT_SITE_ADMIN_TENANT_ID)->getValue()[0]['value'],
      'tapis_site_admin_tenant_api_endpoint' => $tenant->get(DrupalIds::TENANT_SITE_ADMIN_TENANT_API_ENDPOINT)->getValue()[0]['uri'],

      'satellite_proxy_url' => $tenant->get(DrupalIds::TENANT_SATELLITE_PROXY_URL)->getValue()[0]['uri'],
      'apps_domain' => $tenant->get(DrupalIds::TENANT_APPS_BASE_DOMAIN)->getValue()[0]['value'],

      'services' => [
        'authenticator' => [
          'service' => $tenant->get(DrupalIds::TENANT_AUTHENTICATOR_SERVICE)->getValue()[0]['value'],
          'service_password' => $authenticator_service_password_key_id ? $this->keyRepository->getKey($authenticator_service_password_key_id)->getKeyValue() : NULL,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_tenant\Exception\TapisTenantException
   */
  public function getTenantInfoFromFormState($tenant, &$form_state): array {
    // Throw an exception if the tenant's site is in maintenance mode.
    if (!$this->ignoreMaintenanceMode && $this->isTenantInMaintenanceMode($tenant->id())) {
      throw new TapisTenantException("This Tapis site is in maintenance mode, please try again later.");
    }

    $authenticator_service_password_key_id = $form_state->getValue(DrupalIds::TENANT_AUTHENTICATOR_SERVICE_PASSWORD)[0]['target_id'];

    if (isset($form_state->getValue(DrupalIds::MAINTENANCE_MODE_TENANT)[0]['value'])) {
      $maintenanceMode = $form_state->getValue(DrupalIds::MAINTENANCE_MODE_TENANT)[0]['value'];
    }
    else {
      $maintenanceMode = FALSE;
    }
    return [
      'id' => $tenant->id(),

      'tapis_id' => $form_state->getValue(DrupalIds::TENANT_TAPIS_ID)[0]['value'],
      'tapis_api_endpoint' => $form_state->getValue(DrupalIds::TENANT_API_ENDPOINT)[0]['uri'],
      'maintenance_mode' => boolval($maintenanceMode),

      'tenant' => $tenant,

      'tapis_site_id' => $form_state->getValue(DrupalIds::TENANT_SITE_TAPIS_ID)[0]['value'],
      'tapis_site_admin_tenant_id' => $form_state->getValue(DrupalIds::TENANT_SITE_ADMIN_TENANT_ID)[0]['value'],
      'tapis_site_admin_tenant_api_endpoint' => $form_state->getValue(DrupalIds::TENANT_SITE_ADMIN_TENANT_API_ENDPOINT)[0]['uri'],

      'satellite_proxy_url' => $form_state->getValue(DrupalIds::TENANT_SATELLITE_PROXY_URL)[0]['uri'],
      'apps_domain' => $form_state->getValue(DrupalIds::TENANT_APPS_BASE_DOMAIN)[0]['value'],

      'services' => [
        'authenticator' => [
          'service' => $form_state->getValue(DrupalIds::TENANT_AUTHENTICATOR_SERVICE)[0]['value'],
          'service_password' => $authenticator_service_password_key_id ? $this->keyRepository->getKey($authenticator_service_password_key_id)->getKeyValue() : NULL,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getDrupalFieldNameMappingsForTenants(): array {
    return [
      'tapis_id' => DrupalIds::TENANT_TAPIS_ID,
      "tapis_api_endpoint" => DrupalIds::TENANT_API_ENDPOINT,
      "maintenance_mode" => DrupalIds::MAINTENANCE_MODE_TENANT,

      "tapis_site_id" => DrupalIds::TENANT_SITE_TAPIS_ID,
      "tapis_site_admin_tenant_id" => DrupalIds::TENANT_SITE_ADMIN_TENANT_ID,
      "tapis_site_admin_tenant_api_endpoint" => DrupalIds::TENANT_SITE_ADMIN_TENANT_API_ENDPOINT,

      "satellite_proxy_url" => DrupalIds::TENANT_SATELLITE_PROXY_URL,
      "apps_domain" => DrupalIds::TENANT_APPS_BASE_DOMAIN,

      'services' => [
        'authenticator' => [
          'service' => DrupalIds::TENANT_AUTHENTICATOR_SERVICE,
          'service_password' => DrupalIds::TENANT_AUTHENTICATOR_SERVICE_PASSWORD,
        ],
      ],
    ];
  }

}
