<?php

namespace Drupal\tapis_tenant\TapisProvider;

use Drupal\node\NodeInterface;

/**
 * This file defines an interface for accessing Tapis APIs.
 *
 * @file TapisSiteTenantProviderInterface.php
 */

/**
 * This interface outlines how this module interacts w/ Tapis.
 */
interface TapisSiteTenantProviderInterface {

  /**
   * Ignore maintenance mode.
   *
   * (don't throw errors if a Tapis site/tenant is in maintenance mode).
   *
   * @param bool $ignore
   *   Maintenance Mode flag.
   */
  public function setIgnoreMaintenanceMode($ignore);

  /**
   * Check if this tenant is in maintenance mode.
   *
   * @param int $tenantId
   *   The Tapis tenant ID.
   */
  public function isTenantInMaintenanceMode($tenantId);

  /**
   * Check whether a Tapis site with the given id & admin base url exists.
   *
   * @param string $tapisSiteId
   *   The id of the site in Tapis.
   * @param string $siteAdminTenantBaseUrl
   *   The base url for the Tapis site's admin tenant.
   *
   * @returns TRUE if the site exists, FALSE otherwise
   */
  public function doesTapisSiteExist($tapisSiteId, $siteAdminTenantBaseUrl);

  /**
   * Check whether a Tapis tenant with the given id & admin base url exists.
   *
   * @param string $tapisTenantId
   *   The id of the tenant in Tapis.
   * @param string $siteAdminTenantBaseUrl
   *   The base url for the tenant's site's admin tenant.
   *
   * @returns TRUE if the tenant exists, FALSE otherwise
   */
  public function doesTapisTenantExist($tapisTenantId, $siteAdminTenantBaseUrl);

  /**
   * Get a Tapis tenant's information using its ID.
   *
   * @param int $tenantId
   *   The Tapis tenant id.
   *
   * @returns mixed
   *   An associative array containing the tenant's info.
   */
  public function getTenantInfo($tenantId);

  /**
   * Get a Tapis tenant's information using its Node entity.
   *
   * @param \Drupal\node\NodeInterface $tenant
   *   The Tapis tenant Node entity.
   *
   * @returns mixed
   *   An associative array containing the tenant's info.
   */
  public function getTenantInfoFromTenant(NodeInterface $tenant);

  /**
   * Get a Tapis tenant's information using its form state.
   */
  public function getTenantInfoFromFormState(NodeInterface $tenant, &$form_state);

  /**
   * Get a mapping from the Tapis tenant info keys.
   *
   * To their Drupal field machine names.
   */
  public function getDrupalFieldNameMappingsForTenants();

  /**
   * Get a list of the Tapis services enabled for the given site id.
   *
   * @param int $tapisSiteId
   *   The Tapis site id.
   * @param string $siteAdminTenantBaseUrl
   *   The Tapis site's admin tenant's base url.
   *
   * @return array
   *   of strings representing Tapis services
   */
  public function getTapisSiteServices($tapisSiteId, $siteAdminTenantBaseUrl);

  /**
   * Check the health for a given Tapis site's service.
   *
   * @param string $siteAdminTenantBaseUrl
   *   The Tapis site's admin tenant's base url.
   * @param string $service
   *   The Tapis service to check the health for.
   *
   * @return bool
   *   indicating the health of the Tapis service.
   *   True if healthy, false otherwise
   */
  public function getTapisServiceHealth($siteAdminTenantBaseUrl, $service);

}
