<?php

namespace Drupal\tapis_tenant;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * {@inheritdoc}
 */
class TapisSettings implements TapisSettingsInterface {
  /**
   * Drupal's module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Drupal's entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Tapis site tenant provider.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */
  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The 'key' module's key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The mudule handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The enetity type manager.
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository.
   */
  public function __construct(
    ModuleHandlerInterface $moduleHandler,
    EntityTypeManagerInterface $entityTypeManager,
    TapisSiteTenantProviderInterface $tapisSiteTenantProvider,
    KeyRepositoryInterface $keyRepository
  ) {
    $this->moduleHandler = $moduleHandler;
    $this->entityTypeManager = $entityTypeManager;
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
    $this->keyRepository = $keyRepository;
  }

  /**
   * Create instances.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
      $container->get('tapis_tenant.tapis_site_tenant_provider'),
      $container->get("key.repository")
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validateTenantSettings($tenantId, $newSettings): array {
    $validation = ['is_valid' => TRUE, 'errors' => []];
    $settings = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);

    $drupalFieldNameMappings = $this->tapisSiteTenantProvider->getDrupalFieldNameMappingsForTenants();

    foreach ($newSettings as $key => $value) {
      if ($key === "services") {
        foreach ($value as $service => $service_info) {
          $service_name_field_name = $drupalFieldNameMappings['services'][$service]['service'];
          $service_password_field_name = $drupalFieldNameMappings['services'][$service]['service_password'];

          // Ensure that this service name is a non-empty string.
          $service_name = $service_info['service'];
          if (!is_string($service_name) || empty($service_name)) {
            $validation['errors'][] = [
              'field_name' => $service_name_field_name,
              'error' => "The name for $service is invalid.",
            ];
            break;
          }

          // Ensure that this service password key exists.
          if (!$this->moduleHandler->moduleExists("key")) {
            $validation['errors'][] = [
              'field_name' => $service_password_field_name,
              'error' => "The key for $service's service password doesn't exist.",
            ];
            break;
          }

          $servicePassword = $service_info['service_password'];
          if (!$servicePassword) {
            $validation['errors'][] = [
              'field_name' => $service_password_field_name,
              'error' => "The key for $service's service password doesn't exist.",
            ];
            break;
          }
        }
      }
      elseif ($key === "tenant" || $key === "tapis_id") {
        continue;
      }
      elseif (!!$settings[$key] && $value !== $settings[$key] && !$this->canUpdateTenantSetting($key)) {
        // We're changing this setting key but we're not allowed to change it
        // (and the old value is truthy).
        $field_name = $drupalFieldNameMappings[$key] ?? NULL;
        $validation['errors'][] = [
          'field_name' => $field_name,
          'error' => "You cannot change this setting.",
        ];
      }
    }
    if (count($validation['errors']) > 0) {
      $validation['is_valid'] = FALSE;
    }
    return $validation;
  }

  /**
   * {@inheritdoc}
   */
  public function canUpdateTenantSetting($settingKey): bool {
    if ($settingKey === "id") {
      return FALSE;
    }

    $settings_that_can_always_be_updated = [
      "tapis_api_endpoint",
      "service_account_username",
      "maintenance_mode",

      "tapis_site_admin_tenant_api_endpoint",

      "satellite_proxy_url",
      "apps_domain",

      "services",
    ];
    if (in_array($settingKey, $settings_that_can_always_be_updated)) {
      return TRUE;
    }

    // If the setting key is not in the list of settings that can always be
    // updated, then we can only update it if there are no Drupal Tapis entities
    // in the site.
    // For ex: most of the Tapis site-related fields are
    // in this second category.
    // Check if any Drupal Tapis entities exist in the site.
    // tapis_auth.
    if ($this->moduleHandler->moduleExists("tapis_auth")) {
      // Check if any Tapis tokens exist.
      $numTapisTokens = $this->entityTypeManager
        ->getStorage('tapis_token')
        ->getQuery()
        ->accessCheck(FALSE)
        ->count()
        ->execute();

      if ($numTapisTokens > 0) {
        return FALSE;
      }
    }

    // tapis_system.
    if ($this->moduleHandler->moduleExists("tapis_system")) {
      // Check if any Tapis systems exist.
      $numTapisSystems = $this->entityTypeManager
        ->getStorage('node')
        ->getQuery()
        ->accessCheck(FALSE)
        ->condition('type', 'system')
        ->count()
        ->execute();

      if ($numTapisSystems > 0) {
        return FALSE;
      }

      // Check if any Tapis system credentials exist.
      $numTapisSystemCredentials = $this->entityTypeManager
        ->getStorage('tapis_system_credential')
        ->getQuery()
        ->accessCheck(FALSE)
        ->count()
        ->execute();

      if ($numTapisSystemCredentials > 0) {
        return FALSE;
      }
    }

    // tapis_app.
    if ($this->moduleHandler->moduleExists("tapis_app")) {
      // Check if any Tapis apps exist.
      $numTapisApps = $this->entityTypeManager
        ->getStorage('node')
        ->getQuery()
        ->accessCheck(FALSE)
        ->condition('type', 'app')
        ->count()
        ->execute();

      if ($numTapisApps > 0) {
        return FALSE;
      }
    }

    // tapis_job.
    if ($this->moduleHandler->moduleExists("tapis_job")) {
      // Check if any Tapis jobs exist.
      $numTapisJobs = $this->entityTypeManager
        ->getStorage('tapis_job')
        ->getQuery()
        ->accessCheck(FALSE)
        ->count()
        ->execute();

      if ($numTapisJobs > 0) {
        return FALSE;
      }

      // Check if any Tapis job access links exist.
      $numTapisJobAccessLinks = $this->entityTypeManager
        ->getStorage('job_access_link')
        ->getQuery()
        ->accessCheck(FALSE)
        ->count()
        ->execute();

      if ($numTapisJobAccessLinks > 0) {
        return FALSE;
      }
    }

    return TRUE;
  }

}
