<?php

namespace Drupal\tavily\Plugin\AiFunctionCall;

use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Drupal\tavily\TavilyApi;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of getting a summary back.
 */
#[FunctionCall(
  id: 'tavily:search',
  function_name: 'tavily_search',
  name: 'Tavily Search',
  description: 'This can search the web and give back urls and can also scrape them.',
  group: 'information_tools',
  context_definitions: [
    'query' => new ContextDefinition(
      data_type: 'string',
      label: 'Query',
      description: 'The query to search for on Tavily.',
      required: TRUE,
    ),
    'auto_parameters' => new ContextDefinition(
      data_type: 'boolean',
      label: 'Auto Parameters',
      description: 'Whether to automatically add parameters to the query from context.',
      required: FALSE,
    ),
    'topic' => new ContextDefinition(
      data_type: 'string',
      label: 'Topic',
      description: 'The category of the seach, only needed if auto_parameters is FALSE. genera or news is possible. news will return news articles, general will return general search results.',
      required: FALSE,
      default_value: 'general',
      constraints: [
        'Choice' => [
          'general',
          'news',
        ],
      ],
    ),
    'chunks_per_source' => new ContextDefinition(
      data_type: 'integer',
      label: 'Chunks per Source',
      description: 'The number of chunks to return per source.',
      required: FALSE,
      default_value: 3,
    ),
    'max_results' => new ContextDefinition(
      data_type: 'integer',
      label: 'Max Results',
      description: 'The maximum number of results to return.',
      required: FALSE,
      default_value: 5,
    ),
    'days' => new ContextDefinition(
      data_type: 'integer',
      label: 'Days',
      description: 'The number of days to look back for results. Only used if topic is news.',
      required: FALSE,
      default_value: 7,
    ),
    'include_raw_content' => new ContextDefinition(
      data_type: 'string',
      label: 'Include Raw Content',
      description: 'Whether to include the raw content of the pages in the results. Will return in markdown without html',
      required: FALSE,
      default_value: FALSE,
      constraints: [
        'Choice' => [
          'True',
          'False',
          'html',
          'markdown',
        ],
      ],
    ),
    'include_images' => new ContextDefinition(
      data_type: 'boolean',
      label: 'Include Images',
      description: 'Whether to include images in the results.',
      required: FALSE,
      default_value: FALSE,
    ),
    'include_image_descriptions' => new ContextDefinition(
      data_type: 'boolean',
      label: 'Include Image Descriptions',
      description: 'Whether to include image descriptions in the results.',
      required: FALSE,
      default_value: FALSE,
    ),
    'include_favicon' => new ContextDefinition(
      data_type: 'boolean',
      label: 'Include Favicon',
      description: 'Whether to include the favicon of the source in the results.',
      required: FALSE,
      default_value: FALSE,
    ),
    'country' => new ContextDefinition(
      data_type: 'string',
      label: 'Country',
      description: 'Only available if the topic is general. This boosts the results for a specific country.',
      required: FALSE,
      default_value: 'united states',
      constraints: [
        'Choice' => [
          'afghanistan',
          'albania',
          'algeria',
          'andorra',
          'angola',
          'argentina',
          'armenia',
          'australia',
          'austria',
          'azerbaijan',
          'bahamas',
          'bahrain',
          'bangladesh',
          'barbados',
          'belarus',
          'belgium',
          'belize',
          'benin',
          'bhutan',
          'bolivia',
          'bosnia and herzegovina',
          'botswana',
          'brazil',
          'brunei',
          'bulgaria',
          'burkina faso',
          'burundi',
          'cambodia',
          'cameroon',
          'canada',
          'cape verde',
          'central african republic',
          'chad',
          'chile',
          'china',
          'colombia',
          'comoros',
          'congo',
          'costa rica',
          'croatia',
          'cuba',
          'cyprus',
          'czech republic',
          'denmark',
          'djibouti',
          'dominican republic',
          'ecuador',
          'egypt',
          'el salvador',
          'equatorial guinea',
          'eritrea',
          'estonia',
          'ethiopia',
          'fiji',
          'finland',
          'france',
          'gabon',
          'gambia',
          'georgia',
          'germany',
          'ghana',
          'greece',
          'guatemala',
          'guinea',
          'haiti',
          'honduras',
          'hungary',
          'iceland',
          'india',
          'indonesia',
          'iran',
          'iraq',
          'ireland',
          'israel',
          'italy',
          'jamaica',
          'japan',
          'jordan',
          'kazakhstan',
          'kenya',
          'kuwait',
          'kyrgyzstan',
          'latvia',
          'lebanon',
          'lesotho',
          'liberia',
          'libya',
          'liechtenstein',
          'lithuania',
          'luxembourg',
          'madagascar',
          'malawi',
          'malaysia',
          'maldives',
          'mali',
          'malta',
          'mauritania',
          'mauritius',
          'mexico',
          'moldova',
          'monaco',
          'mongolia',
          'montenegro',
          'morocco',
          'mozambique',
          'myanmar',
          'namibia',
          'nepal',
          'netherlands',
          'new zealand',
          'nicaragua',
          'niger',
          'nigeria',
          'north korea',
          'north macedonia',
          'norway',
          'oman',
          'pakistan',
          'panama',
          'papua new guinea',
          'paraguay',
          'peru',
          'philippines',
          'poland',
          'portugal',
          'qatar',
          'romania',
          'russia',
          'rwanda',
          'saudi arabia',
          'senegal',
          'serbia',
          'singapore',
          'slovakia',
          'slovenia',
          'somalia',
          'south africa',
          'south korea',
          'south sudan',
          'spain',
          'sri lanka',
          'sudan',
          'sweden',
          'switzerland',
          'syria',
          'taiwan',
          'tajikistan',
          'tanzania',
          'thailand',
          'togo',
          'trinidad and tobago',
          'tunisia',
          'turkey',
          'turkmenistan',
          'uganda',
          'ukraine',
          'united arab emirates',
          'united kingdom',
          'united states',
          'uruguay',
          'uzbekistan',
          'venezuela',
          'vietnam',
          'yemen',
          'zambia',
          'zimbabwe',
        ],
      ],
    ),
  ],
)]
class TavilySearch extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * The Tavily API service.
   *
   * @var \Drupal\tavily\TavilyApi
   */
  protected TavilyApi $tavilyApi;

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->tavilyApi = $container->get('tavily.api');
    $instance->currentUser = $container->get('current_user');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Collect the context values.
    $query = $this->getContextValue('query');
    $auto_parameters = $this->getContextValue('auto_parameters', TRUE);
    $topic = $this->getContextValue('topic', 'general');
    $chunks_per_source = $this->getContextValue('chunks_per_source', 3);
    $max_results = $this->getContextValue('max_results', 5);
    $days = $this->getContextValue('days', 7);
    $include_raw_content = $this->getContextValue('include_raw_content', FALSE);
    $include_images = $this->getContextValue('include_images', FALSE);
    $include_image_descriptions = $this->getContextValue('include_image_descriptions', FALSE);
    $include_favicon = $this->getContextValue('include_favicon', FALSE);
    $country = $this->getContextValue('country', 'united states');

    // Convert include_raw_content to boolean if it's a boolean string.
    if (in_array(strtolower($include_raw_content), ['true', 'false'], TRUE)) {
      $include_raw_content = strtolower($include_raw_content) === 'true';
    }

    // Check so the query is really set.
    if (empty($query)) {
      throw new \InvalidArgumentException('The query context value is required and cannot be empty.');
    }

    // Check so the user has use tavily tools permission.
    if (!$this->currentUser->hasPermission('use tavily tools')) {
      throw new \Exception('You do not have permission to access this function.');
    }

    // Prepare the options for the Tavily API.
    $options = [
      'auto_parameters' => $auto_parameters,
      'topic' => $topic,
      'chunks_per_source' => $chunks_per_source,
      'max_results' => $max_results,
      'days' => $days,
      'include_raw_content' => $include_raw_content,
      'include_images' => $include_images,
      'include_image_descriptions' => $include_image_descriptions,
      'include_favicon' => $include_favicon,
      'country' => $country,
    ];
    // Call the Tavily API search method.
    $result = $this->tavilyApi->search($query, $options);
    $this->setOutput(Yaml::dump($result, 10, 2));
  }

}
