<?php

declare(strict_types=1);

namespace Drupal\tawk_to\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\Utility\Token;
use Drupal\tawk_to\Cache\TawkToCacheManager;

/**
 * Defines the rendering tawk.to service.
 */
class TawkToEmbedRender implements TrustedCallbackInterface {

  /**
   * The tawk.to embed URL.
   */
  const string EMBED_URL = 'https://embed.tawk.to';

  /**
   * The widget page id.
   */
  protected string $widgetPageId;

  /**
   * The widget id.
   */
  protected string $widgetId;

  /**
   * The user name.
   */
  protected string $userName;

  /**
   * The user email.
   */
  protected string $userEmail;

  /**
   * The cache manager.
   */
  protected TawkToCacheManager $cacheManager;

  /**
   * The condition plugin definition.
   */
  protected TawkToConditionPluginsHandler $conditionPluginsHandler;

  /**
   * Constructs the TawkToEmbedRender.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\tawk_to\Service\TawkToConditionPluginsHandler $conditionPluginsHandler
   *   The tawk.to access controller handler.
   * @param \Drupal\tawk_to\Cache\TawkToCacheManager $cacheManager
   *   The cache manager.
   */
  public function __construct(ConfigFactoryInterface $configFactory, Token $token, TawkToConditionPluginsHandler $conditionPluginsHandler, TawkToCacheManager $cacheManager) {
    $this->conditionPluginsHandler = $conditionPluginsHandler;
    $this->cacheManager = $cacheManager;
    $this->widgetPageId = $configFactory->get('tawk_to.settings')->get('tawk_to_widget_page_id');
    $this->widgetId = $configFactory->get('tawk_to.settings')->get('tawk_to_widget_id');
    if ($configFactory->get('tawk_to.settings')->get('show_user_name')) {
      $userName = $configFactory->get('tawk_to.settings')->get('user_name');
      $this->userName = $token->replace($userName, [], ['clear' => TRUE]);
    }
    if ($configFactory->get('tawk_to.settings')->get('show_user_email')) {
      $userEmail = $configFactory->get('tawk_to.settings')->get('user_email');
      $this->userEmail = $token->replace($userEmail, [], ['clear' => TRUE]);
    }
  }

  /**
   * Checks access to the current requests and return renderable/empty array.
   *
   * @return array
   *   The render renderable array.
   */
  public function render(): ?array {
    if ($this->widgetPageId === '' || $this->widgetId === '') {
      return [];
    }
    if ($this->conditionPluginsHandler->checkAccess()) {
      return [
        '#theme' => 'tawk_to',
        '#items' => [
          'page_id' => $this->widgetPageId,
          'widget_id' => $this->widgetId,
          'embed_url' => self::EMBED_URL,
          'user_name' => $this->userName,
          'user_email' => $this->userEmail,
        ],
        '#cache' => [
          'contexts' => $this->cacheManager->getCacheContexts(),
          'tags' => $this->cacheManager->getCacheTags(),
        ],
      ];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks(): array {
    return ['render'];
  }

}
