<?php

namespace Drupal\taxonomy_enum\Drush;

use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drush\Commands\field\EntityTypeBundleValidationTrait;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush hooks for integrating the taxonomy_enum module with other commands.
 */
class TaxonomyEnumHooks extends DrushCommands {

  use EntityTypeBundleValidationTrait;

  /**
   * Constructs a TaxonomyEnumHooks object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
  ) {
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
    );
  }

  /**
   * Add taxonomy_enum related options to the command.
   */
  #[CLI\Hook(type: HookManager::OPTION_HOOK, target: 'vocabulary:create')]
  public function hookOption(Command $command): void {
    if (!$this->isInstalled()) {
      return;
    }

    $command->addOption(
      'enum-class',
      '',
      InputOption::VALUE_OPTIONAL,
      'The enum class to use for this vocabulary.'
    );

    $command->addOption(
      'enum-lock-terms',
      '',
      InputOption::VALUE_OPTIONAL,
      'Whether to lock terms (true/false).'
    );

    $command->addOption(
      'enum-delete-orphaned',
      '',
      InputOption::VALUE_OPTIONAL,
      'Whether to delete orphaned terms (true/false).'
    );
  }

  /**
   * Prompt for values for the taxonomy_enum related options.
   */
  #[CLI\Hook(type: HookManager::ON_EVENT, target: 'vocabulary-set-options')]
  public function hookSetOptions(): void {
    if (!$this->isInstalled()) {
      return;
    }

    $this->input->setOption(
      'enum-class',
      $this->input->getOption('enum-class') ?? $this->askClass()
    );

    if (!$this->input->getOption('enum-class')) {
      return;
    }

    $this->input->setOption(
      'enum-lock-terms',
      $this->input->getOption('enum-lock-terms') ?? $this->askLockTerms()
    );

    $this->input->setOption(
      'enum-delete-orphaned',
      $this->input->getOption('enum-delete-orphaned') ?? $this->askDeleteOrphaned()
    );
  }

  /**
   * Handle field storage creation for taxonomy_enum fields.
   */
  #[CLI\Hook(type: HookManager::ON_EVENT, target: 'field-create-field-storage')]
  public function hookFieldStorage(array $values, InputInterface $input): array {
    if ($input->getOption('field-type') === 'taxonomy_enum') {
      $values['settings']['target_type'] = 'taxonomy_term';
    }

    return $values;
  }

  /**
   * Handle field config creation for taxonomy_enum fields.
   */
  #[CLI\Hook(type: HookManager::ON_EVENT, target: 'field-create-field-config')]
  public function hookFieldConfig(array $values, InputInterface $input): array {
    if ($input->getOption('field-type') === 'taxonomy_enum') {
      $values['settings']['handler_settings']['target_bundles'] = $this->getTargetBundles($input);
    }

    return $values;
  }

  /**
   * Assign the enum class to the vocabulary being created.
   */
  #[CLI\Hook(type: HookManager::ON_EVENT, target: 'vocabulary-create')]
  public function hookCreate(array &$values): void {
    if (!$this->isInstalled()) {
      return;
    }

    $values['third_party_settings']['taxonomy_enum']['enum_name'] = $this->input()->getOption('enum-class');
    $values['third_party_settings']['taxonomy_enum']['lock_terms'] = $this->input()->getOption('enum-lock-terms');
    $values['third_party_settings']['taxonomy_enum']['delete_orphaned_terms'] = $this->input()->getOption('enum-delete-orphaned');
    $values['dependencies']['module'][] = 'taxonomy_enum';
  }

  /**
   * Get target bundles from input or by prompting the user.
   */
  protected function getTargetBundles(InputInterface $input): ?array {
    // For the 'target_bundles' setting, a NULL value is equivalent to "allow
    // entities from any bundle to be referenced" and an empty array value is
    // equivalent to "no entities from any bundle can be referenced".
    $targetBundles = NULL;

    if ($referencedBundle = $input->getOption('target-bundle')) {
      $this->validateBundle('taxonomy_term', $referencedBundle);
      $referencedBundles = [$referencedBundle];
    }
    else {
      $referencedBundles = $this->askReferencedBundles();
    }

    if (!empty($referencedBundles)) {
      $targetBundles = array_combine($referencedBundles, $referencedBundles);
    }

    return $targetBundles;
  }

  /**
   * Prompt for referenced bundles.
   */
  protected function askReferencedBundles(): ?array {
    $storage = $this->entityTypeManager
      ->getStorage('taxonomy_vocabulary');
    $ids = $storage->getQuery()
      ->exists('third_party_settings.taxonomy_enum.enum_name')
      ->accessCheck(FALSE)
      ->execute();

    if ($ids === []) {
      return [];
    }

    $bundles = $storage->loadMultiple($ids);
    $choices = [];

    foreach ($bundles as $bundle) {
      $label = $this->input->getOption('show-machine-names') ? $bundle->id() : $bundle->label();
      $choices[$bundle->id()] = $label;
    }

    return $this->io()->multiselect('Referenced bundles', $choices);
  }

  /**
   * Prompt for the enum class.
   */
  protected function askClass(): ?string {
    return $this->io()->ask('Enum class');
  }

  /**
   * Prompt whether to lock terms.
   */
  protected function askLockTerms(): ?string {
    return $this->io()->confirm('Lock terms', FALSE);
  }

  /**
   * Prompt whether to delete orphaned terms.
   */
  protected function askDeleteOrphaned(): ?string {
    return $this->io()->confirm('Delete orphaned terms', FALSE);
  }

  /**
   * Check whether the taxonomy_enum module is installed.
   */
  protected function isInstalled(): bool {
    return $this->moduleHandler->moduleExists('taxonomy_enum');
  }

}
