<?php

declare(strict_types=1);

namespace Drupal\taxonomy_machine_name_field_formatter\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Entity\Exception\UndefinedLinkTemplateException;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceFormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;

#[FieldFormatter(
  id: 'taxonomy_machine_name_machine_name_label',
  label: new TranslatableMarkup('Machine name with Label'),
  description: new TranslatableMarkup('Machine name with a Label'),
  field_types: [
    'entity_reference',
  ],
)]
class MachineNameLabelFormatter extends EntityReferenceFormatterBase {

  public static function defaultSettings() {
    return [
        'link' => FALSE,
        'seperated_link' => TRUE,
        'raw' => FALSE,
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */

  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements['raw'] = [
      '#title' => $this->t('Send raw values to the field template.'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('raw'),
      '#description' => $this->t('Raw values will disable the link option as no render elements are being send to the template.'),
    ];

    $elements['link'] = [
      '#title' => $this->t('Link label to the referenced entity'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('link'),
      '#states' => [
        'enabled' => [
          ':input[name*="[settings][raw]"]' => ['checked' => FALSE],
        ],
      ],
    ];
    $elements['seperated_link'] = [
      '#title' => $this->t('Link and label as seperated items'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('seperated_link'),
      '#states' => [
        'enabled' => [
          ':input[name*="[settings][link]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    if ($this->getSetting('raw')) {
      return [$this->t('Raw mode is enabled')];
    }

    $link_state = $this->getSetting('link');
    $summary = [];
    $summary[] = $link_state ? $this->t('Link to the referenced entity') : $this->t('No link');
    if ($link_state) {
      $summary[] = $this->getSetting('seperated_link') ? $this->t('Link and label are seperated') : $this->t('Label is a link');
    }
    return $summary;
  }

  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $as_link = (bool) $this->getSetting('link');
    $separate_elements = $as_link && (bool) $this->getSetting('seperated_link');
    $is_raw = (bool) $this->getSetting('raw');

    $elements = [];
    foreach ($this->getEntitiesToView($items, $langcode) as $delta => $term) {
      if ($is_raw) {
        $elements[$delta]['machine_name'] = (string) $term->get('machine_name')->value;
        $elements[$delta]['label'] = (string) $term->label();
        continue;
      }

      $elements[$delta]['machine_name']['#markup'] = $term->get('machine_name')->value;
      if (!$as_link) {
        $elements[$delta]['label']['#markup'] = $term->label();
        continue;
      }
      $uri = $this->getUri($term);
      if (!$uri) {
        $elements[$delta]['label']['#markup'] = $term->label();
        continue;
      }
      if ($separate_elements) {
        $elements[$delta]['link']['#markup'] = $uri->toString();
        $elements[$delta]['label']['#markup'] = $term->label();
      }
      else {
        $elements[$delta]['label'] = [
          '#type' => 'link',
          '#title' => $term->label(),
          '#url' => $uri,
          '#options' => $uri->getOptions(),
        ];
      }
    }

    return $elements;
  }

  private function getUri(EntityInterface $term): ?Url {
    $uri = NULL;
    if (!$term->isNew()) {
      try {
        $uri = $term->toUrl();
      }
      catch (UndefinedLinkTemplateException|EntityMalformedException $e) {
        // These exceptions are by \Drupal\Core\Entity\Entity::toUrl()
        // And should not happen on taxonomy term at all,
        // But just in case it does, we catch it for you
      }
    }
    return $uri;
  }

  public static function isApplicable(FieldDefinitionInterface $field_definition): bool {
    return $field_definition->getFieldStorageDefinition()
        ->getSetting('target_type') === 'taxonomy_term';
  }

}
