<?php

declare(strict_types=1);

namespace Drupal\Tests\taxonomy_machine_name_field_formatter\Kernel;

use Drupal;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Url;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Kernel tests for MachineNameLabelFormatter::viewElements().
 *
 * @group taxonomy_machine_name_field_formatter
 */
class MachineNameLabelFormatterKernelTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'entity_test',
    'taxonomy',
    'taxonomy_machine_name',
    'taxonomy_machine_name_field_formatter',
  ];

  /**
   * The vocabulary ID used for tests.
   */
  protected const VID = 'test_vocab';

  /**
   * The field name used for tests.
   */
  protected const FIELD_NAME = 'field_terms';

  /** @var \Drupal\taxonomy\Entity\Term */
  protected Term $term;

  /** @var \Drupal\entity_test\Entity\EntityTest */
  protected EntityTest $entity;

  protected function setUp(): void {
    parent::setUp();

    // Install required entity schemas and basic system tables.
    $this->installEntitySchema('user');
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('entity_test');
    $this->installSchema('system', ['sequences']);

    // Install configuration for taxonomy to allow creating vocabularies.
    $this->installConfig(['system', 'taxonomy']);

    // Create a vocabulary.
    Vocabulary::create([
      'vid' => self::VID,
      'name' => 'Test Vocab',
    ])->save();

    // Create one taxonomy term; taxonomy_machine_name will set machine_name.
    $this->term = Term::create([
      'vid' => self::VID,
      'name' => 'Apple',
    ]);
    $this->term->save();

    // Create an entity reference field on entity_test that targets taxonomy_term.
    FieldStorageConfig::create([
      'field_name' => self::FIELD_NAME,
      'entity_type' => 'entity_test',
      'type' => 'entity_reference',
      'cardinality' => -1,
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
    ])->save();

    FieldConfig::create([
      'field_name' => self::FIELD_NAME,
      'entity_type' => 'entity_test',
      'bundle' => 'entity_test',
      'label' => 'Terms',
      'settings' => [
        'handler' => 'default',
        'handler_settings' => [
          'target_bundles' => [self::VID => self::VID],
        ],
      ],
    ])->save();

    // Create an entity_test entity referencing our term.
    $this->entity = EntityTest::create([]);
    $this->entity->set(self::FIELD_NAME, [
      ['target_id' => $this->term->id()],
    ]);
    $this->entity->save();
  }

  public function testNoLinkRendersMachineNameAndLabelInlineTemplates(): void {
    $items = $this->entity->get(self::FIELD_NAME);

    $formatter = $this->getFormatter($items, [
      'link' => FALSE,
      'seperated_link' => TRUE,
    ]);

    // Ensure referenced entities are prepared for the formatter.
    $formatter->prepareView([$items]);

    $elements = $formatter->viewElements($items, 'en');

    $this->assertArrayHasKey(0, $elements);
    $delta = $elements[0];

    // Machine name inline template is always present.
    $this->assertSame('inline_template', $delta['machine_name']['#type']);
    $this->assertSame($this->term->get('machine_name')->value, $delta['machine_name']['#context']['value']);

    // Label should be a plain inline template (no link).
    $this->assertSame('inline_template', $delta['label']['#type']);
    $this->assertSame('Apple', $delta['label']['#context']['value']);

    // No separate link element should be present when link = FALSE.
    $this->assertArrayNotHasKey('link', $delta);
  }

  public function testLinkSeparatedRendersLinkAndLabelTemplates(): void {
    $items = $this->entity->get(self::FIELD_NAME);

    $formatter = $this->getFormatter($items, [
      'link' => TRUE,
      'seperated_link' => TRUE,
    ]);

    $formatter->prepareView([$items]);
    $elements = $formatter->viewElements($items, 'en');

    $this->assertArrayHasKey(0, $elements);
    $delta = $elements[0];

    $this->assertSame('inline_template', $delta['machine_name']['#type']);

    // Separate link is an inline template with URL string.
    $this->assertSame('inline_template', $delta['link']['#type']);
    $this->assertSame($this->term->toUrl()
      ->toString(), $delta['link']['#context']['value']);

    // Label remains an inline template when separated.
    $this->assertSame('inline_template', $delta['label']['#type']);
    $this->assertSame('Apple', $delta['label']['#context']['value']);
  }

  public function testLinkAsLabelRendersDrupalLink(): void {
    $items = $this->entity->get(self::FIELD_NAME);

    $formatter = $this->getFormatter($items, [
      'link' => TRUE,
      'seperated_link' => FALSE,
    ]);

    $formatter->prepareView([$items]);
    $elements = $formatter->viewElements($items, 'en');

    $this->assertArrayHasKey(0, $elements);
    $delta = $elements[0];

    // Machine name still present.
    $this->assertSame('inline_template', $delta['machine_name']['#type']);

    // Label becomes a link render array.
    $this->assertSame('link', $delta['label']['#type']);
    $this->assertSame('Apple', $delta['label']['#title']);

    $this->assertInstanceOf(Url::class, $delta['label']['#url']);
    $this->assertSame($this->term->toUrl()
      ->toString(), $delta['label']['#url']->toString());
  }

  /**
   * Helper to instantiate the formatter with the provided settings.
   */
  protected function getFormatter(FieldItemListInterface $items, array $settings) {
    $manager = Drupal::service('plugin.manager.field.formatter');
    return $manager->getInstance([
      'field_definition' => $items->getFieldDefinition(),
      'view_mode' => 'full',
      'configuration' => [
        'type' => 'taxonomy_machine_name_machine_name_label',
        'settings' => $settings,
        'label' => 'hidden',
        'third_party_settings' => [],
      ],
    ]);
  }

}
