<?php

declare(strict_types=1);

namespace Drupal\taxonomy_ordinal\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\IntegerFormatter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\taxonomy\TermInterface;
use Drupal\taxonomy\VocabularyInterface;

/**
 * Formatter for the 'taxonomy_ordinal' field type.
 *
 * @FieldFormatter(
 *   id = "taxonomy_ordinal_formatter",
 *   label = @Translation("Taxonomy Ordinal"),
 *   field_types = {
 *     "taxonomy_ordinal",
 *     "integer"
 *   }
 * )
 */
class TaxonomyOrdinalFormatter extends IntegerFormatter {

  const array STRING_FORMATTER_OPTIONS = [
    'formatter_item',
    'formatter_short',
    'formatter_medium',
    'formatter_long'
  ];

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array
  {
    return ['ordinal_format' => 'formatter_item'];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array
  {

    $elements['ordinal_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Ordinal number format'),
      '#options' => [
        'formatter_item' => $this->t('Item, no path'),
        'formatter_short' => $this->t('Short, with taxonomy path'),
        'formatter_medium' => $this->t('Medium, with taxonomy path'),
        'formatter_long' => $this->t('Long, with taxonomy path'),
      ],
      '#default_value' => $this->getSetting('ordinal_format'),
      '#weight' => 0,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array
  {
    $summary = [];

    if ($this->getSetting('ordinal_format')) {
      $summary[] = $this->t('Format: %format', ['%format' => $this->getSetting('ordinal_format')]);
    }

    return $summary;
  }


  /**
   * Restrict formatter so it only shows up for taxonomy_term:weight.
   */
  public static function isApplicable($field_definition): bool {
    // Only applicable to the taxonomy term 'weight' base field.
    return ($field_definition->getType() === 'taxonomy_ordinal')
      || (($field_definition->getTargetEntityTypeId() === 'taxonomy_term') && ($field_definition->getName() === 'weight') && ($field_definition->getType() === 'integer'));
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $formatted_string = '';
    /** @var TermInterface $term */
    $term = NULL;
    $value = (!$items->isEmpty()) ? (int) $items->getString() : 0;

    // Get the required string formatter.
    $format = $this->getSetting('ordinal_format');
    $format = in_array($format, self::STRING_FORMATTER_OPTIONS) ? $format : 'formatter_item';

    /** @var EntityInterface $entity */
    $entity = $items->getParent()->getEntity();

    if ($items->getFieldDefinition()->getType() === 'taxonomy_ordinal') {
      $number_format = $items->getFieldDefinition()->getSetting('number_format') ?? '123';
      $converted_number = $this->convert($value, $number_format);

      $formatter = $items->getFieldDefinition()->getSetting($format) ?? ' %on';
      $formatted_string = str_replace('%on', $converted_number, $formatter);
      if ($format !== 'formatter_item') {
        $term_field = $items->getFieldDefinition()->getSetting('taxonomy_reference');
        if ($entity instanceof EntityInterface && $term_field && $entity->hasField($term_field)) {
          $term = $entity->get($term_field)->entity;
          if ($term instanceof TermInterface) {
            $formatted_taxonomy_path = $this->getFormattedTermPath($term, $format);
            $formatted_string = $formatted_taxonomy_path . $formatted_string;
          }
        }
      }
    } elseif (($items->getFieldDefinition()->getType() === 'integer') && ($entity instanceof TermInterface)) {
      $formatted_string = $this->getFormattedTermPath($entity, $format);
    }

    // $output = parent::viewElements($items, $langcode);
    return !empty($formatted_string) ? [['#markup' => $formatted_string]] : [];
  }

  /**
   * @param TermInterface $term
   *   The term for which to build the term path.
   * @param string $format
   *   The format to use. Expect an item from STRING_FORMATTER_OPTIONS (see above).
   *
   * @return string|null
   *   Ready formatted term path. (E.g. 'Chap. 1, Sec. 3.2')
   */
  protected function getFormattedTermPath(TermInterface $term, string $format): ?string {
    $formatted_string = '';
    if (!in_array($format, self::STRING_FORMATTER_OPTIONS)) {
      return '#?#';
    }
    /** @var string $ordinal_path e.g. '1.3.2' */
    if ($ordinal_path = taxonomy_ordinal__get_ordinal_path($term)) {
      $ordinal_array = explode('.', $ordinal_path);

      /** @var VocabularyInterface $vocab */
      $vocab = Vocabulary::load($term->bundle());
      $number_formats = $vocab->getThirdPartySetting('taxonomy_ordinal', 'number_format');
      $number_formats = ($number_formats) ? explode('|', (string) $number_formats) : [];
      $formatters = $vocab->getThirdPartySetting('taxonomy_ordinal', $format);
      $formatters = ($formatters) ? explode('|', (string) $formatters) : [];
      if ($format == 'formatter_item') {
        // Output just the last part of the ordinal number address.
        $item_level = count($ordinal_array) - 1;
        $number_format = $number_formats[$item_level] ?? '123';
        $formatter = $formatters[$item_level] ?? '.%on';
        $num = (int) $ordinal_array[$item_level] ?? 0;
        $converted_number = $this->convert($num, $number_format);
        $formatted_string = str_replace('%on', $converted_number, $formatter);
      } else {
        foreach ($ordinal_array as $item_level => $number_format) {
          $number_format = $number_formats[$item_level] ?? '123';
          $formatter = $formatters[$item_level] ?? '.%on';
          $num = (int) $ordinal_array[$item_level] ?? 0;
          $converted_number = $this->convert($num, $number_format);
          $formatted_string .= str_replace('%on', $converted_number, $formatter);
        }
      }
    }
    return $formatted_string;
  }

  /**
   * Convert a positive integer to a target notation.
   *
   * @param int $n
   *   1-based positive integer. Values < 1 are invalid.
   * @param string $format
   *   Target notation: "abc", "ABC", "iii", "III", "αβγ", "ΑΒΓ".
   *
   * @return string
   *   Converted representation.
   */
  public function convert(int $n, string $format): string {
    if ($n < 1) {
      // For ordinals/alpha/roman notations, 0 or negatives have no canonical form.
      return '#';
    }

    switch ($format) {
      case 'abc':
        return $this->toLatinAlphabetic($n, true);

      case 'ABC':
        return $this->toLatinAlphabetic($n, false);

      case 'iii':
        return strtolower($this->toRoman($n));

      case 'III':
        return $this->toRoman($n);

      case 'αβγ':
        // Greek lowercase alphabet without final sigma (ς), use σ for consistency.
        $alphabet = [
          'α','β','γ','δ','ε','ζ','η','θ','ι','κ','λ','μ',
          'ν','ξ','ο','π','ρ','σ','τ','υ','φ','χ','ψ','ω',
        ];
        return $this->toBijectiveBase($n, $alphabet);

      case 'ΑΒΓ':
        $alphabet = [
          'Α','Β','Γ','Δ','Ε','Ζ','Η','Θ','Ι','Κ','Λ','Μ',
          'Ν','Ξ','Ο','Π','Ρ','Σ','Τ','Υ','Φ','Χ','Ψ','Ω',
        ];
        return $this->toBijectiveBase($n, $alphabet);

      default:
        return "$n";
    }
  }

  /**
   * Latin alphabet converter (bijective base-26).
   *
   * @param int $n
   * @param bool $lowercase
   *
   * @return string
   */
  private function toLatinAlphabetic(int $n, bool $lowercase = true): string {
    $alphabet = $lowercase ? range('a', 'z') : range('A', 'Z');
    return $this->toBijectiveBase($n, $alphabet);
  }

  /**
   * Generic bijective base-N converter using the provided alphabet.
   *
   * Example: with ['a'..'z'], 1->a, 26->z, 27->aa.
   *
   * @param int $n
   *   1-based positive integer.
   * @param string[] $alphabet
   *   Array of unique "digits" (strings). Length defines the base.
   *
   * @return string
   */
  private function toBijectiveBase(int $n, array $alphabet): string {
    $base = count($alphabet);
    if ($base < 2) {
      // A base of less than 2 is not meaningful for this notation.
      throw new InvalidArgumentException('OrdinalConverter: alphabet must have at least 2 symbols.');
    }

    $result = '';
    while ($n > 0) {
      $n--; // shift to 0-based indexing
      $result = $alphabet[$n % $base] . $result;
      $n = intdiv($n, $base);
    }
    return $result;
  }

  /**
   * Convert integer to Roman numerals (classic additive/subtractive form).
   *
   * Implementation is unbounded in practice; for n > 3999 it continues with repeating 'M'.
   *
   * @param int $n
   *   1-based positive integer.
   *
   * @return string
   */
  private function toRoman(int $n): string {
    $map = [
      1000 => 'M',
      900  => 'CM',
      500  => 'D',
      400  => 'CD',
      100  => 'C',
      90   => 'XC',
      50   => 'L',
      40   => 'XL',
      10   => 'X',
      9    => 'IX',
      5    => 'V',
      4    => 'IV',
      1    => 'I',
    ];

    $res = '';
    foreach ($map as $value => $symbol) {
      if ($n <= 0) {
        break;
      }
      $count = intdiv($n, $value);
      if ($count > 0) {
        // Append the symbol $count times.
        $res .= str_repeat($symbol, $count);
        $n -= $value * $count;
      }
    }
    return $res;
  }

}
