<?php
declare(strict_types=1);

namespace Drupal\taxonomy_ordinal\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\taxonomy\VocabularyForm;

/**
 * Extends the core vocabulary form to inject module-specific settings.
 *
 * This class is used for both the 'add' and 'default' handlers via
 * hook_entity_type_alter(). It only adds fields when the current vocabulary
 * is enabled in module settings; otherwise it behaves like core.
 */
class VocabularyFormExt extends VocabularyForm {

  const string NUMBER_FORMAT_DEFAULT = '123|ABC|iii';
  const string FORMATTER_ITEM_DEFAULT = 'Chapter %on|Section %on| %on';
  const string FORMATTER_SHORT_DEFAULT = '%on|.%on|.%on';
  const string FORMATTER_MEDIUM_DEFAULT = 'Ch. %on|, Sec. %on|.%on';
  const string FORMATTER_LONG_DEFAULT = 'Chapter %on|, Section %on|, %on.';

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Build the original form first.
    $form = parent::buildForm($form, $form_state);

    // Get the vocabulary being edited/created.
    /** @var \Drupal\taxonomy\Entity\Vocabulary $vocabulary */
    $vocabulary = $this->entity;
    $enabled = ($vocabulary instanceof Vocabulary) && taxonomy_ordinal__vocab_enabled($vocabulary->id());
    if (!$enabled) {
      return $form;
    }

    // Read existing third-party settings (for defaults).
    $tp_namespace = 'taxonomy_ordinal';
    $current_defaults = [
      'number_format' => $vocabulary->getThirdPartySetting($tp_namespace, 'number_format', self::NUMBER_FORMAT_DEFAULT),
      'formatter_item' => $vocabulary->getThirdPartySetting($tp_namespace, 'formatter_item', self::FORMATTER_ITEM_DEFAULT),
      'formatter_short' => $vocabulary->getThirdPartySetting($tp_namespace, 'formatter_short', self::FORMATTER_SHORT_DEFAULT),
      'formatter_medium' => $vocabulary->getThirdPartySetting($tp_namespace, 'formatter_medium', self::FORMATTER_MEDIUM_DEFAULT),
      'formatter_long' => $vocabulary->getThirdPartySetting($tp_namespace, 'formatter_long', self::FORMATTER_LONG_DEFAULT),
    ];

    // Add a details element with your custom settings.
    $form['taxonomy_ordinal'] = [
      '#type' => 'details',
      '#title' => $this->t('Taxonomy Ordinal'),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#weight' => 90,
    ];

    $form['taxonomy_ordinal']['number_format'] = [
      '#type' => 'textfield',
      '#default_value' => $current_defaults['number_format'],
      '#title' => $this->t('Number format'),
      '#description' => $this->t("Enter the number format for converting ordinal numbers. Use pipe char '|' to separate nesting levels.<br>Option: '%options'<br>Example: '%example'", [
        '%example' => self::NUMBER_FORMAT_DEFAULT,
        '%options' => '123, abc, ABC, iii, III, αβγ, ΑΒΓ'
      ]),
      '#maxlength' => '255',
    ];

    $form['taxonomy_ordinal']['formatter_item'] = [
      '#type' => 'textfield',
      '#default_value' => $current_defaults['formatter_item'],
      '#title' => $this->t('Item format (no taxonomy path)'),
      '#description' => $this->t("Enter a formatter string. Use pipe char '|' to separate nesting levels, use '%on' where to inject the ordinal number.<br>Example: '%example'", ['%example' => self::FORMATTER_ITEM_DEFAULT]),
      '#maxlength' => '255',
    ];

    $form['taxonomy_ordinal']['formatter_short'] = [
      '#type' => 'textfield',
      '#default_value' => $current_defaults['formatter_short'],
      '#title' => $this->t('Short format (with path)'),
      '#description' => $this->t("Enter a formatter string. Use pipe char '|' to separate nesting levels, use '%on' to inject the ordinal number.<br>Example: '%example'", ['%example' => self::FORMATTER_SHORT_DEFAULT]),
      '#maxlength' => '255',
    ];

    $form['taxonomy_ordinal']['formatter_medium'] = [
      '#type' => 'textfield',
      '#default_value' => $current_defaults['formatter_medium'],
      '#title' => $this->t('Medium format (with path)'),
      '#description' => $this->t("Enter a formatter string. Use pipe char '|' to separate nesting levels, use '%on' to inject the ordinal number.<br>Example: '%example'", ['%example' => self::FORMATTER_MEDIUM_DEFAULT]),
      '#maxlength' => '255',
    ];

    $form['taxonomy_ordinal']['formatter_long'] = [
      '#type' => 'textfield',
      '#default_value' => $current_defaults['formatter_long'],
      '#title' => $this->t('Long format (with path)'),
      '#description' => $this->t("Enter a formatter string. Use pipe char '|' to separate nesting levels, use '%on' to inject the ordinal number.<br>Example: '%example'", ['%example' => self::FORMATTER_LONG_DEFAULT]),
      '#maxlength' => '255',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {
    // Inject/save our settings before parent::submitForm() persists the entity.
    /** @var \Drupal\taxonomy\Entity\Vocabulary $vocabulary */
    $vocabulary = $this->entity;
    if ($vocabulary instanceof Vocabulary && $form_state->hasValue('taxonomy_ordinal')) {
      $values = (array) $form_state->getValue('taxonomy_ordinal');

      // Persist as third-party settings to the vocabulary config entity.
      $vocabulary->setThirdPartySetting('taxonomy_ordinal', 'number_format', ($values['number_format'] ?? self::NUMBER_FORMAT_DEFAULT));
      $vocabulary->setThirdPartySetting('taxonomy_ordinal', 'formatter_item', ($values['formatter_item'] ?? self::FORMATTER_ITEM_DEFAULT));
      $vocabulary->setThirdPartySetting('taxonomy_ordinal', 'formatter_short', ($values['formatter_short'] ?? self::FORMATTER_SHORT_DEFAULT));
      $vocabulary->setThirdPartySetting('taxonomy_ordinal', 'formatter_medium', ($values['formatter_medium'] ?? self::FORMATTER_MEDIUM_DEFAULT));
      $vocabulary->setThirdPartySetting('taxonomy_ordinal', 'formatter_long', ($values['formatter_long'] ?? self::FORMATTER_LONG_DEFAULT));

    }

    // Proceed with core save and messages.
    parent::submitForm($form, $form_state);
  }

}
