<?php

declare(strict_types=1);

namespace Drupal\taxonomy_ordinal\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Field\Plugin\Field\FieldType\IntegerItem;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'taxonomy_ordinal' field type.
 *
 * @FieldType(
 *   id = "taxonomy_ordinal",
 *   label = @Translation("Taxonomy Ordinal"),
 *   description = @Translation("Stores a non-negative integer used as an ordinal."),
 *   category = @Translation("Number"),
 *   default_widget = "taxonomy_ordinal_widget",
 *   default_formatter = "taxonomy_ordinal_formatter",
 *   cardinality = 1,
 * )
 */
class TaxonomyOrdinalItem extends IntegerItem {

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings(): array {
    // Note: IntegerItem uses 'min'/'max' settings for Range constraints.
    return [
        'min' => 0,
        'max' => 999,
        'formatter_item' => '(%on)',
        'formatter_short' => '(%on)',
        'formatter_medium' => '(I %on)',
        'formatter_long' => 'Item %on',
        'number_format' => 'abc',
        // Stores the machine name of the selected taxonomy reference field.
        'taxonomy_reference' => '',
      ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    // Inherit the 'value' property and constraints from IntegerItem.
    $properties = parent::propertyDefinitions($field_definition);
    // IntegerItem will automatically apply a Range constraint using our settings (min/max).
    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    // Same schema as IntegerItem, but explicitly unsigned since min >= 0.
    $schema = parent::schema($field_definition);
    if (isset($schema['columns']['value'])) {
      $schema['columns']['value']['unsigned'] = TRUE;
    }
    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state): array {
    $elements = [];

    // Maximum value setting.
    $elements['max'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum value'),
      '#default_value' => (int) $this->getSetting('max'),
      '#min' => 0,
      '#required' => TRUE,
      '#description' => $this->t('Upper bound for the ordinal (inclusive).'),
    ];

    // Fixed minimum = 0 (informational).
    $elements['min'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum value'),
      '#default_value' => 0,
      '#min' => 0,
      '#disabled' => TRUE,
      '#description' => $this->t('Minimum is fixed to 0 for this field type.'),
    ];

    // Build options: all entity reference fields on the target entity/bundle that reference taxonomy_term.
    $options = $this->buildTaxonomyReferenceOptions();
    $elements['taxonomy_reference'] = [
      '#type' => 'select',
      '#title' => $this->t('Taxonomy'),
      '#options' => $options,
      '#empty_option' => $this->t('- None -'),
      '#default_value' => (string) $this->getSetting('taxonomy_reference'),
      '#description' => $this->t('Select an entity reference field on this entity that references taxonomy terms.'),
    ];

    $elements['number_format'] = [
      '#type' => 'textfield',
      '#default_value' => (string) $this->getSetting('number_format'),
      '#title' => $this->t('Number format'),
      '#description' => $this->t("Enter the number format for converting ordinal numbers.<br>Option: '%options'", ['%options' => '123, abc, ABC, iii, III, αβγ, ΑΒΓ']),
      '#maxlength' => '3',
    ];

    $elements['formatter_item'] = [
      '#type' => 'textfield',
      '#default_value' => (string) $this->getSetting('formatter_item'),
      '#title' => $this->t('Item formatter'),
      '#description' => $this->t("Enter a formatter string. Use '%on' where to inject the ordinal number.<br>Example: '%example'", ['%example' => '(%on)']),
      '#maxlength' => '255',
    ];

    $elements['formatter_short'] = [
      '#type' => 'textfield',
      '#default_value' => (string) $this->getSetting('formatter_short'),
      '#title' => $this->t('Short formatter'),
      '#description' => $this->t("Enter a formatter string. Use '%on' where to inject the ordinal number.<br>Example: '%example'", ['%example' => ' (%on)']),
      '#maxlength' => '255',
    ];

    $elements['formatter_medium'] = [
      '#type' => 'textfield',
      '#default_value' => (string) $this->getSetting('formatter_medium'),
      '#title' => $this->t('Upload directory'),
      '#description' => $this->t("Enter a formatter string. Use '%on' where to inject the ordinal number.<br>Example: '%example'", ['%example' => 'I (%on)']),
      '#maxlength' => '255',
    ];

    $elements['formatter_long'] = [
      '#type' => 'textfield',
      '#default_value' => (string) $this->getSetting('formatter_long'),
      '#title' => $this->t('Upload directory'),
      '#description' => $this->t("Enter a formatter string. Use '%on' where to inject the ordinal number.<br>Example: '%example'", ['%example' => 'Item (%on)']),
      '#maxlength' => '255',
    ];

    return $elements;
  }

  /**
   * Build select options for taxonomy term entity reference fields on the target entity + bundle.
   *
   * @return array
   *   Options keyed by field name, labeled with "Label (field_name)".
   */
  protected function buildTaxonomyReferenceOptions(): array {
    /** @var \Drupal\Core\Entity\EntityFieldManagerInterface $efm */
    $efm = \Drupal::service('entity_field.manager');
    $entity_type = $this->getFieldDefinition()->getTargetEntityTypeId();
    $bundle = $this->getFieldDefinition()->getTargetBundle();

    $options = [];
    if (!$entity_type || !$bundle) {
      return $options;
    }

    // Get all field definitions for this target bundle.
    $definitions = $efm->getFieldDefinitions($entity_type, $bundle);

    foreach ($definitions as $field_name => $definition) {
      $item_class = $definition->getItemDefinition()->getClass();
      // Look for entity reference fields targeting taxonomy_term.
      if (is_a($item_class, \Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItem::class, TRUE)) {
        $handler_settings = $definition->getSetting('handler_settings') ?? [];
        $target_type = $definition->getSetting('target_type');
        if ($target_type === 'taxonomy_term') {
          $label = $definition->getLabel();
          $options[$field_name] = (string) $this->t('@label (@name)', [
            '@label' => $label,
            '@name' => $field_name,
          ]);
        }
      }
    }

    // Sort options for a stable UX.
    asort($options, SORT_NATURAL | SORT_FLAG_CASE);
    return $options;
  }

  /**
   * {@inheritdoc}
   *
   * Keep settings human-readable on the "Field settings summary".
   * @param FieldDefinitionInterface $field_definition
   */
  public static function fieldSettingsSummary(FieldDefinitionInterface $field_definition): array {
    $summary = [];
    $min = (int) $field_definition->getSetting('min');
    $max = (int) $field_definition->getSetting('max');
    $summary[] = t('Range: @min to @max', ['@min' => $min, '@max' => $max]);

    $tax_ref = (string) $field_definition->getSetting('taxonomy_reference');
    $summary[] = $tax_ref
      ? t('Taxonomy reference: @field', ['@field' => $tax_ref])
      : t('Taxonomy reference: none');

    return $summary;
  }

}
