<?php

declare(strict_types=1);

namespace Drupal\taxonomy_ordinal\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\taxonomy\VocabularyInterface;

/**
 * Configuration form for Taxonomy Ordinal settings.
 */
final class TaxonomyOrdinalSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    // Unique form ID.
    return 'taxonomy_ordinal_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    // This config object stores our settings.
    return ['taxonomy_ordinal.settings'];
  }

  /**
   * Builds a key => label list of all taxonomy vocabularies.
   *
   * @return array
   *   An associative array of vocabulary IDs => labels.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getVocabularyOptions(): array {
    // Load all vocabularies via the entity type manager.
    /** @var \Drupal\taxonomy\VocabularyInterface[] $vocabularies */
    $vocabularies = \Drupal::entityTypeManager()
      ->getStorage('taxonomy_vocabulary')
      ->loadMultiple();

    $options = [];
    foreach ($vocabularies as $vocabulary) {
      // Ensure we have the expected interface.
      if ($vocabulary instanceof VocabularyInterface) {
        $options[$vocabulary->id()] = $vocabulary->label();
      }
    }

    // Sort options by label for predictable UX.
    asort($options, SORT_NATURAL | SORT_FLAG_CASE);
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('taxonomy_ordinal.settings');

    $form['description'] = [
      '#markup' => $this->t('Select the vocabularies for which "Taxonomy Ordinal" should be enabled.'),
    ];

    $form['enabled_vocabularies'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enable Taxonomy Ordinal'),
      // Options are all taxonomy vocabularies (vid => label).
      '#options' => $this->getVocabularyOptions(),
      // Default value is an array keyed by vid with vid as value.
      '#default_value' => $config->get('enabled_vocabularies') ?? [],
      '#description' => $this->t('Choose one or more vocabularies.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Convert the checkboxes array into a clean list of selected VIDs.
    // Checkboxes return either the key (selected) or 0 (unselected).
    $values = $form_state->getValue('enabled_vocabularies') ?? [];
    $selected = array_values(array_filter($values));
    // Persist the normalized values back into form state for submit.
    $form_state->setValue('enabled_vocabularies', $selected);
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Save sanitized selection as config.
    $selected = $form_state->getValue('enabled_vocabularies') ?? [];
    // Ensure a plain sequential array of strings (VIDs).
    $selected = array_values($selected);

    $this->configFactory->getEditable('taxonomy_ordinal.settings')
      ->set('enabled_vocabularies', $selected)
      ->save();

    parent::submitForm($form, $form_state);
    $this->messenger()->addStatus($this->t('Settings have been saved.'));
  }

}
