<?php

namespace Drupal\taxonomy_overview\Form;

use Drupal\node\Entity\Node;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\taxonomy\Entity\Term;

/**
 *
 */
class TagsOverviewTermMergeForm extends FormBase {

  protected $termIds;

  /**
   *
   */
  public function getFormId() {
    return 'term_variation_merge_form';
  }

  /**
   *
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $route_match = \Drupal::routeMatch();
    $vocabulary = $route_match->getParameter('taxonomy_vocabulary');

    $request = \Drupal::request();
    $query_params = $request->query->all();
    $term_ids = $request->query->get('term_ids');

    $this->termIds = array_filter(explode(',', $term_ids));
    $terms = Term::loadMultiple($this->termIds);

    if(count($terms) > 1) {

      $form['#attached']['library'][] = 'taxonomy_overview/taxonomy_overview.form';

      $form['description'] = [
        '#markup' => $this->t('<h3>You are about to merge the following terms.</h3>'),
      ];
      
      $form['description2'] = [
        '#markup' => $this->t('The unselected terms will be merged into the chosen one.</h3>'),
      ];

      $form['fieldset1'] = [
        '#type' => 'fieldset'
      ];
    
      $form['fieldset1']['target_tid'] = [
        '#type' => 'radios',
        '#title' => $this->t('Select the term to keep'),
        '#options' => array_map(fn($t) => $t->label(), $terms),
        '#required' => TRUE,
      ];

      $form['fieldset1']['remove_after'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Remove terms after merge?'),
        '#required' => FALSE,
      ];

      $form['undo'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('I agree that this operation cannot be undone!'),
        '#required' => TRUE,
      ];

      $form['actions'] = [
        '#type' => 'actions',
      ];

      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Merge Terms'),
        '#button_type' => 'primary',
        // '#disabled' => TRUE,
        '#states' => [
          'enabled' => [
            ':input[name="undo"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    return $form;
  }

  /**
   *
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $route_match = \Drupal::routeMatch();
    $vocabulary = $route_match->getParameter('taxonomy_vocabulary');

    $remove_terms = $form_state->getValue('remove_after');
    $target_tid = $form_state->getValue('target_tid');
    $tids_to_replace = $this->termIds;
    unset($tids_to_replace[array_search($target_tid, $tids_to_replace)]);

    $entitys_fields = $this->get_fields_entity_reference($vocabulary);
    $operations = [];

    foreach ($entitys_fields as $entity_bundle => $types) {
      if ($entity_bundle == 'node') {
        // Find all nodes referencing the terms to be replaced.
        foreach ($types as $conten_type => $fields) {
          foreach ($fields as $field_name) {
            $query = \Drupal::entityQuery('node')
              ->accessCheck(FALSE)
              ->condition($field_name, $tids_to_replace, 'IN');
            $nids = $query->execute();

            foreach ($nids as $nid) {
              $operations[] = [
                [self::class, 'processNodeMerge'],
                [
                  $nid,
                  $target_tid,
                  $tids_to_replace,
                  $field_name,
                ],
              ];
            }
          }
        }
      }
      elseif ($entity_bundle == 'paragraph' && class_exists('\Drupal\paragraphs\Entity\Paragraph')) {
        foreach ($types as $fields) {
          foreach ($fields as $field_name) {
            $paragraph_ids = \Drupal::entityQuery('paragraph')
              ->accessCheck(FALSE)
              ->condition('status', 1)
              ->latestRevision()
              ->condition($field_name . '.target_id', $tids_to_replace, "IN")
              ->execute();

            foreach ($paragraph_ids as $revision_id => $pid) {
              $operations[] = [
                [self::class, 'processParagraphMerge'],
                [
                  $pid,
                  $revision_id,
                  $target_tid,
                  $tids_to_replace,
                  $field_name,
                ],
              ];
            }
          }
        }
      }
    }

    // Final batch step: optionally delete old terms.
    if ($remove_terms) {
      foreach ($tids_to_replace as $tid) {
        $operations[] = [
          [self::class, 'processCleanTerms'],
          [
            $tid
          ],
        ];
      }
    }

    $batch = [
      'title' => $this->t('Merging taxonomy terms'),
      'operations' => $operations,
      'finished' => [self::class, 'batchFinished'],
    ];

    batch_set($batch);
  }

  /**
   *
   */
  public static function processNodeMerge($nid, $target_tid, $tids_to_replace, $field_name, &$context) {
    $node = Node::load($nid);
    if (!$node) {
      $context['results'][] = "Node " . $nid . " not found.";
      return;
    }

    $translations = $node->getTranslationLanguages();
    foreach ($translations as $language) {
      $langcode = $language->getId();
      if ($node->hasTranslation($langcode)) {
        $node = $node->getTranslation($langcode);

        $tags = $node->get($field_name)->getValue();
        $new_tags = [];

        foreach ($tags as $key => $tag) {
          $tid = $tag['target_id'];
          if (in_array($tid, $tids_to_replace)) {
            $tid = $target_tid;
          }
          $new_tags[$key] = ['target_id' => $tid];
        }

        $node->set($field_name, array_values($new_tags));
        $node->save();
      }
    }

    $context['results'][] = "Node " . $nid . " " . $field_name . " updated with merged terms.";
  }

  /**
   *
   */
  public static function processCleanTerms($tid, &$context) {
    $term = Term::load($tid);
    if ($term instanceof Term) {
      $label = $term->label();
      $term->delete();
      $context['results'][] = "Deleted term \"$label\" (tid: $tid)";
    }
  }

  /**
   *
   */
  public static function processParagraphMerge($paragraph_id, $revision_id, $target_tid, $tids_to_replace, $field_name, &$context) {

    $entity_type_manager = \Drupal::entityTypeManager();
    $paragraph_revision = $entity_type_manager->getStorage('paragraph')->loadRevision($revision_id);

    $translations = $paragraph_revision->getTranslationLanguages();
    foreach ($translations as $language) {
      $langcode = $language->getId();
      if ($paragraph_revision->hasTranslation($langcode)) {
        $paragraph_revision = $paragraph_revision->getTranslation($langcode);
        $values = $paragraph_revision->get($field_name)->getValue();

        $new_values = [];

        foreach ($values as $key => $value) {
          $tid = $value['target_id'];
          if (in_array($tid, $tids_to_replace)) {
            $tid = $target_tid;
          }
          $new_values[$key] = ['target_id' => $tid];
        }

        $paragraph_revision->set($field_name, $new_values);
        $paragraph_revision->save();
      }
    }
    $context['results'][] = "Paragraph " . $paragraph_id . " on revision " . $revision_id . " " . $field_name . " updated with merged terms.";
  }

  /**
   *
   */
  public static function batchFinished($success, $results, $operations) {
    $messenger = \Drupal::messenger();
    if ($success) {
      foreach ($results as $message) {
        $messenger->addStatus($message);
      }
    }
    else {
      $messenger->addError(t('An error occurred during the term merge process.'));
    }
  }

  /**
   * Return all fields that use this vocabulary as a field entity reference.
   */
  public function get_fields_entity_reference($vocabulary) {

    $arr_data = [];
    // Load all field storage configurations.
    $field_storage_configs = \Drupal::entityTypeManager()
      ->getStorage('field_storage_config')
      ->loadMultiple();

    foreach ($field_storage_configs as $field_storage_config) {
      if ($field_storage_config->getType() === 'entity_reference' || $field_storage_config->getType() === 'entity_reference_revisions') {
        $settings = $field_storage_config->getSettings();

        if (isset($settings['target_type']) && $settings['target_type'] === 'taxonomy_term') {
          $field_configs = \Drupal::entityTypeManager()
            ->getStorage('field_config')
            ->loadByProperties(['field_name' => $field_storage_config->getName()]);
          foreach ($field_configs as $field_config) {
            $field_settings = $field_config->getSetting('handler_settings');
            if (isset($field_settings['target_bundles'][$vocabulary])) {
              // Output or process the bundle and field name.
              $entity_type = $field_config->getTargetEntityTypeId();
              if (!isset($arr_data[$entity_type][$field_config->getTargetBundle()])) {
                $arr_data[$entity_type][$field_config->getTargetBundle()] = [];
              }
              $arr_data[$entity_type][$field_config->getTargetBundle()][] = $field_config->getName();
            }
          }
        }
      }
    }
    return $arr_data;
  }

}
