<?php

namespace Drupal\taxonomy_overview\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Drupal\node\Entity\Node;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;
use Symfony\Component\HttpFoundation\Request;

/**
 *
 */
class TagsOverviewController extends ControllerBase {

  private $vocabulary_id;
  private $taxonomy_term;

  /**
   *
   */
  public function content(Request $request) {
    $term_node_pages_content_types = [];
    $term_paragraph_by_content_type = [];
    $paragraph_node_used_bundle = [];
    $term_links_translatable = [];
    $term_duplicates_count = [];
    $paragraph_node_used = [];
    $term_translations = [];
    $term_node_counts = [];
    $paragraph_bundle = [];
    $paragraph_fields = [];
    $paragraph_count = [];
    $paragraph_ids = [];
    $node_fields = [];
    $term_names = [];
    $rows = [];

    $route_match = \Drupal::routeMatch();
    $this->vocabulary_id = $route_match->getParameter('taxonomy_vocabulary');
    $this->taxonomy_term = $route_match->getParameter('taxonomy_term');
    if ($this->taxonomy_term instanceof Term) {
      $this->vocabulary_id = $this->taxonomy_term->bundle();
    }

    $entity_fields = $this->check_entity_reference_vocabulary($this->vocabulary_id);
    if (isset($entity_fields['node'])) {
      foreach ($entity_fields['node'] as $key => $value) {
        $node_fields = array_merge(array_values($value), $node_fields);
      }
      $node_fields = array_unique($node_fields);

      foreach ($entity_fields['node'] as $key => $value) {
        $paragraph_fields = array_merge(array_values($value), $paragraph_fields);
      }
      $paragraph_fields = array_unique($paragraph_fields);
    }

    $paragraph_types = $entity_fields['paragraph'] ?? [];

    $search_term = $request->query->get('search', '');
    $sort_by = $request->query->get('sort_by', 'name');
    $sort_order = $request->query->get('sort_order', 'desc');
    $random_bg = $request->query->get('random_bg') == 1;

    $query = \Drupal::entityQuery('taxonomy_term')
      ->accessCheck(FALSE)
      ->condition('vid', $this->vocabulary_id);

    if (!empty($search_term)) {
      $query->condition('name', '%' . $search_term . '%', 'LIKE');
    }

    if ($this->taxonomy_term) {
      $query->condition('tid', $this->taxonomy_term->id());
    }

    if ($sort_by == 'name') {
      $query->sort('name', $sort_order);
    }

    $term_ids = $query->execute();
    $terms = Term::loadMultiple($term_ids);
    $total_count = count($terms);

    foreach ($terms as $term) {
      $term_name = $term->getName();
      $term_id = $term->id();
      $paragraph_bundle[$term_id] = [];
      $paragraph_node_used[$term_id] = [];
      $paragraph_node_used_bundle[$term_id] = [];
      $term_paragraph_by_content_type[$term_id] = [];

      if (!isset($term_duplicates_count[$term_name])) {
        $term_duplicates_count[$term_name] = 0;
      }

      $term_links_translatable[$term_id] = $this->getTranslationsLinks($term);
      $translations = $this->getTranslations($term);
      $term_translations[$term_id] = implode(', ', $translations);
      if ($paragraph_types) {

        foreach ($paragraph_types as $paragraph_type => $field_names) {
          foreach ($field_names as $field_name) {

            if (!isset($paragraph_count[$term_id])) {
              $paragraph_count[$term_id] = 0;
            }
            if (!isset($paragraph_ids[$term_id])) {
              $paragraph_ids[$term_id] = [];
            }

            $n_paragraphs = \Drupal::entityQuery('paragraph')
              ->accessCheck(FALSE)
              ->condition('status', 1)
              ->condition($field_name, $term_id)
              ->execute();

            $paragraph_count[$term_id] += count($n_paragraphs);
            $paragraph_ids[$term_id] = array_merge($paragraph_ids[$term_id], $n_paragraphs);
          }
        }

        if ($paragraph_ids[$term_id]) {
          foreach ($paragraph_ids[$term_id] as $pid) {
            $p = Paragraph::load($pid);
            $data_used = $this->getParentEntityNode($p);
            $paragraph_bundle[$term_id][] = $p->bundle();
            if ($data_used != NULL) {
              $paragraph_node_used[$term_id][] = $data_used['id'] ?? '';
              $paragraph_node_used_bundle[$term_id][] = $data_used['bundle'] ?? '';
            }
          }
        }
      }

      $node_count_arr = [];

      if ($node_fields) {
        $node_count = \Drupal::entityQuery('node')->accessCheck(FALSE);
        $or_group = $node_count->orConditionGroup();
        foreach ($node_fields as $f) {
          $or_group->condition($f, $term_id, 'in');
        }
        $node_count_arr = $node_count->condition($or_group)
          ->execute();
      }

      $term_node_counts[$term_id] = count(array_unique(array_filter(array_merge($paragraph_node_used[$term_id], $node_count_arr))));
      $term_node_pages_content_types[$term_id] = join(',', array_unique(array_merge($paragraph_node_used_bundle[$term_id], $this->getContentTypesByTaxonomyTerm($term_id))));
      $term_duplicates_count[$term_name]++;
      if (!isset($term_names[$term_name])) {
        $term_names[$term_name] = [];
      }
      $term_names[$term_name][] = $term->id();
    }

    // Sort terms based on the selected criteria.
    if ($sort_by == 'count') {
      uasort($terms, function ($a, $b) use ($term_node_counts, $sort_order) {
        $a_count = $term_node_counts[$a->id()];
        $b_count = $term_node_counts[$b->id()];
        return ($sort_order == 'asc') ? ($a_count <=> $b_count) : ($b_count <=> $a_count);
      });
    }
    elseif ($sort_by == 'node_count_content_type') {
      uasort($terms, function ($a, $b) use ($term_node_pages_content_types, $sort_order) {
        $a_count = $term_node_pages_content_types[$a->id()];
        $b_count = $term_node_pages_content_types[$b->id()];
        return ($sort_order == 'asc') ? ($a_count <=> $b_count) : ($b_count <=> $a_count);
      });
    }
    elseif ($sort_by == 'paragraph_count') {
      uasort($terms, function ($a, $b) use ($paragraph_count, $sort_order) {
        $a_count = $paragraph_count[$a->id()];
        $b_count = $paragraph_count[$b->id()];
        return ($sort_order == 'asc') ? ($a_count <=> $b_count) : ($b_count <=> $a_count);
      });
    }
    elseif ($sort_by == 'translations') {
      uasort($terms, function ($a, $b) use ($term_translations, $sort_order) {
        $a_translations = str_replace('Default: ', '', $term_translations[$a->id()]);
        $b_translations = str_replace('Default: ', '', $term_translations[$b->id()]);
        return ($sort_order == 'asc') ? strcasecmp($a_translations, $b_translations) : strcasecmp($b_translations, $a_translations);
      });
    }

    // Define table headers with sorting links.
    $header = [
      [
        'data' => $this->t('Term ID'),
        // 'field' => $this->taxonomy_term ? NULL : 'tid',
        'specifier' => 'tid',
      ],
      [
        'data' => $this->t('Name'),
        // 'field' => $this->taxonomy_term ? NULL : 'name',
        'specifier' => 'name',
      ],
      [
        'data' => $this->t('Nº Nodes'),
        // 'field' => $this->taxonomy_term ? NULL : 'count',
        'specifier' => 'count',
      ],
      [
        'data' => $this->t('Content Type'),
        // 'field' => $this->taxonomy_term ? NULL : 'node_count_content_type',
        'specifier' => 'node_count_content_type',
      ],
      [
        'data' => $this->t('Nº Paragraph'),
        // 'field' => $this->taxonomy_term ? NULL : 'paragraph_count',
        'specifier' => 'paragraph_count',
      ],
      [
        'data' => $this->t('Paragraph Bundle'),
        // 'field' => 'bundles',
        'specifier' => 'bundles',
      ],
      [
        'data' => $this->t('Translations'),
        // 'field' => $this->taxonomy_term ? NULL : 'translations',
        'specifier' => 'translations',
      ],
      [
        'data' => $this->t('Operations'),
      ],
    ];

    foreach ($header as &$column) {
      if (isset($column['specifier']) && isset($column['field'])) {
        $is_current = ($sort_by == $column['field']);
        $new_order = $is_current && $sort_order == 'asc' ? 'desc' : 'asc';
        $column['data'] = Link::fromTextAndUrl(
          $column['data'],
          Url::fromRoute('<current>', [], [
            'query' => [
              'search' => $search_term,
              'sort_by' => $column['field'],
              'sort_order' => $new_order,
            ],
          ])
        )->toString();
      }
    }

    // Define table rows.
    foreach ($terms as $term) {
      $term_name = $term->getName();
      $term_id = $term->id();

      // Create a link to the taxonomy term.
      $url = Url::fromRoute('entity.taxonomy_term.canonical', ['taxonomy_term' => $term->id()]);
      $link = Link::fromTextAndUrl($term_name, $url);

      $edit_links = $this->getLinks($term);

      // Row for the table.
      $rows[] = [
        'data' => [
          'tid' => $term->id(),
          'name' => Markup::create(join('<br>', $term_links_translatable[$term_id])) ,
          'count' => $term_node_counts[$term_id],
          'node_count_content_type' => $term_node_pages_content_types[$term_id],
          'paragraph_count' => $paragraph_count[$term_id] ?? '',
          'bundles' => join(',', $paragraph_bundle[$term_id]),
          'translations' => $term_translations[$term_id],
          'operations' => [
            'data' => [
              '#type' => 'operations',
              '#links' => $edit_links,
            ],
          ],
        ],
      ];
    }

    // Build the render array.
    $build = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'tags-overview-container',
      ],
      'search_form' => \Drupal::formBuilder()->getForm('Drupal\taxonomy_overview\Form\TagsOverviewForm'),
      'result_count' => [
        '#markup' => $this->t('<h6>Total Tags: @count</h6><h6>Fields: <small>@node_fields</small></h6>', ['@count' => $total_count, '@node_fields' => join(',', $node_fields)]),
      ],
      'term_table' => [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No terms found.'),
        '#attributes' => [
          'class' => ['table-auto-break'],
        ],
      ],
      'pager' => [
        '#type' => 'pager',
        '#element' => 0,
      ],
    ];

    $build['#attached']['library'][] = 'taxonomy_overview/taxonomy_overview.table_style';

    return $build;
  }

  /**
   *
   */
  private function getContentTypesByTaxonomyTerm($term_id) {
    $database = Database::getConnection();

    $query = $database->select('node_field_data', 'nfd')
      ->distinct()
      ->fields('nfd', ['type']);

    $query->innerJoin('taxonomy_index', 'ti', 'nfd.nid = ti.nid');
    $content_types = $query->condition('ti.tid', $term_id)->execute()->fetchCol();

    return $content_types;
  }

  /**
   *
   */
  private function getParentEntityNode($entity) {
    if ($entity instanceof Node) {
      return [
        'bundle' => $entity->bundle(),
        'id' => $entity->id(),
      ];
    }
    if ($entity instanceof Paragraph) {
      $parent = $entity->getParentEntity();
      if ($parent != NULL) {
        return $this->getParentEntityNode($entity->getParentEntity());
      }
    }
    return NULL;
  }

  /**
   * Function to check all fields if they reference the $vocabulary_id entity.
   */
  public function check_entity_reference_vocabulary($vocabulary_id) {
    $arr_data = [];
    $field_storage_configs = \Drupal::entityTypeManager()
      ->getStorage('field_storage_config')
      ->loadMultiple();

    foreach ($field_storage_configs as $field_storage_config) {
      if ($field_storage_config->getType() === 'entity_reference' || $field_storage_config->getType() === 'entity_reference_revisions') {
        $settings = $field_storage_config->getSettings();

        if (isset($settings['target_type']) && $settings['target_type'] === 'taxonomy_term') {
          $field_configs = \Drupal::entityTypeManager()
            ->getStorage('field_config')
            ->loadByProperties(['field_name' => $field_storage_config->getName()]);
          foreach ($field_configs as $field_config) {
            $field_settings = $field_config->getSetting('handler_settings');
            if (isset($field_settings['target_bundles'][$vocabulary_id])) {
              $entity_type = $field_config->getTargetEntityTypeId();
              $arr_data[$entity_type][$field_config->getTargetBundle()][] = $field_config->getName();
            }
          }
        }
      }
    }

    return $arr_data;
  }

  /**
   * Checks if translation is enabled for a taxonomy vocabulary.
   *
   * @return bool
   *   TRUE if translation is enabled for the vocabulary, FALSE otherwise.
   */
  public function isVocabularyTermsTranslationEnabled() {
    // Load the vocabulary entity.
    $vocabulary = Vocabulary::load($this->vocabulary_id);

    if ($vocabulary) {
      $config = \Drupal::config('language.content_settings.taxonomy_term.' . $this->vocabulary_id);
      if ($config) {
        $data = $config->getRawData();
        if (isset($data['third_party_settings']['content_translation']['enabled']) && $data['third_party_settings']['content_translation']['enabled'] === TRUE) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

  /**
   * Get translations for all active languages.
   */
  public function getLinks($term) {
    $language_manager = \Drupal::service('language_manager');
    $languages = $language_manager->getLanguages();

    $edit_links = [
      'edit' => [
        'title' => $this->t('Edit'),
        'url' => Url::fromRoute('entity.taxonomy_term.edit_form', ['taxonomy_term' => $term->id()]),
      ],
      'delete' => [
        'title' => $this->t('Delete'),
        'url' => Url::fromRoute('entity.taxonomy_term.delete_form', ['taxonomy_term' => $term->id()]),
      ],
    ];
    foreach ($languages as $language) {
      $langcode = $language->getId();

      if ($this->isVocabularyTermsTranslationEnabled()) {

        if ($term->hasTranslation($langcode)) {
          $edit_links['edit_' . $langcode] = [
            'title' => $this->t('Edit ' . strtoupper($langcode)),
            'url' => Url::fromRoute('entity.taxonomy_term.edit_form', [
              'taxonomy_term' => $term->id(),
              'langcode' => $langcode,
            ]),
          ];
          $edit_links['edit_' . $langcode] = [
            'title' => $this->t('Delete ' . strtoupper($langcode)),
            'url' => Url::fromRoute('entity.taxonomy_term.delete_form', [
              'taxonomy_term' => $term->id(),
            ], ['language' => \Drupal::languageManager()->getLanguage($langcode)]),
          ];
        }
        else {
          $edit_links['create_' . $langcode] = [
            'title' => $this->t('Create ' . strtoupper($langcode)),
            'url' => Url::fromRoute('entity.taxonomy_term.content_translation_add', [
              'taxonomy_term' => $term->id(),
              'source' => $term->langcode->value,
              'target' => $langcode,
            ], [
              'absolute' => FALSE,
            ]),
          ];
        }

      }
    }
    return $edit_links;
  }

  /**
   *
   */
  public function getTranslationsLinks($term) {
    $language_manager = \Drupal::service('language_manager');
    $languages = $language_manager->getLanguages();

    $term_links_translatable = [];

    foreach ($languages as $language) {
      $langcode = $language->getId();

      if ($term->hasTranslation($langcode)) {
        $term = $term->getTranslation($langcode);
        $term_links_translatable[] = $langcode . ': <a href="' . $term->toUrl()->toString() . '">' . $term->getName() . '</a>';
      }
    }
    return $term_links_translatable;
  }

  /**
   *
   */
  public function getTranslations($term) {
    $language_manager = \Drupal::service('language_manager');
    $languages = $language_manager->getLanguages();
    $translations = [];

    $translations[$term->langcode->value] = $term->langcode->value;

    foreach ($languages as $language) {
      $langcode = $language->getId();

      if (!isset($translations[$langcode]) && $term->hasTranslation($langcode)) {
        $translations[] = $langcode;
      }
    }
    return $translations;
  }

}
