<?php

namespace Drupal\taxonomy_overview\Controller;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Drupal\taxonomy_overview\TagsOverviewTermNormalizer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for grouping and displaying similar taxonomy terms.
 */
class TagsOverviewTermGroupController implements ContainerInjectionInterface {
  use StringTranslationTrait;
  /**
   * The current route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The taxonomy term storage handler.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $termStorage;

  /**
   * Constructs a new TagsOverviewTermGroupController object.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The translate service.
   */
  public function __construct(RouteMatchInterface $route_match, EntityTypeManagerInterface $entity_type_manager, TranslationInterface $string_translation) {
    $this->routeMatch = $route_match;
    $this->termStorage = $entity_type_manager->getStorage('taxonomy_term');
    $this->stringTranslation = $string_translation;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static(
      $container->get('current_route_match'),
      $container->get('entity_type.manager'),
      $container->get('string_translation')
    );
    // string_translation service used by StringTranslationTrait.
    $instance->setStringTranslation($container->get('string_translation'));
    return $instance;
  }

  /**
   * Builds a table grouping similar taxonomy terms by normalization.
   *
   * @return array
   *   A render array for a table of grouped terms with merge links.
   */
  public function groupTerms() {
    $terms = [];

    $vocabulary = $this->routeMatch->getParameter('taxonomy_vocabulary');

    $tids = $this->termStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('vid', $vocabulary)
      ->execute();

    $loaded_terms = $this->termStorage->loadMultiple($tids);
    foreach ($loaded_terms as $term) {
      $terms[$term->id()] = $term->label();
    }

    $normalizer = new TagsOverviewTermNormalizer();
    $groups = $normalizer->groupSimilarTerms($terms);

    $rows = [];
    foreach ($groups as $groupTerms) {
      $links = [];
      $tids = array_keys($groupTerms);
      foreach ($groupTerms as $tid => $label) {
        $url = $this->termStorage->load($tid)->toUrl()->toString();
        $links[] = "<a href='{$url}' target='_blank'>{$label}</a>";
      }

      $merge_url = Url::fromRoute('taxonomy_overview.group_similar.merge_form', [
        'term_ids' => implode(',', $tids),
        'taxonomy_vocabulary' => $vocabulary,
      ]);
      $merge_url->setOptions([
        'attributes' => [
          'class' => ['button', 'button--primary'],
        ],
      ]);

      $merge_link = Link::fromTextAndUrl(
        $this->t('Merge'),
        $merge_url
      )->toString();

      if (count($tids) > 1) {
        $rows[] = [
          'data' => [
            Markup::create(implode(', ', $links)),
            ['data' => $merge_link],
          ],
        ];
      }
    }

    if (!$rows) {
      return [
        '#markup' => $this->t('No similar taxonomy terms found.'),
      ];
    }

    return [
      '#type' => 'fieldset',
      '#title' => $this->t('Grouped Similar Taxonomy Terms'),
      'table' => [
        '#type' => 'table',
        '#header' => [
          $this->t('Grouped Similar Terms'),
          $this->t('Operations'),
        ],
        '#rows' => $rows,
        '#title' => $this->t('Grouped Similar Taxonomy Terms'),
      ],
    ];
  }

}
