<?php

namespace Drupal\taxonomy_overview\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for filtering taxonomy vocabularies.
 */
class VocabularyFilterForm extends FormBase {

  /**
   * The tempstore.private factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * Constructs a VocabularyFilterForm.
   *
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The tempstore.private factory.
   */
  public function __construct(PrivateTempStoreFactory $temp_store_factory) {
    $this->tempStoreFactory = $temp_store_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tempstore.private')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'taxonomy_overview_filter_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $tempstore = $this->tempStoreFactory->get('taxonomy_overview');
    $default = $tempstore->get('filter_name');

    $form['fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Filter taxonomies'),
      '#attributes' => [
        'class' => [
          'taxonomy-overview-filters',
          'container-inline',
          'mb-3',
        ],
      ],
    ];

    $form['fieldset']['filter_name'] = [
      '#type' => 'textfield',
      '#default_value' => $default,
      '#size' => 30,
      '#attributes' => [
        'placeholder' => $this->t('Enter part of taxonomy name'),
        'style' => 'margin-right: 10px;',
      ],
    ];

    $form['fieldset']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'style' => 'margin-right: 5px;',
      ],
    ];

    $form['fieldset']['reset'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset'),
      '#submit' => ['::resetForm'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $value = $form_state->getValue('filter_name');
    $this->tempStoreFactory->get('taxonomy_overview')->set('filter_name', $value);
  }

  /**
   * {@inheritdoc}
   */
  public function resetForm(array &$form, FormStateInterface $form_state) {
    $this->tempStoreFactory->get('taxonomy_overview')->delete('filter_name');
  }

}
