(function (Drupal, drupalSettings) {
  'use strict';

  // Ensure a global, mutable debug controller exists. This allows turning
  // logging on/off per area: 'init', 'render', 'interactions'.
  function ensureDebugConfig(settings) {
    if (window.icicles_debug) {
      return;
    }
    var defaults = { init: true, render: false, interactions: true };
    var fromSettings = (settings && settings.taxonomyTermConfigGroups && settings.taxonomyTermConfigGroups.debug) ? settings.taxonomyTermConfigGroups.debug : {};
    var flags = Object.assign({}, defaults, fromSettings || {});
    var api = {
      isEnabled: function(area) { return !!flags[area]; },
      set: function(area, enabled) { flags[area] = !!enabled; return !!flags[area]; },
      enableAll: function() { Object.keys(defaults).forEach(function(k){ flags[k] = true; }); return api.getAll(); },
      disableAll: function() { Object.keys(defaults).forEach(function(k){ flags[k] = false; }); return api.getAll(); },
      getAll: function() { return Object.assign({}, flags); }
    };
    Object.defineProperty(window, 'icicles_debug', {
      value: api,
      writable: false,
      configurable: false,
      enumerable: true
    });
  }

  // Local log helper that respects icicles_debug flags for info-level logs.
  function debugLog(area) {
    try {
      if (window.icicles_debug && window.icicles_debug.isEnabled(area) && typeof console !== 'undefined' && console.info) {
        var args = Array.prototype.slice.call(arguments, 1);
        console.info.apply(console, ['[icicles_manager][' + area + ']'].concat(args));
      }
    } catch (e) {}
  }

  function isObject(o) {
    return o !== null && typeof o === 'object';
  }

  function deepFreeze(obj) {
    if (!isObject(obj)) {
      return obj;
    }
    Object.getOwnPropertyNames(obj).forEach(function (prop) {
      var value = obj[prop];
      if (isObject(value)) {
        deepFreeze(value);
      }
    });
    return Object.freeze(obj);
  }

  function setGlobalOnce(name, value) {
    if (Object.prototype.hasOwnProperty.call(window, name)) {
      return;
    }
    Object.defineProperty(window, name, {
      value: value,
      writable: false,
      configurable: false,
      enumerable: true
    });
  }

  Drupal.behaviors.icicles_manager = {
    attach: function attach(context, settings) {
      // Ensure debug flags are initialized from drupalSettings (optional overrides).
      ensureDebugConfig(settings);
      // Only act once per page load. Behaviors can re-run; guard by checking
      // if the global has already been set.
      if (window.icicles_manager) {
        // Even if the immutable global already exists, we still want to ensure
        // the mutable transfer store is available once per page.
        ensureTransferStore(settings);
        ensureInteractionHandlers();
        return;
      }

      var data = (settings && settings.taxonomyTermConfigGroups) ? settings.taxonomyTermConfigGroups : null;
      if (!data) {
        // Still ensure the transfer store exists, even if no data payload.
        ensureTransferStore(settings);
        return;
      }

      var payload = {
        vocabulary: data.vocabulary || null,
        tree: (data && typeof data.tree !== 'undefined') ? data.tree : (data && data.terms ? data.terms : null)
      };

      // Compute a stable rank map (id => rank index) from the immutable tree.
      payload.rank = buildImmutableRankObject(payload.tree);

      // Compute a global, immutable color mapping (id => css color string).
      payload.colors = buildImmutableColorsMap(payload.tree);

      // Freeze deeply and expose as a global immutable variable.
      var frozen = deepFreeze(payload);
      setGlobalOnce('icicles_manager', frozen);

      // Also ensure the mutable transfer store is available.
      ensureTransferStore(settings);
      ensureInteractionHandlers();
    }
  };

  /**
   * Ensure a mutable transfer store is exposed globally as window.icicles_transfer.
   * This store is intended to temporarily hold term tree data being transferred
   * between icicles. It provides basic utilities to add/remove terms by id.
   *
   * The global reference is write-protected (non-writable) to avoid replacement,
   * but the object itself and its internal state are mutable.
   */
  function ensureTransferStore(settings) {
    if (window.icicles_transfer) {
      return;
    }

    // Local forest state inside a closure.
    var forest = [];

    // Utilities to manipulate term forests: { id:number, name:string, children:Array }.
    function cloneTreeNode(node) {
      return {
        id: Number(node && node.id != null ? node.id : 0),
        name: String(node && node.name != null ? node.name : ''),
        children: Array.isArray(node && node.children) ? node.children.map(cloneTreeNode) : []
      };
    }

    function toForest(treeOrForest) {
      if (!treeOrForest) return [];
      if (Array.isArray(treeOrForest)) return treeOrForest.map(cloneTreeNode);
      if (typeof treeOrForest === 'object') return [cloneTreeNode(treeOrForest)];
      return [];
    }

    function indexById(nodes) {
      var idx = new Map();
      (nodes || []).forEach(function (n) { idx.set(Number(n.id), n); });
      return idx;
    }

    function mergeForests(baseForest, addForest) {
      var baseIndex = indexById(baseForest);
      (addForest || []).forEach(function (addNode) {
        var id = Number(addNode.id);
        var existing = baseIndex.get(id);
        if (existing) {
          if (addNode.name) existing.name = String(addNode.name);
          if (!Array.isArray(existing.children)) existing.children = [];
          var addChildren = Array.isArray(addNode.children) ? addNode.children : [];
          mergeForests(existing.children, addChildren);
        }
        else {
          baseForest.push(cloneTreeNode(addNode));
        }
      });
      return baseForest;
    }

    function removeByIds(baseForest, removeForest) {
      var ids = new Set();
      (function collect(nodes) {
        (nodes || []).forEach(function (n) {
          ids.add(Number(n.id));
          if (Array.isArray(n.children)) collect(n.children);
        });
      })(removeForest || []);

      function filter(nodes) {
        return (nodes || [])
          .filter(function (n) { return !ids.has(Number(n.id)); })
          .map(function (n) {
            return {
              id: Number(n.id),
              name: String(n.name || ''),
              children: filter(Array.isArray(n.children) ? n.children : [])
            };
          });
      }

      return filter(baseForest || []);
    }

    // Tracks the id of the icicle that most recently added terms into the
    // transfer buffer. This is optional metadata and may be null.
    var sourceId = null;
    // Tracks the default icicle id (string) to receive orphaned terms.
    var defaultIcicleId = null;
    // Pending orphan queues keyed by a stable icicle key (e.g., group UUID).
    // Each value is a forest array to be transferred to default if the icicle
    // does not reappear after an AJAX refresh.
    var pendingOrphans = new Map();

    var api = {
      // Returns a deep copy of the current transfer forest.
      getTerms: function () { return toForest(forest); },
      // Returns a deep copy of the current transfer forest and clears the buffer.
      // Does not modify the remembered sourceId.
      takeTerms: function () { var out = toForest(forest); forest = []; return out; },
      // Deep-merge provided tree/forest into the transfer forest.
      // Optionally pass the id of the source icicle as the second argument.
      addTerms: function (tree, fromIcicleId) {
        var add = toForest(tree);
        mergeForests(forest, add);
        if (typeof fromIcicleId !== 'undefined') {
          sourceId = (fromIcicleId == null) ? null : String(fromIcicleId);
        }
        return this.getTerms();
      },
      // Remove any nodes (by id, including descendants) present in given tree/forest.
      removeTerms: function (tree) {
        var rem = toForest(tree);
        forest = removeByIds(forest, rem);
        return this.getTerms();
      },
      // Clear the transfer buffer (does not alter the remembered source id).
      clear: function () { forest = []; },
      // Queue orphan terms for a specific stable icicle key.
      queuePendingOrphans: function (key, tree) {
        try {
          if (!key) return false;
          var k = String(key);
          var existing = pendingOrphans.get(k) || [];
          var add = toForest(tree);
          var merged = mergeForests(existing.slice(), add);
          pendingOrphans.set(k, merged);
          if (typeof console !== 'undefined' && console.info) {
            try { debugLog('init', 'Queued pending orphans for key', k, { count: (function cn(a){var c=0;(a||[]).forEach(function(n){c++; if(Array.isArray(n.children)) c+=cn(n.children);}); return c; })(add) }); } catch (e) {}
          }
          return true;
        } catch (e) { return false; }
      },
      // Finalize any pending orphans: if an icicle with the same key exists on
      // the page after re-attach, discard its queued orphans (it was not removed);
      // otherwise, transfer its queued terms into the default icicle.
      finalizePendingOrphans: function () {
        try {
          if (!pendingOrphans || pendingOrphans.size === 0) return;
          var registry = (Drupal && Drupal.taxonomyTermIcicles) ? Drupal.taxonomyTermIcicles : {};
          // Build set of keys currently present in registry instances.
          var presentKeys = new Set();
          try {
            Object.keys(registry || {}).forEach(function(id){
              var inst = registry[id];
              var key = inst && inst.key ? String(inst.key) : '';
              if (key) presentKeys.add(key);
            });
          } catch (e) {}
          // Determine default destination instance.
          var defaultId = (typeof this.getDefaultIcicleId === 'function') ? this.getDefaultIcicleId() : null;
          var destInst = defaultId && registry && registry[defaultId] ? registry[defaultId] : null;
          // Iterate pending keys and either discard or transfer.
          Array.from(pendingOrphans.keys()).forEach(function(k){
            var forestForKey = pendingOrphans.get(k) || [];
            if (presentKeys.has(k)) {
              // Icicle still exists; discard queued orphans.
              if (typeof console !== 'undefined' && console.info) {
                try { debugLog('init', 'Discarding pending orphans for key (icicle still present):', k); } catch (e) {}
              }
              pendingOrphans.delete(k);
              return;
            }
            // No icicle present for this key: transfer to default if possible.
            if (destInst && typeof destInst.addTerms === 'function' && Array.isArray(forestForKey) && forestForKey.length) {
              try {
                if (typeof console !== 'undefined' && console.info) {
                  try { debugLog('init', 'Finalizing pending orphans for key', k, '→ default icicle id', String(defaultId)); } catch (e) {}
                }
                destInst.addTerms(forestForKey);
              } catch (e) {}
            } else if (typeof console !== 'undefined' && console.warn) {
              console.warn('[icicles_manager] No valid default icicle to receive pending orphans for key:', k);
            }
            pendingOrphans.delete(k);
          });
        } catch (e) {}
      },
      // Getter/Setter for the last source icicle id.
      getSourceId: function () { return sourceId; },
      setSourceId: function (id) { sourceId = (id == null) ? null : String(id); return sourceId; },
      // Getter/Setter for the default icicle id.
      getDefaultIcicleId: function () { return defaultIcicleId; },
      setDefaultIcicleId: function (id) { defaultIcicleId = (id == null) ? null : String(id); return defaultIcicleId; }
    };

    setGlobalOnce('icicles_transfer', api);
  }

  /**
   * Bind once-only interaction handlers to coordinate icicle term transfers.
   * - On termsIcicle:nodeHoldStart: remove subtree from source icicle and add to transfer buffer with source id.
   * - On mouseup: return any buffered terms back to the source icicle, then clear buffer and reset source id.
   */
  function ensureInteractionHandlers() {
    if (window.__iciclesInteractionHandlersBound) {
      return;
    }
    window.__iciclesInteractionHandlersBound = true;

    // Utility to count total nodes in a forest (includes all descendants).
    function countNodes(nodes) {
      var c = 0;
      (function walk(arr){
        (arr || []).forEach(function(n){
          c++;
          if (Array.isArray(n && n.children)) walk(n.children);
        });
      })(nodes || []);
      return c;
    }

    // Simple floating "ghost" element that follows the mouse while transferring.
    var ghostEl = null;
    var mouseMoveBound = false;
    function ensureGhost() {
      if (ghostEl) return;
      ghostEl = document.createElement('div');
      ghostEl.className = 'terms-icicle__ghost';
      ghostEl.setAttribute('aria-hidden', 'true');
      ghostEl.style.position = 'fixed';
      ghostEl.style.pointerEvents = 'none';
      ghostEl.style.zIndex = '2147483647';
      ghostEl.style.left = '0px';
      ghostEl.style.top = '0px';
      document.body.appendChild(ghostEl);
    }
    function updateGhostContent(forest) {
      if (!ghostEl) return;
      var count = countNodes(forest || []);
      var names = [];
      (forest || []).forEach(function(n){ if (n && (n.name || n.id)) names.push(String(n.name || n.id)); });
      var title = names.length ? names.join(', ') : '';
      ghostEl.innerHTML = '<strong>' + count + '</strong> term' + (count === 1 ? '' : 's') + (title ? ' — ' + title : '');
    }
    function onMouseMove(ev) {
      if (!ghostEl) return;
      var x = (ev && typeof ev.clientX === 'number') ? ev.clientX : 0;
      var y = (ev && typeof ev.clientY === 'number') ? ev.clientY : 0;
      // Offset so the cursor is not obscured.
      ghostEl.style.left = (x + 12) + 'px';
      ghostEl.style.top = (y + 20) + 'px';
    }
    function bindMouseMove() {
      if (mouseMoveBound) return;
      document.addEventListener('mousemove', onMouseMove, true);
      mouseMoveBound = true;
    }
    function removeGhost() {
      if (mouseMoveBound) {
        document.removeEventListener('mousemove', onMouseMove, true);
        mouseMoveBound = false;
      }
      if (ghostEl && ghostEl.parentNode) {
        ghostEl.parentNode.removeChild(ghostEl);
      }
      ghostEl = null;
    }

    // When a node hold starts, move that subtree into the transfer buffer and remove from the source icicle.
    document.addEventListener('termsIcicle:nodeHoldStart', function (e) {
      try {
        var detail = e && e.detail ? e.detail : null;
        if (!detail) return;
        var icicleId = detail.icicleId;
        var tree = detail.tree;
        if (!icicleId || !tree || !Array.isArray(tree)) return;
        var registry = (Drupal && Drupal.taxonomyTermIcicles) ? Drupal.taxonomyTermIcicles : {};
        var inst = registry[icicleId];
        if (!inst || typeof inst.removeTerms !== 'function') return;

        var num = countNodes(tree);
        if (typeof console !== 'undefined' && console.info) {
          try {
            debugLog('interactions', 'transfer start (mousedown): moving', num, 'term(s) from icicle', String(icicleId), 'to icicles_transfer');
          } catch (logErr) {}
        }

        // Remove from the source icicle first.
        inst.removeTerms(tree);
        // Add to the transfer buffer with source id.
        if (window.icicles_transfer && typeof window.icicles_transfer.addTerms === 'function') {
          window.icicles_transfer.addTerms(tree, icicleId);
        }
        // Show and update ghost to follow the mouse.
        ensureGhost();
        updateGhostContent((window.icicles_transfer && typeof window.icicles_transfer.getTerms === 'function') ? window.icicles_transfer.getTerms() : tree);
        bindMouseMove();
      } catch (err) {
        if (typeof console !== 'undefined' && console.error) {
          console.error('[icicles_manager] Error handling nodeHoldStart:', err);
        }
      }
    }, false);

    // On mouseup anywhere, determine drop target and transfer buffered terms accordingly.
    document.addEventListener('mouseup', function (event) {
      try {
        var transfer = window.icicles_transfer;
        if (!transfer) return;
        var srcId = (typeof transfer.getSourceId === 'function') ? transfer.getSourceId() : null;
        var terms = (typeof transfer.takeTerms === 'function') ? transfer.takeTerms() : [];
        var num = countNodes(terms);

        if (!Array.isArray(terms) || !terms.length) {
          if (typeof console !== 'undefined' && console.info) {
            try {
              debugLog('interactions', 'transfer end (mouseup): no terms to transfer', { sourceId: srcId, count: num });
            } catch (logErr) {}
          }
          // Still clear and reset below.
        } else {
          // Detect the icicle under the mouse pointer (also matches the empty placeholder inside it).
          var clientX = event && typeof event.clientX === 'number' ? event.clientX : null;
          var clientY = event && typeof event.clientY === 'number' ? event.clientY : null;
          var hoveredIcicleId = null;
          if (clientX !== null && clientY !== null && document.elementFromPoint) {
            var el = document.elementFromPoint(clientX, clientY);
            if (el && typeof el.closest === 'function') {
              var wrapper = el.closest('.terms-icicle');
              if (wrapper && wrapper.id) {
                hoveredIcicleId = String(wrapper.id);
              }
            }
          }

          var registry = (Drupal && Drupal.taxonomyTermIcicles) ? Drupal.taxonomyTermIcicles : {};
          var destId = hoveredIcicleId || srcId; // Fallback to source if not hovering any icicle.

          if (destId) {
            if (typeof console !== 'undefined' && console.info) {
              try {
                debugLog('interactions', 'transfer end (mouseup): moving', num, 'term(s) from icicles_transfer to icicle', String(destId), hoveredIcicleId ? '(hover target)' : '(source fallback)');
              } catch (logErr) {}
            }
            var inst = registry[destId];
            if (inst && typeof inst.addTerms === 'function') {
              inst.addTerms(terms);
            } else if (typeof console !== 'undefined' && console.warn) {
              console.warn('[icicles_manager] Drop target icicle instance not found for id:', destId);
            }
          } else if (typeof console !== 'undefined' && console.info) {
            try {
              debugLog('interactions', 'transfer end (mouseup): could not resolve destination icicle; discarding transfer terms', { count: num });
            } catch (logErr) {}
          }
        }

        // Ensure buffer is empty and source id cleared regardless.
        if (typeof transfer.clear === 'function') transfer.clear();
        if (typeof transfer.setSourceId === 'function') transfer.setSourceId(null);
        // Always remove the global dragging class on mouseup (safety net).
        if (document && document.body && document.body.classList) {
          document.body.classList.remove('terms-icicle--dragging');
        }
        // Remove the ghost now that transfer is complete/cancelled.
        removeGhost();
      } catch (err) {
        if (typeof console !== 'undefined' && console.error) {
          console.error('[icicles_manager] Error handling mouseup:', err);
        }
        // Safety: ensure ghost removed on errors as well.
        removeGhost();
      }
    }, false);
  }


  // Build a rank map (plain object id => rank) using preorder traversal to preserve
  // the original sibling order from the immutable reference tree.
  function buildImmutableRankObject(referenceForest) {
    var rank = Object.create(null);
    var i = 0;
    (function walk(nodes){
      (nodes || []).forEach(function(n){
        var idNum = Number(n && n.id);
        if (!Object.prototype.hasOwnProperty.call(rank, idNum)) {
          rank[idNum] = i++;
        }
        if (Array.isArray(n && n.children)) walk(n.children);
      });
    })(referenceForest || []);
    return rank;
  }

  // Build an immutable color map (id => color string) based on the immutable tree.
  // Coloring strategy:
  // - Default: assign distinct colors to top-level nodes (depth 1 in the forest) and
  //   propagate that color to their descendants.
  // - If the forest has exactly one top-level node, assign neutral color to that top
  //   node and assign distinct colors to its children (depth 2), propagating to their
  //   descendants. This mirrors the special-case used in the renderer previously.
  function buildImmutableColorsMap(referenceForest) {
    // Normalize to forest (array of nodes)
    var forest = Array.isArray(referenceForest) ? referenceForest : (referenceForest ? [referenceForest] : []);

    function countTop(nodes) { return (Array.isArray(nodes) ? nodes.length : 0) || 0; }
    var hasSingleTop = countTop(forest) === 1;

    // Determine categories set at the coloring level.
    var categories = [];
    if (!hasSingleTop) {
      categories = forest.slice();
    } else {
      var onlyTop = forest[0] || null;
      categories = (onlyTop && Array.isArray(onlyTop.children)) ? onlyTop.children.slice() : [];
    }

    // Build a palette of distinct, visually spaced HSL colors.
    var n = Math.max(1, categories.length);
    function hsl(i, total) {
      var hue = (i * (360 / Math.max(1, total))) % 360;
      var sat = 65; // percent
      var light = 55; // percent
      return 'hsl(' + hue + ', ' + sat + '%, ' + light + '%)';
    }

    var categoryColorById = Object.create(null);
    for (var c = 0; c < categories.length; c++) {
      var cat = categories[c];
      if (!cat) continue;
      var id = Number(cat.id);
      categoryColorById[id] = hsl(c, n);
    }

    var NEUTRAL = '#ccc';
    var colors = Object.create(null);

    // Walk the forest, assigning colors based on the strategy above.
    function walk(nodes, depth, inheritedColor) {
      (nodes || []).forEach(function (node) {
        var idNum = Number(node && node.id);
        var ownColor = inheritedColor;
        if (!hasSingleTop) {
          // Depth 1 nodes get their category color; descendants inherit.
          if (depth === 1) {
            ownColor = categoryColorById[idNum] || NEUTRAL;
          }
        } else {
          // Single top: depth 1 = neutral; depth 2 = category colors; deeper inherit.
          if (depth === 1) {
            ownColor = NEUTRAL;
          } else if (depth === 2) {
            ownColor = categoryColorById[idNum] || NEUTRAL;
          }
        }
        colors[idNum] = ownColor || NEUTRAL;
        if (Array.isArray(node && node.children)) {
          walk(node.children, depth + 1, colors[idNum]);
        }
      });
    }

    walk(forest, 1, NEUTRAL);

    return colors;
  }

})(Drupal, drupalSettings);
