(function (Drupal, once, drupalSettings) {
  'use strict';

  // Info-level debug helper that respects the global icicles_debug flags.
  function tiDebug(area) {
    try {
      if (window.icicles_debug && typeof window.icicles_debug.isEnabled === 'function' && window.icicles_debug.isEnabled(area) && typeof console !== 'undefined' && console.info) {
        var args = Array.prototype.slice.call(arguments, 1);
        console.info.apply(console, ['[terms_icicle][' + area + ']'].concat(args));
      }
    } catch (e) {}
  }

  /**
   * Terms Icicle renderer and state manager.
   *
   * This script renders a zoomable D3 icicle representing taxonomy terms and
   * exposes a small per-element API to mutate the underlying tree (add/remove
   * terms) while keeping rendering concerns separate. The D3 rendering reads
   * from the mutable in-memory tree and is re-triggered after changes.
   *
   * Key design notes:
   * - The data model is a forest of nodes: { id: number, name: string, children: Node[] }.
   * - addTerms() and removeTerms() update a per-instance selected id set, then rebuild
   *   the forest strictly from the global immutable vocabulary tree, pruning out
   *   all non-selected terms while preserving relative hierarchy of selected nodes.
   * - Each .terms-icicle element on the page gets its own instance/state.
   * - D3 v7 is used to compute a partition layout suitable for icicles.
   */

  // Registry for optional external access (debugging/integration).
  Drupal.taxonomyTermIcicles = Drupal.taxonomyTermIcicles || {};

  // Internal map from element => instance to keep per-instance state.
  const instances = new WeakMap();


  // Simple unique id generator for elements.
  let icicleCounter = 0;
  /**
   * Ensure the element has an id attribute and return it.
   * @param {HTMLElement} el - The wrapper element for an icicle instance.
   * @returns {string} - The ensured or newly generated id.
   */
  function ensureElementId(el) {
    if (!el.id) {
      icicleCounter += 1;
      el.id = `terms-icicle-${icicleCounter}`;
    }
    return el.id;
  }

  // Utilities to work with trees of terms: { id, name, children: [] }.
  /**
   * Deep-clone a term tree node into a normalized structure.
   * @param {{id:number|string,name?:string,children?:Array}} node
   * @returns {{id:number,name:string,children:Array}}
   */
  function cloneTreeNode(node) {
    return {
      id: Number(node.id),
      name: String((node.name != null) ? node.name : ''),
      children: Array.isArray(node.children) ? node.children.map(cloneTreeNode) : [],
    };
  }

  /**
   * Normalize input into a forest (array of nodes) and deep-clone the result.
   * Accepts a single node object or an array of nodes; returns a new array.
   * @param {Object|Array|null|undefined} treeOrForest
   * @returns {Array<{id:number,name:string,children:Array}>}
   */
  function toForest(treeOrForest) {
    if (!treeOrForest) return [];
    // If it looks like a single node, wrap it in an array; otherwise assume array.
    if (Array.isArray(treeOrForest)) return treeOrForest.map(cloneTreeNode);
    if (typeof treeOrForest === 'object') return [cloneTreeNode(treeOrForest)];
    return [];
  }

  /**
   * Collect all ids from a forest (including descendants).
   * @param {Array} forest
   * @returns {Set<number>}
   */
  function collectIdsFromForest(forest) {
    const ids = new Set();
    (function walk(nodes){
      (nodes || []).forEach(n => {
        ids.add(Number(n.id));
        if (Array.isArray(n.children)) walk(n.children);
      });
    })(forest || []);
    return ids;
  }

  /**
   * Build a forest from the immutable reference, keeping only nodes whose ids
   * are in selectedIds. If a non-selected node has selected descendants, those
   * descendants are promoted (the unselected node is removed), preserving
   * relative hierarchy among selected nodes.
   * @param {Array} referenceForest - immutable forest
   * @param {Set<number>} selectedIds
   * @returns {Array}
   */
  function buildForestFromSelection(referenceForest, selectedIds) {
    function pruneNode(node) {
      const keptChildren = (node.children || []).flatMap(pruneNode);
      const isSelected = selectedIds.has(Number(node.id));
      if (isSelected) {
        return [{ id: Number(node.id), name: String(node.name || ''), children: keptChildren }];
      }
      // Not selected: promote children if any.
      return keptChildren;
    }
    return (referenceForest || []).flatMap(pruneNode);
  }


  /**
   * Create and register a new icicle instance for a given wrapper element.
   * - Reads per-element settings from drupalSettings using the element id.
   * - Initializes a mutable 'terms' forest (empty or full, per settings).
   * - Returns an instance API to interact with this element.
   * @param {HTMLElement} el - The .terms-icicle wrapper element.
   * @returns {{
   *   Terms: Array,
   *   getTerms: function():Array,
   *   render: function():void,
   *   addTerms: function(Object|Array):Array,
   *   removeTerms: function(Object|Array):Array
   * }}
   */
  function createInstance(el) {
    // Ensure the element has an ID to look up per-element settings.
    const id = ensureElementId(el);

    // Initialize from drupalSettings and immutable manager tree if present.
    const globalSettings = (drupalSettings && drupalSettings.taxonomyTermConfigGroups) || {};
    const elementSettings = (globalSettings.elements && globalSettings.elements[id]) || {};
    const initialTreeArray = Array.isArray(globalSettings.tree) ? globalSettings.tree : [];
    const immutableTree = (window.icicles_manager && Array.isArray(window.icicles_manager.tree)) ? window.icicles_manager.tree : initialTreeArray;
    const initFull = !!elementSettings.initFull;
    const isDefault = !!elementSettings.isDefault;
    const icicleKey = (elementSettings && elementSettings.key) ? String(elementSettings.key) : (el.getAttribute('data-icicle-key') || '');

    // Debugging: Log initialization details.
    if (typeof console !== 'undefined' && console.info) {
      tiDebug('init', 'Initialized icicle id:', id);
      tiDebug('init', 'Start mode:', initFull ? 'full (all terms)' : 'empty');
    }

    // Rank helper: require global immutable rank map from icicles_manager; error if missing.
    function rankOf(id) {
      const idNum = Number(id);
      const manager = (window && window.icicles_manager) ? window.icicles_manager : null;
      const globalRank = manager && manager.rank;
      if (!globalRank || typeof globalRank !== 'object') {
        throw new Error('[terms_icicle] Missing global rank map from icicles_manager. This should be initialized before terms_icicle.');
      }
      return Object.prototype.hasOwnProperty.call(globalRank, idNum) ? globalRank[idNum] : Number.MAX_SAFE_INTEGER;
    }

    // Selection-driven model: track selected ids per icicle, rebuild from immutable reference.
    const input = el.querySelector('.terms-icicle__input');
    let selectedIds = new Set();
    // If a hidden input carries a JSON forest, initialize from it; otherwise respect initFull.
    (function initSelectionFromInput(){
      let initialForest = null;
      if (input && typeof input.value === 'string' && input.value.trim().length) {
        try {
          const parsed = JSON.parse(input.value);
          initialForest = toForest(parsed);
        } catch (e) {
          // Ignore parse errors; fall back to initFull behavior.
        }
      }
      if (initialForest && initialForest.length) {
        selectedIds = collectIdsFromForest(initialForest);
      }
      else if (initFull) {
        selectedIds = collectIdsFromForest(immutableTree);
      }
      else {
        selectedIds = new Set();
      }
    })();

    // Backing terms array for rendering; we always rebuild from immutableTree using selectedIds.
    const terms = [];
    function rebuildFromSelection() {
      const rebuilt = buildForestFromSelection(immutableTree, selectedIds);
      terms.length = 0;
      rebuilt.forEach(n => terms.push(n));
      // Keep the hidden input synchronized so Drupal form submissions/AJAX preserve state.
      if (input) {
        try { input.value = JSON.stringify(terms); } catch (e) {}
        try { input.dispatchEvent(new Event('input', { bubbles: true })); } catch (e) {}
        try { input.dispatchEvent(new Event('change', { bubbles: true })); } catch (e) {}
      }
    }
    // Initial build
    rebuildFromSelection();

    // Rendering helpers
    /**
     * Resolve the DOM node where the icicle should render.
     * @returns {HTMLElement} The inner .terms-icicle__state container if present, otherwise the wrapper element.
     */
    function getTarget() {
      return el.querySelector('.terms-icicle__state') || el;
    }
    /**
     * Remove all child nodes from a target element.
     * @param {HTMLElement} elm
     */
    function clear(elm) {
      while (elm.firstChild) elm.removeChild(elm.firstChild);
    }
    /**
     * Render an accessible empty-state box when no terms are present.
     * @param {HTMLElement} target - Container where the message should render.
     */
    function renderEmpty(target) {
      const box = document.createElement('div');
      box.className = 'terms-icicle__empty-box';
      const p = document.createElement('p');
      p.className = 'terms-icicle__empty-text';
      const msg = 'No terms selected. Start by adding terms to the grouping.';
      p.textContent = Drupal.t ? Drupal.t(msg) : msg;
      box.appendChild(p);
      target.appendChild(box);
    }
    /**
     * Count nodes in a forest by walking all descendants.
     * Pure utility, does not mutate input.
     * @param {Array} nodes
     * @returns {number}
     */
    function countNodes(nodes) {
      let c = 0;
      (function walk(arr){
        (arr || []).forEach(n => {
          c++;
          if (Array.isArray(n.children)) walk(n.children);
        });
      })(nodes || []);
      return c;
    }
    /**
     * Render the current terms forest as a D3 zoomable icicle.
     * - Computes container-driven width/height for responsiveness.
     * - Builds a single-root hierarchy and D3 partition layout.
     * - Appends SVG groups, rects, and text, wiring up click-to-zoom.
     * - Fallbacks to a JSON <pre> if D3 is not available.
     */
    function render() {
      const target = getTarget();
      clear(target);
      if (typeof console !== 'undefined' && console.info) {
        try {
          tiDebug('render', 'render: start', { totalNodes: countNodes(terms) });
        } catch (e) {}
      }

      const forest = terms;
      if (!forest || forest.length === 0) {
        renderEmpty(target);
        return;
      }

      // Ensure D3 is available.
      if (typeof window.d3 === 'undefined') {
        // Fallback: simple text output if D3 failed to load.
        const fallback = document.createElement('pre');
        fallback.textContent = JSON.stringify(forest, null, 2);
        target.appendChild(fallback);
        return;
      }

      const d3 = window.d3;

      // Compute dimensions from the container, with sensible fallbacks.
      const rect = target.getBoundingClientRect();
      const width = Math.max(320, Math.floor(rect.width || 928));
      const height = Math.max(400, Math.min(1200, Math.floor(rect.height || 600)));
      if (typeof console !== 'undefined' && console.info) {
        try {
          tiDebug('render', 'render: container', { width, height });
        } catch (e) {}
      }

      // Prepare the root data for a single-root partition layout.
      const rootData = { name: 'root', children: forest };
      const hierarchy = d3.hierarchy(rootData)
        .sum(d => (d && typeof d.value === 'number') ? d.value : 1)
        .sort((a, b) => {
          const aid = (a && a.data && a.depth > 0) ? Number(a.data.id) : -1;
          const bid = (b && b.data && b.depth > 0) ? Number(b.data.id) : -1;
          const ar = rankOf(aid);
          const br = rankOf(bid);
          return ar - br;
        });

      const root = d3.partition()
        .size([height, (hierarchy.height + 1) * width / 3])
        (hierarchy);

      // Colors: use immutable global colors map assigned on initial load via icicles_manager.
      const manager = (window && window.icicles_manager) ? window.icicles_manager : null;
      const globalColors = manager && manager.colors ? manager.colors : null;
      function colorForNode(d) {
        const id = d && d.data && d.depth > 0 ? Number(d.data.id) : null;
        if (id != null && globalColors && Object.prototype.hasOwnProperty.call(globalColors, id)) {
          return globalColors[id];
        }
        return '#ccc';
      }

      // SVG container.
      const svg = d3.select(target)
        .append('svg')
        .attr('viewBox', [0, 0, width, height])
        .attr('width', width)
        .attr('height', height)
        .attr('style', 'max-width: 100%; height: auto; font: 10px sans-serif;');

      const cell = svg
        .selectAll('g')
        .data(root.descendants().filter(d => d.depth > 0))
        .join('g')
        .attr('transform', d => `translate(${d.y0},${d.x0})`);

      // Press-and-hold transfer detection state (per render/instance).
      let holdTimer = null;
      let isHolding = false;
      let suppressNextClick = false;
      const holdDelayMs = 200;

      const rects = cell.append('rect')
        .attr('width', d => Math.max(0, d.y1 - d.y0 - 1))
        .attr('height', d => rectHeight(d))
        .attr('fill-opacity', 0.6)
        .attr('fill', d => colorForNode(d))
        .style('cursor', 'pointer')
        .on('click', clicked)
        .on('mousedown', nodeMouseDown);

      function nodeMouseDown(event, p) {
        try {
          // Prevent native text selection and other default behaviors during drag.
          if (event && typeof event.preventDefault === 'function') {
            event.preventDefault();
          }
          // Add a global flag class to disable user-select across the page while dragging.
          if (document && document.body && document.body.classList) {
            document.body.classList.add('terms-icicle--dragging');
          }
          // Start a press-and-hold timer. Only when the user holds beyond the
          // threshold do we initiate a transfer. Quick clicks will zoom.
          isHolding = false;
          if (holdTimer) {
            clearTimeout(holdTimer);
            holdTimer = null;
          }
          const subtree = cloneTreeNode(p && p.data ? p.data : {});
          const payload = [subtree];

          holdTimer = setTimeout(() => {
            try {
              isHolding = true;
              suppressNextClick = true; // Prevent zoom when a hold starts.
              const custom = new CustomEvent('termsIcicle:nodeHoldStart', {
                bubbles: true,
                detail: { icicleId: id, tree: payload }
              });
              el.dispatchEvent(custom);
            } catch (e2) {
              if (typeof console !== 'undefined' && console.error) {
                console.error('[terms_icicle] nodeHoldStart dispatch error:', e2);
              }
            }
          }, holdDelayMs);

          // Finish/cancel logic on mouseup: clear timer; if hold happened, we
          // just suppress click; if not, allow click to proceed (zoom).
          const onUp = (ev) => {
            try {
              if (holdTimer) {
                clearTimeout(holdTimer);
                holdTimer = null;
              }
              // Always remove the global dragging class on mouseup.
              if (document && document.body && document.body.classList) {
                document.body.classList.remove('terms-icicle--dragging');
              }
            } finally {
              window.removeEventListener('mouseup', onUp, true);
            }
          };
          // Use capture so we clear timer before click fires.
          window.addEventListener('mouseup', onUp, true);
        } catch (e) {
          if (typeof console !== 'undefined' && console.error) {
            console.error('[terms_icicle] nodeMouseDown error:', e);
          }
        }
      }

      const texts = cell.append('text')
        .style('user-select', 'none')
        .attr('pointer-events', 'none')
        .attr('x', 4)
        .attr('y', 13)
        .attr('fill-opacity', d => +labelVisible(d));

      texts.append('tspan').text(d => d.data && d.data.name ? d.data.name : String(d.data && d.data.id ? d.data.id : ''));

      const format = d3.format(',d');
      const tspans = texts.append('tspan')
        .attr('fill-opacity', d => labelVisible(d) * 0.7)
        .text(d => ` ${format(d.value)}`);

      cell.append('title')
        .text(d => `${d.ancestors().filter(a => a.depth > 0).map(a => (a.data && a.data.name) ? a.data.name : String(a.data && a.data.id ? a.data.id : '')).reverse().join('/')}` + (d.value ? `\n${format(d.value)}` : ''));

      // Zoom behavior on click, adapted from example.
      let focus = root;

      // Auto-focus: if there is exactly one top-level term, make it the initial focus.
      if (root.children && root.children.length === 1) {
        focus = root.children[0];
        root.each(d => d.target = {
          x0: (d.x0 - focus.x0) / (focus.x1 - focus.x0) * height,
          x1: (d.x1 - focus.x0) / (focus.x1 - focus.x0) * height,
          y0: d.y0 - focus.y0,
          y1: d.y1 - focus.y0,
        });
        // Apply the focused coordinates immediately without animation.
        cell.attr('transform', d => `translate(${d.target.y0},${d.target.x0})`);
        rects.attr('height', d => rectHeight(d.target));
        texts.attr('fill-opacity', d => +labelVisible(d.target));
        tspans.attr('fill-opacity', d => labelVisible(d.target) * 0.7);
      }
      /**
       * Handle click interactions to zoom in/out of the icicle.
       * - If the clicked node is already focused, zoom out to its parent.
       * - Computes normalized target coordinates for a smooth transition.
       * @param {MouseEvent} event
       * @param {any} p - d3 partition node for the clicked segment.
       */
      function clicked(event, p) {
        // If a press-and-hold initiated a transfer, suppress the zoom click.
        if (suppressNextClick) {
          suppressNextClick = false;
          if (event && typeof event.stopImmediatePropagation === 'function') event.stopImmediatePropagation();
          if (event && typeof event.preventDefault === 'function') event.preventDefault();
          return;
        }
        const prevFocus = focus;
        // Toggle: if clicking the same node again, move focus to its parent.
        if (focus === p) {
          p = p.parent || p; // IMPORTANT: reassign p so calculations use the parent.
        }
        focus = p;
        if (typeof console !== 'undefined' && console.info) {
          try {
            const from = prevFocus && prevFocus.data ? (prevFocus.data.name || prevFocus.data.id || 'root') : 'root';
            const to = focus && focus.data ? (focus.data.name || focus.data.id || 'root') : 'root';
            tiDebug('interactions', 'click:', {
              clicked: (event && event.type) || 'click',
              node: p && p.data ? { name: p.data.name || null, id: p.data.id || null, depth: p.depth } : null,
              fromFocus: String(from),
              toFocus: String(to)
            });
          } catch (e) {}
        }
        root.each(d => d.target = {
          x0: (d.x0 - p.x0) / (p.x1 - p.x0) * height,
          x1: (d.x1 - p.x0) / (p.x1 - p.x0) * height,
          y0: d.y0 - p.y0,
          y1: d.y1 - p.y0,
        });
        const t = cell.transition().duration(750)
          .attr('transform', d => `translate(${d.target.y0},${d.target.x0})`);
        rects.transition(t).attr('height', d => rectHeight(d.target));
        texts.transition(t).attr('fill-opacity', d => +labelVisible(d.target));
        tspans.transition(t).attr('fill-opacity', d => labelVisible(d.target) * 0.7);
      }

      /**
       * Compute rectangle height with a minimal gap between rows.
       * Uses the node's x-extent (vertical space) with a 0–1px padding.
       * @param {{x0:number,x1:number}} d
       * @returns {number}
       */
      function rectHeight(d) {
        const h = d.x1 - d.x0;
        return Math.max(0, h - Math.min(1, h / 2));
      }

      /**
       * Determine if the text label for a node should be visible at current zoom.
       * Visible when horizontally within viewport and tall enough to read.
       * @param {{y0:number,y1:number,x0:number,x1:number}} d
       * @returns {boolean}
       */
      function labelVisible(d) {
        return d.y1 <= width && d.y0 >= 0 && (d.x1 - d.x0) > 16;
      }
      if (typeof console !== 'undefined' && console.info) {
        try {
          tiDebug('render', 'render: done');
        } catch (e) {}
      }
    }

    const instance = {
      /**
       * Mutable per-instance Terms forest backing the visualization.
       * External callers SHOULD treat this as internal and prefer API methods.
       * @type {Array<{id:number,name:string,children:Array}>}
       */
      Terms: terms,
      /**
       * Get a deep copy of the current terms forest.
       * @returns {Array<{id:number,name:string,children:Array}>}
       */
      getTerms() {
        return toForest(terms);
      },
      // Force a re-render of the visual state.
      render,
      /**
       * Select IDs from a tree (or array), rebuild from immutable reference, then re-render.
       * - Updates a per-instance selectedIds set based on payload IDs.
       * - Rebuilds Terms strictly from the global immutable tree, pruning others.
       * - Dispatches 'termsIcicle:updated' with { action: 'add', terms } detail.
       * @param {Object|Array} tree
       * @returns {Array} deep copy of updated terms
       */
      addTerms(tree) {
        const forest = toForest(tree);
        const ids = collectIdsFromForest(forest);
        if (typeof console !== 'undefined' && console.info) {
          try {
            tiDebug('interactions', 'addTerms(selection): before', { totalNodes: countNodes(terms), selectIds: ids.size });
          } catch (e) {}
        }
        ids.forEach(id => selectedIds.add(Number(id)));
        rebuildFromSelection();
        if (typeof console !== 'undefined' && console.info) {
          try {
            tiDebug('interactions', 'addTerms(selection): after', { totalNodes: countNodes(terms) });
          } catch (e) {}
        }
        render();
        el.dispatchEvent(new CustomEvent('termsIcicle:updated', { detail: { action: 'add', terms: this.getTerms() } }));
        return this.getTerms();
      },
      /**
       * Deselect IDs present in the provided tree/forest, rebuild from immutable, then re-render.
       * - Updates the per-instance selectedIds set by removing payload IDs.
       * - Rebuilds Terms strictly from the global immutable tree, pruning deselected nodes.
       * - Dispatches 'termsIcicle:updated' with { action: 'remove', terms } detail.
       * @param {Object|Array} tree
       * @returns {Array} deep copy of updated terms
       */
      removeTerms(tree) {
        const forest = toForest(tree);
        const ids = collectIdsFromForest(forest);
        if (typeof console !== 'undefined' && console.info) {
          try {
            tiDebug('interactions', 'removeTerms(selection): before', { totalNodes: countNodes(terms), deselectIds: ids.size });
          } catch (e) {}
        }
        ids.forEach(id => selectedIds.delete(Number(id)));
        rebuildFromSelection();
        if (typeof console !== 'undefined' && console.info) {
          try {
            tiDebug('interactions', 'removeTerms(selection): after', { totalNodes: countNodes(terms) });
          } catch (e) {}
        }
        render();
        el.dispatchEvent(new CustomEvent('termsIcicle:updated', { detail: { action: 'remove', terms: this.getTerms() } }));
        return this.getTerms();
      },
      /**
       * Destructor: clean up DOM and references when this icicle is removed.
       * @param {string} [reason] Optional reason or trigger (e.g., 'unload').
       */
      destroy(reason) {
        try { tiDebug('init', 'Destroying icicle', { id, reason }); } catch (e) {}

        // Take a snapshot of current terms before we do anything else.
        var currentTerms = [];
        try { currentTerms = this.getTerms(); } catch (e) { currentTerms = []; }
        var numTerms = 0;
        try {
          numTerms = (function count(nodes){ var c=0; (nodes||[]).forEach(function(n){ c++; if(Array.isArray(n.children)) c += count(n.children); }); return c; })(currentTerms);
        } catch (e) {}

        // If there are terms, try to move them into the default icicle (if configured).
        if (Array.isArray(currentTerms) && currentTerms.length) {
          try {
            var transfer = (window && window.icicles_transfer) ? window.icicles_transfer : null;
            var queued = false;
            try {
              if (transfer && typeof transfer.queuePendingOrphans === 'function' && icicleKey) {
                queued = !!transfer.queuePendingOrphans(icicleKey, currentTerms);
                if (queued) {
                  try { tiDebug('init', 'Queued terms for deferred transfer on destroy', { from: id, key: icicleKey, count: numTerms }); } catch (e) {}
                }
              }
            } catch (qe) {}

            if (!queued) {
              // Fallback to immediate transfer to default icicle (legacy behavior).
              var defaultId = (transfer && typeof transfer.getDefaultIcicleId === 'function') ? transfer.getDefaultIcicleId() : null;
              var registry = (Drupal && Drupal.taxonomyTermIcicles) ? Drupal.taxonomyTermIcicles : {};
              var canTransfer = defaultId && String(defaultId) !== String(id) && registry && registry[defaultId] && typeof registry[defaultId].addTerms === 'function';
              if (canTransfer) {
                try { tiDebug('init', 'Transferring terms to default icicle before destroy (immediate fallback)', { from: id, to: String(defaultId), count: numTerms }); } catch (e) {}
                try { registry[defaultId].addTerms(currentTerms); } catch (e) {}
              }
              else {
                if (typeof console !== 'undefined' && console.warn) {
                  console.warn('[terms_icicle] No valid default icicle set. Deleting', numTerms, 'term(s) from destroyed icicle', String(id), '— terms integrity may be broken.');
                }
              }
            }
          } catch (e) {}
        }

        // Clear this instance's selection/state to remove all terms, without re-rendering.
        try { if (typeof selectedIds !== 'undefined' && selectedIds && typeof selectedIds.clear === 'function') { selectedIds.clear(); } } catch (e) {}
        try { rebuildFromSelection(); } catch (e) {}

        // Clear any rendered content.
        try { clear(getTarget()); } catch (e) {}
        // Remove from internal/external registries.
        try { instances.delete(el); } catch (e) {}
        try { if (Drupal.taxonomyTermIcicles && Drupal.taxonomyTermIcicles[id]) { delete Drupal.taxonomyTermIcicles[id]; } } catch (e) {}
        // Dispatch a custom event to allow external listeners to react.
        try { el.dispatchEvent(new CustomEvent('termsIcicle:destroyed', { detail: { id, reason } })); } catch (e) {}
      },
    };

    // Store for internal and optional external access.
    // Store an optional stable key on the instance for cross-AJAX correlation.
    instance.key = icicleKey || '';

    // Store for internal and optional external access.
    instances.set(el, instance);
    Drupal.taxonomyTermIcicles[id] = instance;

    // If this icicle is flagged as the default destination, record its id in the manager transfer store.
    try {
      if (isDefault && window.icicles_transfer && typeof window.icicles_transfer.setDefaultIcicleId === 'function') {
        window.icicles_transfer.setDefaultIcicleId(id);
        try { tiDebug('init', 'Registered default icicle id', id); } catch (e) {}
        // Verify that the default id was actually recorded.
        try {
          var recorded = (typeof window.icicles_transfer.getDefaultIcicleId === 'function') ? window.icicles_transfer.getDefaultIcicleId() : null;
          if (String(recorded) !== String(id)) {
            if (typeof console !== 'undefined' && console.debug) {
              console.debug('[terms_icicle] Default icicle NOT recorded as expected.', { attempted: String(id), actual: String(recorded) });
            }
            try { tiDebug('init', 'Default icicle NOT recorded as expected', { attempted: String(id), actual: String(recorded) }); } catch (e) {}
          }
        } catch (e) {}
      } else if (isDefault) {
        // We expected to set the default icicle, but cannot because the transfer store or setter is unavailable.
        if (typeof console !== 'undefined' && console.debug) {
          console.debug('[terms_icicle] Could not set default icicle id — transfer store not ready or setter missing.', { id: String(id) });
        }
        try { tiDebug('init', 'Could not set default icicle id — transfer store not ready or setter missing', { id: String(id) }); } catch (e) {}
      }
    } catch (e) {}

    // Initial render
    render();

    return instance;
  }

  /**
   * Drupal behavior that initializes Terms Icicle instances on page attach.
   * Uses once() to ensure each .terms-icicle element is only initialized once
   * even if behaviors re-run due to AJAX or other lifecycle events.
   */
  Drupal.behaviors.termsIcicle = {
    /**
     * Behavior attach hook entry point.
     * @param {HTMLElement|Document} context - Drupal behavior context (may be an AJAX fragment).
     */
    attach(context) {
      try {
        const elements = once('terms-icicle', '.terms-icicle', context);
        if (typeof console !== 'undefined' && console.info) {
          tiDebug('init', 'Behavior attach: found', elements.length, 'element(s).');
        }
        elements.forEach(el => {
          try {
            createInstance(el);
          } catch (e) {
            if (typeof console !== 'undefined' && console.error) {
              console.error('[terms_icicle] Error during createInstance:', e);
            }
          }
        });
        // After initializing instances for this attach cycle, finalize any pending
        // orphan transfers: only move terms from icicles that did not reappear.
        try {
          setTimeout(function(){
            try {
              var transfer = (window && window.icicles_transfer) ? window.icicles_transfer : null;
              if (transfer && typeof transfer.finalizePendingOrphans === 'function') {
                transfer.finalizePendingOrphans();
              }
            } catch (e) {}
          }, 0);
        } catch (e) {}
      } catch (e) {
        if (typeof console !== 'undefined' && console.error) {
          console.error('[terms_icicle] Behavior attach error:', e);
        }
      }
    },
    /**
     * Behavior detach hook: called when Drupal removes or replaces DOM.
     * We use this to run a destructor for any .terms-icicle within the context.
     * @param {HTMLElement|Document} context
     * @param {object} settings
     * @param {string} trigger - 'unload' when content is being removed by AJAX.
     */
    detach(context, settings, trigger) {
      try {
        // Only act for actual DOM removals initiated by Drupal AJAX.
        if (trigger !== 'unload') {
          return;
        }
        // Guard against global/document-level detach calls which would sweep the whole page.
        var ctxEl = (context && context.nodeType === 1) ? context : null;
        if (!ctxEl) {
          return;
        }
        if (ctxEl === document || ctxEl === document.body || ctxEl === document.documentElement) {
          return;
        }

        const list = [];
        try {
          if (typeof ctxEl.querySelectorAll === 'function') {
            Array.prototype.forEach.call(ctxEl.querySelectorAll('.terms-icicle'), function (el) { list.push(el); });
          }
          if (ctxEl.classList && ctxEl.classList.contains('terms-icicle')) {
            list.push(ctxEl);
          }
        } catch (e) {}

        list.forEach(function (el) {
          const inst = instances.get(el);
          if (!inst) return;
          try { inst.destroy('unload'); } catch (e) {}
        });
      } catch (e) {
        if (typeof console !== 'undefined' && console.error) {
          console.error('[terms_icicle] Behavior detach error:', e);
        }
      }
    },
  };

})(Drupal, once, drupalSettings);
