<?php

namespace Drupal\taxonomy_term_config_groups;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Access control handler to prevent editing, deleting, cloning, or creating
 * taxonomy group type config entities via the UI.
 */
class TaxonomyGroupTypeAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity for which to check access.
   * @param string $operation
   *   The operation being performed: 'view', 'update', 'delete', etc.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user for which to check access.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    switch ($operation) {
      case 'view':
        // Allow viewing the bundle definition to users who can administer
        // taxonomy. This is needed for UI integrations like Field UI tabs.
        return AccessResult::allowedIfHasPermission($account, 'administer taxonomy');

      case 'update':
      case 'delete':
        // Disallow updates and deletions entirely.
        return AccessResult::forbidden();
    }

    return parent::checkAccess($entity, $operation, $account);
  }

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user for which to check access.
   * @param array<string, mixed> $context
   *   Additional access context.
   * @param string|null $entity_bundle
   *   (optional) The config entity bundle id.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    // Disallow creating new taxonomy group types via the UI.
    return AccessResult::forbidden();
  }

}
