<?php

namespace Drupal\taxonomy_term_config_groups;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * List builder for Taxonomy group type bundles.
 */
class TaxonomyGroupTypeListBuilder extends ConfigEntityListBuilder {

  /**
   * The entity type manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $dfConfigFactory;

  /**
   * Module handler service.
   */
  protected ModuleHandlerInterface $dfModuleHandler;

  /**
   * {@inheritdoc}
   *
   * @return static
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    /** @var static $instance */
    $instance = parent::createInstance($container, $entity_type);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->dfConfigFactory = $container->get('config.factory');
    $instance->dfModuleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   */
  public function buildHeader(): array {
    $header['label'] = $this->t('Label');
    $header['id'] = $this->t('Machine name');
    $header['vocabulary'] = $this->t('Vocabulary');
    $header['grouping'] = $this->t('Term grouping');
    $header['enabled'] = $this->t('Enabled');
    $header['groups'] = $this->t('Groups');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The taxonomy group type config entity.
   *
   * @return array<string, mixed>
   *   The row render array.
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\taxonomy_term_config_groups\Entity\TaxonomyGroupType $entity */
    $row['label'] = $entity->label();
    $row['id'] = $entity->id();

    // Try to infer the related vocabulary from the bundle id pattern 'vocab_<vid>'.
    $vocabulary_link = $this->t('-');
    $grouping_link = $this->t('-');
    $enabled_text = $this->t('Unknown');
    $id_raw = $entity->id();
    $id = is_scalar($id_raw) ? (string) $id_raw : '';
    $prefix = 'vocab_';
    if ($id !== '' && str_starts_with($id, $prefix)) {
      $vid = substr($id, strlen($prefix));
      $vocab = $this->entityTypeManager->getStorage('taxonomy_vocabulary')->load($vid);
      if ($vocab) {
        $url = Url::fromRoute('entity.taxonomy_vocabulary.edit_form', ['taxonomy_vocabulary' => $vid]);
        $vocabulary_link = Link::fromTextAndUrl((string) $vocab->label(), $url)->toRenderable();
      }
      else {
        $vocabulary_link = $this->t('@vid (missing)', ['@vid' => $vid]);
      }
      // Compute enabled flag from config.
      $enabled_vocabs = (array) ($this->dfConfigFactory->get('taxonomy_term_config_groups.settings')->get('enabled_vocabs') ?? []);
      $enabled = !empty($enabled_vocabs[$vid]);
      $enabled_text = $enabled ? $this->t('Yes') : $this->t('No');

      // Only show the grouping link when enabled for this vocabulary.
      if ($enabled && $vocab) {
        try {
          $group_url = Url::fromRoute('taxonomy_term_config_groups.grouping_form', ['taxonomy_vocabulary' => $vid]);
          $grouping_link = Link::fromTextAndUrl($this->t('Configure grouping'), $group_url)->toRenderable();
        }
        catch (\Throwable $e) {
          // Ignore if route is not available for any reason.
        }
      }
    }
    else {
      // Non-standard id; cannot infer vocabulary -> treat as disabled.
      $enabled_text = $this->t('No');
    }

    // Wrap render arrays in 'data' to prevent theme_table treating them as attributes.
    $row['vocabulary'] = is_array($vocabulary_link) ? ['data' => $vocabulary_link] : $vocabulary_link;
    $row['grouping'] = is_array($grouping_link) ? ['data' => $grouping_link] : $grouping_link;
    $row['enabled'] = (string) $enabled_text;

    // Count groups for this bundle using taxonomy_group storage.
    $group_storage = $this->entityTypeManager->getStorage('taxonomy_group');
    $groups = $group_storage->loadByProperties(['type' => $entity->id()]);
    $count = count($groups);
    $row['groups'] = (string) $count;

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   */
  public function getOperations(EntityInterface $entity): array {
    $operations = parent::getOperations($entity);

    // Disallow editing, deleting, or cloning from the operations list.
    unset($operations['edit'], $operations['delete'], $operations['clone'], $operations['duplicate']);

    // Add a 'Manage fields' operation when Field UI is available.
    if ($this->dfModuleHandler->moduleExists('field_ui')) {
      try {
        $url = Url::fromRoute('entity.taxonomy_group.field_ui_fields', [
          'taxonomy_group_type' => $entity->id(),
        ]);
        $operations['manage-fields'] = [
          'title' => $this->t('Manage fields'),
          'weight' => 20,
          'url' => $this->ensureDestination($url),
        ];
      }
      catch (\Throwable $e) {
        // If the route is not available, do not add the operation.
      }
    }

    return $operations;
  }

}
