# Terms Icicle form element (terms_icicle)

This document describes how to use the Terms Icicle form element provided by the taxonomy_term_config_groups module. The element renders a zoomable "icicle" visualization of taxonomy terms using D3 and lets users build a hierarchical selection of terms. It is designed to be reusable in your own Drupal forms beyond the included TaxonomyTermGroupingForm.

At a glance:
- Form API element type: `terms_icicle`
- Submitted value: an array "forest" of nodes `{ id: int, name: string, children: [] }`
- Visualization: D3 v7, auto-attached libraries
- Interactions: click to zoom, press-and-hold to transfer between icicles (when multiple are on the page)


## When to use
Use the `terms_icicle` element when you need a user-friendly UI to select a subset of terms from a vocabulary while preserving some visual sense of their hierarchy. The element works standalone or in pairs/groups to move selections between multiple icicles.


## Basic usage in a form
The element is a standard Form API element that you can add in `buildForm()`:

```php
use Drupal\Core\Form\FormStateInterface;

public function buildForm(array $form, FormStateInterface $form_state): array {
  // 1) Build and attach the immutable taxonomy tree (see next section for details).
  $vid = 'my_vocabulary';
  $tree_forest = mymodule_build_taxonomy_forest($vid); // [{ id, name, children: [...] }, ...]

  $form['#attached']['drupalSettings']['taxonomyTermConfigGroups'] = [
    'vocabulary' => [
      'id' => $vid,
      'label' => 'My Vocabulary',
    ],
    'tree' => $tree_forest,
    // Optional debug flags (see Advanced / Debugging).
    // 'debug' => [ 'init' => false, 'render' => false, 'interactions' => false ],
  ];

  // 2) Add the terms icicle element.
  $form['terms'] = [
    '#type' => 'terms_icicle',
    '#title' => $this->t('Select terms'),

    // If TRUE and no default value is provided, initialize with all terms selected.
    // If you provide a default, set this to FALSE so the default is respected.
    '#init_full' => FALSE,

    // Optional: show a custom empty-state message.
    '#empty_text' => $this->t('No terms selected yet. Click or transfer terms into this area.'),

    // Optional: treat this icicle as the default destination when others are removed.
    '#is_default' => FALSE,

    // Optional: a stable key (e.g. group UUID) used across AJAX to correlate instances.
    '#icicle_key' => '',

    // Default value can be an array forest, a single node, or a JSON string of the same shape.
    // Minimal forest only needs IDs – names can be blank; the JS restores hierarchy from the immutable tree.
    '#default_value' => [
      ['id' => 12, 'name' => '', 'children' => []],
      ['id' => 34, 'name' => '', 'children' => [
        ['id' => 56, 'name' => '', 'children' => []],
      ]],
    ],
  ];

  return $form;
}
```

Notes:
- You do NOT need to manually attach the JavaScript libraries when using `#type => 'terms_icicle'`. The element attaches its own libraries.
- You DO need to attach `drupalSettings.taxonomyTermConfigGroups.tree` and `vocabulary` as shown, so the JS has the immutable reference tree to render.


## Building the immutable taxonomy tree (drupalSettings)
The libraries expect a single, immutable "reference" forest representing the vocabulary tree. A minimal structure is:

```php
// Forest (array of nodes): [{ id: int, name: string, children: [...] }, ...]
$settings['taxonomyTermConfigGroups'] = [
  'vocabulary' => ['id' => $vid, 'label' => $vocab_label],
  'tree' => [
    ['id' => 1, 'name' => 'Top A', 'children' => [
      ['id' => 2, 'name' => 'Child A1', 'children' => []],
    ]],
    ['id' => 3, 'name' => 'Top B', 'children' => []],
  ],
];
```

The TaxonomyTermGroupingForm builds this forest by:
- Loading a flat list: `taxonomy_term_storage->loadTree($vid, 0, NULL, FALSE)`
- Converting it to a nested forest: nodes of `{ id, name, children: [] }`

You can replicate a similar helper (simplified example):

```php
function mymodule_build_taxonomy_forest(string $vid): array {
  $items = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->loadTree($vid, 0, NULL, FALSE);
  // Index of id => reference to node in $forest or nested children arrays.
  $forest = [];
  $index = [];
  foreach ($items as $it) {
    $node = [
      'id' => (int) $it->tid,
      'name' => (string) $it->name,
      'children' => [],
    ];
    $parent = (int) reset($it->parents);
    if ($parent === 0) {
      $forest[] = $node;
      $index[$node['id']] = &$forest[array_key_last($forest)];
    }
    else {
      // Defer children until parent exists; for brevity, this example assumes parent comes first.
      if (isset($index[$parent])) {
        $index[$parent]['children'][] = $node;
        $index[$node['id']] = &$index[$parent]['children'][array_key_last($index[$parent]['children'])];
      }
    }
  }
  return $forest;
}
```

If your load order is not guaranteed, implement a two-pass approach like the module’s `unflattenTerms()` (see source) which safely resolves parents.


## Element options (Form API properties)
The element supports these notable properties (see src/Element/TermsIcicle.php):
- `#init_full` (bool): If TRUE and there is no `#default_value`, the icicle starts with all terms selected. If you provide a default value, set this to FALSE.
- `#is_default` (bool): Marks one icicle as the default destination for orphaned terms when other icicles are destroyed (e.g., via AJAX remove). Recommended to set TRUE for the main/left icicle when using multiple icicles on a page.
- `#empty_text` (string): Override the empty-state message.
- `#icicle_key` (string): A stable identifier (like a group UUID) that survives AJAX rebuilds. Used by the JS to decide whether to discard or transfer pending orphaned terms if an icicle disappears and reappears.
- `#default_value` (array|object|json-string): The selected terms forest. Can be a single node or an array of nodes; JSON is also accepted. Minimal nodes need only `id`; `name` and `children` are optional.

The element also exposes/uses:
- `#attributes.id`: A DOM id is ensured automatically. You normally do not need to set it.
- Hidden input: `.terms-icicle__input` holds the JSON state for AJAX/submit.
- Visual target container: `.terms-icicle__state` where the D3 chart is rendered.


## Libraries and assets
The `terms_icicle` element attaches the required libraries for you:
- `taxonomy_term_config_groups/icicles_manager`
- `taxonomy_term_config_groups/terms_icicle`

These depend upon:
- `taxonomy_term_config_groups/d3` (D3 v7 provided via Composer in vendor/npm-asset/d3)
- Core: `core/drupal`, `core/drupalSettings`, `core/once`

Note: D3 is installed by Composer and referenced from the module's libraries.yml, avoiding external CDNs per Drupal.org policy.

Additional styling used by the built-in grouping page is provided by `taxonomy_term_config_groups/grouping_form` but is not required for the element itself.


## Required drupalSettings structure
You must attach the immutable reference data that powers the icicles:

```php
$form['#attached']['drupalSettings']['taxonomyTermConfigGroups'] = [
  'vocabulary' => [ 'id' => $vid, 'label' => $label ],
  'tree' => $forest, // REQUIRED
  // Optional: toggle console-style debug flags.
  // 'debug' => [ 'init' => false, 'render' => false, 'interactions' => false ],
];
```

Per-element settings are attached automatically by the element during preRender under:

```
drupalSettings.taxonomyTermConfigGroups.elements[<element_dom_id>] = {
  initFull: boolean,
  isDefault: boolean,
  key: string
}
```

You do not need to set `elements[...]` yourself.


## What gets submitted (value shape)
On submit (and via AJAX), the element’s value is a normalized PHP array forest:

```php
$value = $form_state->getValue('terms');
// Example shape:
// [
//   ['id' => 12, 'name' => '...', 'children' => []],
//   ['id' => 34, 'name' => '...', 'children' => [ ['id' => 56, 'name' => '...', 'children' => []] ]],
// ]
```

You can extract IDs with a small helper:

```php
function mymodule_terms_ids_from_forest($forest): array {
  $ids = [];
  $stack = is_array($forest) ? $forest : [];
  while ($stack) {
    $node = array_pop($stack);
    if (is_array($node)) {
      if (isset($node['id'])) {
        $ids[] = (int) $node['id'];
      }
      if (!empty($node['children']) && is_array($node['children'])) {
        foreach ($node['children'] as $child) {
          $stack[] = $child;
        }
      }
    }
  }
  $ids = array_values(array_unique(array_map('intval', $ids)));
  sort($ids, SORT_NUMERIC);
  return $ids;
}
```

The element’s `#value_callback` ensures that:
- JSON strings are decoded into arrays
- Single nodes are wrapped as one-item forests
- Invalid input gracefully becomes an empty array


## JavaScript behavior and events
Two scripts collaborate:
- `icicles_manager.js` sets up an immutable global with the reference `tree`, a `rank` map (for stable ordering), and per-term `colors`. It also provides a mutable transfer store used when transferring terms between icicles.
- `terms_icicle.js` initializes each `.terms-icicle` element, renders the D3 chart, and keeps the hidden input in sync.

Custom DOM events (dispatched on the element wrapper):
- `termsIcicle:updated` — fired after `addTerms()` or `removeTerms()`. Detail: `{ action: 'add'|'remove', terms: <forest> }`.
- `termsIcicle:destroyed` — fired when an instance is torn down (e.g., AJAX detach). Detail: `{ id: <element_id>, reason: 'unload' | string }`.
- `termsIcicle:nodeHoldStart` — fired when the user presses-and-holds on a node to transfer. Detail: `{ icicleId: <element_id>, tree: <forest of the held subtree> }`. The manager listens for this to coordinate transfers.

Accessing instances (for advanced integrations):
- Each instance is stored on `Drupal.taxonomyTermIcicles[<element_id>]` and exposes methods:
  - `getTerms(): Array` — deep copy of current forest
  - `render(): void` — re-render
  - `addTerms(treeOrForest): Array` — select IDs present in the payload (from the immutable reference), rebuild, and render
  - `removeTerms(treeOrForest): Array` — deselect IDs present in the payload, rebuild, and render
  - `destroy(reason?): void` — manually tear down an instance

Transfer store (advanced):
- `window.icicles_transfer` is a globally exposed, mutable buffer with methods:
  - `addTerms(tree, fromIcicleId?)`, `takeTerms()`, `getTerms()`, `clear()`
  - `getSourceId()`, `setSourceId(id)`
  - `getDefaultIcicleId()`, `setDefaultIcicleId(id)` — the default destination icicle
  - `queuePendingOrphans(key, tree)`, `finalizePendingOrphans()` — supports AJAX rebuilds when an icicle goes away and may or may not reappear


## Multiple icicles on one page
You can place several `terms_icicle` elements in the same form. Common patterns:
- Mark one icicle with `#is_default => TRUE` so that if another icicle is destroyed, its terms have a home to go to.
- Set a stable `#icicle_key` (e.g. a group UUID) for any icicle that might be rebuilt via AJAX. If an icicle disappears across an AJAX rebuild and does not reappear with the same key, its queued terms will be finalized into the default icicle.
- The user can press-and-hold on a node to pick it up and release the mouse over another icicle to drop it there. A floating ghost chip shows how many terms are being transferred.


## Twig/markup overview (rendered by the element)
Rendered structure (simplified):

```html
<div class="terms-icicle" id="terms-icicle-1" data-icicle-key="...">
  <div class="terms-icicle__state">
    <!-- JS renders the SVG here; an empty-state box is used as fallback -->
    <div class="terms-icicle__empty-box">
      <p class="terms-icicle__empty-text">No terms selected...</p>
    </div>
  </div>
  <input type="hidden" class="terms-icicle__input" value="[...json forest...]" />
</div>
```

You don’t need to build this markup yourself; the element and template handle it.


## AJAX considerations
- The behavior uses `once()` to initialize each element only once per attach.
- On AJAX detach (`trigger === 'unload'`), the instance calls `destroy()` which attempts to either queue its terms (by `#icicle_key`) or transfer them to the default icicle immediately.
- On the next attach, pending orphan queues are finalized: if an icicle with the same `#icicle_key` reappears, its queue is discarded; otherwise, queued terms are transferred into the default icicle.


## Debugging
You can enable console logging per area via drupalSettings:

```php
$form['#attached']['drupalSettings']['taxonomyTermConfigGroups']['debug'] = [
  'init' => TRUE,
  'render' => TRUE,
  'interactions' => TRUE,
];
```


## Edge cases and tips
- If you provide `#default_value`, set `#init_full` to FALSE so the default is not replaced by the full vocabulary.
- Minimal default values are fine: `{ id: <tid>, children: [] }` (names are optional).
- The element gracefully handles empty/missing trees, but it will render an empty state. Always provide `drupalSettings.taxonomyTermConfigGroups.tree`.
- The D3 library is loaded from a CDN by default. If your environment blocks external JS, override the `taxonomy_term_config_groups/d3` library definition to point to a local asset.
- To extract term IDs from the submitted value, traverse the forest as shown above; both parent and descendant IDs are included.


## References
- Element class: `src/Element/TermsIcicle.php`
- Example form: `src/Form/TaxonomyTermGroupingForm.php`
- JS: `js/icicles_manager.js`, `js/terms_icicle.js`
- Twig template: `templates/terms-icicle.html.twig`
- Libraries: `taxonomy_term_config_groups.libraries.yml`
