<?php

namespace Drupal\taxonomy_term_config_groups\Element;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\FormElementBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides a simple Terms Icicle form element.
 *
 * @FormElement("terms_icicle")
 */
class TermsIcicle extends FormElementBase {

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   */
  public function getInfo(): array {
    return [
      '#input' => TRUE,
      // Use a dedicated theme to render the element wrapper and empty state.
      '#theme' => 'terms_icicle',
      '#attributes' => [
        'class' => ['terms-icicle'],
      ],
      // Whether the icicle should be initialized with all available terms.
      '#init_full' => FALSE,
      // Whether this icicle instance should be treated as the default destination
      // for orphaned terms when other icicles are destroyed.
      '#is_default' => FALSE,
      // Empty-state text can be overridden per instance if desired.
      '#empty_text' => new TranslatableMarkup('No terms selected. Start by adding terms to the grouping.'),
      '#process' => [
        [static::class, 'processAjaxForm'],
        [static::class, 'processGroup'],
      ],
      '#pre_render' => [
        [static::class, 'preRenderGroup'],
      ],
      '#attached' => [
        'library' => [
          'taxonomy_term_config_groups/icicles_manager',
          'taxonomy_term_config_groups/terms_icicle',
        ],
      ],
      // Explicit value callback to decode JSON into arrays for form state.
      '#value_callback' => [static::class, 'valueCallback'],
    ];
  }

  /**
   * Process callback: keeps element structure consistent.
   *
   * @param array<string,mixed> $element
   *   The element render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array<string,mixed> $complete_form
   *   The complete form render array.
   *
   * @return array<string,mixed>
   *   The processed element.
   */
  public static function processGroup(&$element, FormStateInterface $form_state, &$complete_form) {
    // Ensure expected child container exists.
    if (!isset($element['content']) || !is_array($element['content'])) {
      $element['content'] = ['#type' => 'container'];
    }
    /** @var array<string,mixed> $content */
    $content = $element['content'];

    // Create a hidden input to carry the JSON-encoded state. Place it as a
    // sibling of the actual render target so it won't be cleared by JS.
    if (!isset($content['input'])) {
      $default = '';
      if (isset($element['#default_value'])) {
        if (is_array($element['#default_value'])) {
          $default = (string) json_encode($element['#default_value']);
        }
        elseif (is_string($element['#default_value'])) {
          $default = $element['#default_value'];
        }
      }
      $content['input'] = [
        '#type' => 'hidden',
        '#default_value' => $default,
        '#attributes' => [
          'class' => ['terms-icicle__input'],
        ],
      ];
      // Submit the hidden input under the same parents as the main element so
      // the element's valueCallback receives the raw JSON string directly.
      $content['input']['#parents'] = $element['#parents'];
    }

    // Ensure a predictable container for the visualization target.
    if (!isset($content['state']) || !is_array($content['state'])) {
      $content['state'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['terms-icicle__state'],
        ],
      ];
    }
    /** @var array<string,mixed> $state */
    $state = $content['state'];
    if (!isset($state['#attributes']) || !is_array($state['#attributes'])) {
      $state['#attributes'] = [];
    }
    /** @var array<string,mixed> $attributes */
    $attributes = $state['#attributes'];
    if (!isset($attributes['class']) || !is_array($attributes['class'])) {
      $attributes['class'] = [];
    }
    if (!in_array('terms-icicle__state', $attributes['class'], TRUE)) {
      $attributes['class'][] = 'terms-icicle__state';
    }
    $state['#attributes'] = $attributes;
    $content['state'] = $state;

    // Server-side empty-state fallback: render a box with text so the page
    // shows meaningful UI even if JS is disabled or not yet executed.
    $state_keys = array_keys($content['state']);
    if (!isset($content['state']['#markup']) && empty(array_diff($state_keys, ['#type', '#attributes', '#markup']))) {
      $text = 'No terms selected.';
      if (isset($element['#empty_text'])) {
        $empty_text = $element['#empty_text'];
        if (is_string($empty_text)) {
          $text = $empty_text;
        }
        elseif ($empty_text instanceof TranslatableMarkup) {
          $text = (string) $empty_text;
        }
      }
      $content['state']['#markup'] = '<div class="terms-icicle__empty-box"><p class="terms-icicle__empty-text">' . $text . '</p></div>';
      $content['state']['#allowed_tags'] = ['div', 'p'];
    }

    // Write back the normalized content structure.
    $element['content'] = $content;

    return $element;
  }

  /**
   * Value callback: decode the submitted JSON into a PHP array (forest).
   *
   * Ensures that Form API consumers read and write arrays instead of JSON
   * strings. The browser still submits a JSON string via the hidden input, but
   * the value exposed through $form_state is a normalized array.
   *
   * @param array<string,mixed> $element
   *   The element render array.
   * @param mixed $input
   *   The raw submitted input or programmatic value.
   *
   * @return array<int|string,mixed>
   *   A forest array.
   */
  public static function valueCallback(&$element, $input, FormStateInterface $form_state) {
    // Helper: normalize any array into a forest (wrap a single assoc node).
    $toForest = static function ($value): array {
      if (is_array($value)) {
        // If associative, wrap as a single-node forest; otherwise assume list.
        $is_list = array_keys($value) === range(0, count($value) - 1);
        return $is_list ? $value : [$value];
      }
      return [];
    };

    if ($input !== FALSE && $input !== NULL) {
      // Input will be either the JSON string (since we aligned #parents) or an
      // already-decoded array in programmatic submissions.
      if (is_string($input)) {
        $input = trim($input);
        if ($input === '') {
          return [];
        }
        $decoded = json_decode($input, TRUE);
        if (json_last_error() === JSON_ERROR_NONE && $decoded !== NULL) {
          return $toForest($decoded);
        }
        // On parse failure, return empty array rather than raw string.
        return [];
      }
      if (is_array($input)) {
        // In case the hidden input did not align parents, attempt to extract it.
        if (isset($input['content']) && is_array($input['content']) && isset($input['content']['input']) && is_string($input['content']['input'])) {
          $decoded = json_decode($input['content']['input'], TRUE);
          if (json_last_error() === JSON_ERROR_NONE && $decoded !== NULL) {
            return $toForest($decoded);
          }
          return [];
        }
        return $toForest($input);
      }
      // Any other scalar: ignore.
      return [];
    }

    // No input (initial build/AJAX): use default value if provided.
    if (isset($element['#default_value'])) {
      if (is_array($element['#default_value'])) {
        return $toForest($element['#default_value']);
      }
      if (is_string($element['#default_value'])) {
        $decoded = json_decode($element['#default_value'], TRUE);
        if (json_last_error() === JSON_ERROR_NONE && $decoded !== NULL) {
          return $toForest($decoded);
        }
      }
    }
    return [];
  }

  /**
   * Pre-render callback: ensure DOM id and attach per-element settings.
   *
   * @param array<string,mixed> $element
   *   The element render array.
   *
   * @return array<string,mixed>
   *   The processed element.
   */
  public static function preRenderGroup($element) {
    // Ensure the element has a stable id so JS can look up settings.
    $attributes = [];
    if (isset($element['#attributes']) && is_array($element['#attributes'])) {
      /** @var array<string,mixed> $attributes */
      $attributes = $element['#attributes'];
    }
    if (empty($attributes['id'])) {
      $attributes['id'] = Html::getUniqueId('terms-icicle');
    }
    // Ensure $id is a string for array keys and JS.
    $id_value = $attributes['id'];
    $id = is_scalar($id_value) ? (string) $id_value : Html::getUniqueId('terms-icicle');
    $attributes['id'] = $id;
    $element['#attributes'] = $attributes;

    $init_full = !empty($element['#init_full']);
    $is_default = !empty($element['#is_default']);

    // Optional stable icicle key (e.g., group UUID) for lifecycle coordination across AJAX.
    $icicle_key = isset($element['#icicle_key']) && is_scalar($element['#icicle_key']) ? (string) $element['#icicle_key'] : '';
    if ($icicle_key !== '') {
      // Expose as a data attribute to assist non-settings consumers.
      if (!isset($attributes['data-icicle-key'])) {
        $attributes['data-icicle-key'] = $icicle_key;
        $element['#attributes'] = $attributes;
      }
    }

    // Attach per-element drupalSettings so JS can initialize appropriately.
    // $id computed above as a string-safe identifier.
    // Build drupalSettings structure under #attached.
    $attached = [];
    if (isset($element['#attached']) && is_array($element['#attached'])) {
      /** @var array<string,mixed> $attached */
      $attached = $element['#attached'];
    }
    if (!isset($attached['drupalSettings']) || !is_array($attached['drupalSettings'])) {
      $attached['drupalSettings'] = [];
    }
    /** @var array<string,mixed> $drupal_settings */
    $drupal_settings = $attached['drupalSettings'];
    if (!isset($drupal_settings['taxonomyTermConfigGroups']) || !is_array($drupal_settings['taxonomyTermConfigGroups'])) {
      $drupal_settings['taxonomyTermConfigGroups'] = [];
    }
    if (!isset($drupal_settings['taxonomyTermConfigGroups']['elements']) || !is_array($drupal_settings['taxonomyTermConfigGroups']['elements'])) {
      $drupal_settings['taxonomyTermConfigGroups']['elements'] = [];
    }
    $drupal_settings['taxonomyTermConfigGroups']['elements'][$id] = [
      'initFull' => (bool) $init_full,
      'isDefault' => (bool) $is_default,
      'key' => $icicle_key,
    ];
    $attached['drupalSettings'] = $drupal_settings;
    $element['#attached'] = $attached;

    return $element;
  }

}
