<?php

namespace Drupal\taxonomy_term_config_groups\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\taxonomy\TermInterface;
use Drupal\taxonomy_term_config_groups\Entity\TaxonomyGroup;

/**
 * Provides helpers to look up taxonomy group config for terms/vocabularies.
 */
class TaxonomyGroupLookup {

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Get the taxonomy group for a given term (entity or TID).
   *
   * If multiple groups reference the term (should not normally happen), the
   * first matching group by ascending entity ID is returned for determinism.
   * Returns NULL when grouping is disabled for the vocabulary or no match.
   */
  public function getGroupForTerm(TermInterface|int $term): ?TaxonomyGroup {
    $term_entity = $this->ensureTermEntity($term);
    if (!$term_entity) {
      return NULL;
    }

    $vid = (string) $term_entity->bundle();
    if (!$this->isEnabledForVocabulary($vid)) {
      return NULL;
    }

    $tid = (int) $term_entity->id();
    if ($tid <= 0) {
      return NULL;
    }

    $bundle_id = \taxonomy_term_config_groups_bundle_id_from_vid($vid);

    // Query for groups in the vocabulary's bundle that reference this term.
    $storage = $this->entityTypeManager->getStorage('taxonomy_group');

    $query = $storage->getQuery()
      ->condition('type', $bundle_id)
      ->condition('field_terms.target_id', $tid)
      ->sort('id', 'ASC')
      ->range(0, 1)
      ->accessCheck(FALSE);

    $ids = $query->execute();
    if (empty($ids)) {
      return NULL;
    }
    $entities = $storage->loadMultiple($ids);
    /** @var \Drupal\taxonomy_term_config_groups\Entity\TaxonomyGroup|null $first */
    $first = reset($entities);
    return $first instanceof TaxonomyGroup ? $first : NULL;
  }

  /**
   * Get all taxonomy groups for a given vocabulary id.
   *
   * @return array<int, TaxonomyGroup>
   *   Loaded group entities keyed by entity ID. Empty when disabled or none.
   */
  public function getGroupsForVocabulary(string $vid): array {
    if (!$this->isEnabledForVocabulary($vid)) {
      return [];
    }
    $bundle_id = \taxonomy_term_config_groups_bundle_id_from_vid($vid);
    $storage = $this->entityTypeManager->getStorage('taxonomy_group');
    $ids = $storage->getQuery()
      ->condition('type', $bundle_id)
      ->sort('label', 'ASC')
      ->accessCheck(FALSE)
      ->execute();
    if (empty($ids)) {
      return [];
    }
    $entities = $storage->loadMultiple($ids);
    // Narrow typing for consumers.
    return array_filter($entities, static fn ($e) => $e instanceof TaxonomyGroup);
  }

  /**
   * Resolve a taxonomy Term entity from either an entity or a TID.
   */
  protected function ensureTermEntity(TermInterface|int $term): ?TermInterface {
    if ($term instanceof TermInterface) {
      return $term;
    }
    $tid = (int) $term;
    if ($tid <= 0) {
      return NULL;
    }
    $storage = $this->entityTypeManager->getStorage('taxonomy_term');
    $entity = $storage->load($tid);
    return $entity instanceof TermInterface ? $entity : NULL;
  }

  /**
   * Check if grouping is enabled for a vocabulary.
   */
  protected function isEnabledForVocabulary(string $vid): bool {
    $config = $this->configFactory->get('taxonomy_term_config_groups.settings');
    $enabled = (array) ($config->get('enabled_vocabs') ?? []);
    return !empty($enabled[$vid]);
  }

}
