# Taxonomy Term Config Groups — Module README

This module adds a reusable way to define configuration "groups" of taxonomy terms on a per‑vocabulary basis. Instead of repeating the same settings on every term, you create one or more Taxonomy Groups and assign terms to each group. Groups are fieldable entities, so site builders can attach any fields needed (booleans, text, references, etc.) and then other custom code can read those fields and apply behavior for all terms in that group.

Key points at a glance:
- Per‑vocabulary toggle: enable/disable grouping on each taxonomy vocabulary.
- Fieldable groups: a Taxonomy Group is a content entity; its bundle is created per vocabulary.
- Visual grouping UI: a D3 "Terms Icicle" element is used to select and organize terms into groups.
- Safe lifecycle: disabling preserves data; deleting the vocabulary deletes its group bundle and groups.


## Contents
- What it provides
- Enabling/disabling grouping for a vocabulary
- What happens when enabled/disabled/deleted
- Entities and why groups are fieldable
- Grouping UI and how it saves data
- Example use‑case
- Developer notes and references


## What it provides
- A content entity: Taxonomy Group (entity type: `taxonomy_group`).
  - Base fields: label, machine_name (auto‑generated), field_terms (stores selected term IDs), created, changed.
  - Bundle: `taxonomy_group_type` (config entity), created per vocabulary.
- A per‑vocabulary admin UI to build groups of terms using a zoomable icicle visualization.
- Programmatic helpers and lifecycle hooks to keep bundles and data in sync with vocabularies.


## Enable/disable grouping per vocabulary
Grouping is controlled on each vocabulary’s edit form.

How to enable:
1. Go to: Administration → Structure → Taxonomy → Edit the desired vocabulary.
2. Check “Enable term config groups for this vocabulary”.
3. Save the vocabulary.

What happens when you enable:
- The module creates (or updates) a bundle for Taxonomy Groups named `vocab_<vid>` where `<vid>` is the vocabulary machine name. The bundle label is derived from the vocabulary label (e.g., “<Vocabulary> groups”).
- A “Configure term grouping” operation appears for that vocabulary (in its operations drop‑down and on the Group Types list page) linking to the grouping UI.

How to disable:
- Uncheck the same checkbox on the vocabulary edit form and save.

What happens when you disable:
- The vocabulary is marked as disabled for grouping in the module configuration, and the “Configure term grouping” operation is hidden.
- IMPORTANT: Existing data is preserved (the bundle and its `taxonomy_group` entities are NOT deleted). This allows you to re‑enable later without losing work.

Where the flag is stored:
- Configuration object: `taxonomy_term_config_groups.settings`
- Key: `enabled_vocabs[<vid>] = TRUE|FALSE`


## What happens when a vocabulary is deleted
If you delete a vocabulary that has grouping enabled (or previously enabled), the module performs a cleanup:
- It warns on the confirmation form that deleting the vocabulary will also delete the associated group bundle and groups, and shows the number of groups affected.
- On delete, it removes:
  - All `taxonomy_group` entities whose bundle is `vocab_<vid>`.
  - The `taxonomy_group_type` bundle `vocab_<vid>` itself.
  - The `enabled_vocabs[<vid>]` flag in configuration.

This keeps configuration consistent and avoids orphaned group entities.


## Entities and why groups are fieldable
Two entity types are central to this module:

- taxonomy_group_type (Config entity, bundle of `taxonomy_group`)
  - ID pattern: `vocab_<vid>`—one bundle per vocabulary.
  - Managed programmatically (created/updated automatically) but there is a list at: /admin/structure/taxonomy-groups/types.
  - If the bundle ID follows `vocab_<vid>`, a config dependency on the vocabulary is declared, so the relationship shows up in config dependency reports.

- taxonomy_group (Content entity)
  - Field UI base route: `entity.taxonomy_group_type.edit_form`, so bundles are fieldable with Field UI.
  - Base fields include `field_terms` (entity_reference to taxonomy_term), used to store the selected terms for the group.
  - The entity auto‑generates a unique `machine_name` from the label on save (per bundle) for convenient programmatic use.

Why fieldable?
- Site builders can attach fields to the group bundles to express configuration shared by all included terms (e.g., “Show in filter”, “Facet boost weight”, “Associated landing page”, “Shipping class”, “Discount code”, etc.). Downstream code reads group entities to derive behavior for any term belonging to that group.

Managing fields:
- Navigate to “Taxonomy group types” at /admin/structure/taxonomy-groups/types.
- When Field UI is enabled, a “Manage fields” operation is available for each bundle.


## Grouping UI and how it saves data
Path:
- /admin/structure/taxonomy/manage/{taxonomy_vocabulary}/grouping
- Also accessible via the vocabulary’s operations when grouping is enabled: “Configure term grouping”.

What you see:
- A two‑column UI:
  - Left: an icicle of all vocabulary terms (the default destination for unassigned or orphaned terms).
  - Right: a list of groups you add. Each group has a label, an internal ID (read‑only machine name shown after save), a Terms Icicle, and a “Group configuration” details section that exposes fields from the `taxonomy_group` bundle (via the entity form display for that bundle).

How it works:
- The UI is built by `TaxonomyTermGroupingForm` and uses a custom form element `#type => terms_icicle` for term selection.
- Terms can be transferred between icicles by press‑and‑hold and mouse release. The left icicle starts with unassigned terms; each group’s icicle holds the terms assigned to that group.
- On Save:
  - The form creates/updates/deletes `taxonomy_group` entities for the vocabulary’s bundle (`vocab_<vid>`), based on the groups present in the form.
  - The selected terms for each group are stored in the group’s base field `field_terms`.
  - Any fields you added to the group bundle are extracted from the “Group configuration” subform and saved on the corresponding `taxonomy_group` entity.

Important implementation details:
- Groups in the UI are keyed by a stable UUID in the form state, which is used to either update an existing entity with that UUID or create a new one preserving it.
- The icicles render from an immutable reference tree of terms (attached via `drupalSettings`) and maintain the selected subset per icicle.

For details on the element and JS behaviors, see: src/Element/README.md (Terms Icicle form element).


## Example use‑case
Imagine a product taxonomy with hundreds of terms across a deep hierarchy. You want to control several behaviors collectively—for example:
- Faceted search boosting (give certain applications higher priority).
- Show/hide groups of terms together in navigation.
- Attach a landing page and marketing copy shared by many terms.
- Map groups of terms to fulfillment rules (e.g., a shipping class or courier).

Without this module, you would need to add new fields directly to the taxonomy term and repeat the same values across many terms, or write ad‑hoc configuration elsewhere. With Taxonomy Term Config Groups:
- You enable grouping on the “Applications” vocabulary.
- Create groups like “Civil & Infrastructure”, “Commercial Landscaping”, “Domestic DIY”.
- Assign the relevant terms to each group using the icicle UI.
- Add fields to the group bundle such as “Show in filter”, “Boost weight”, “Landing page”, “Shipping class”.
- Your custom business logic (in another module) reads the group entity for a term and applies the settings consistently anywhere that term appears.

This keeps configuration DRY, auditable in one place, and far faster to maintain than per‑term edits.


## Developer notes and references
Key code locations:
- Entities
  - Content entity: `src/Entity/TaxonomyGroup.php`
  - Config bundle: `src/Entity/TaxonomyGroupType.php`
  - List builder: `src/TaxonomyGroupTypeListBuilder.php`
- Form/UI
  - Grouping form: `src/Form/TaxonomyTermGroupingForm.php`
  - Terms Icicle element: `src/Element/TermsIcicle.php` and `src/Element/README.md`
  - Libraries: `taxonomy_term_config_groups.libraries.yml` (`icicles_manager`, `terms_icicle`, D3 via Composer vendor)
  - Twig template for element: `templates/terms-icicle.html.twig`
- Hooks and lifecycle
  - Vocabulary edit alter / enable toggle: `taxonomy_term_config_groups_form_taxonomy_vocabulary_form_alter()`
  - Enable/disable submit: `taxonomy_term_config_groups_vocabulary_form_submit()`
  - Vocabulary operations link: `taxonomy_term_config_groups_entity_operation_alter()`
  - Delete confirm warning: `taxonomy_term_config_groups_form_taxonomy_vocabulary_confirm_delete_alter()`
  - Cleanup on vocabulary delete: `taxonomy_term_config_groups_entity_delete()`
  - Bundle helpers: `taxonomy_term_config_groups_ensure_bundle()`, `taxonomy_term_config_groups_delete_bundle()`
  - Bundle id helper: `taxonomy_term_config_groups_bundle_id_from_vid()`

Routes:
- List bundles: `/admin/structure/taxonomy-groups/types`
- Grouping UI: `/admin/structure/taxonomy/manage/{taxonomy_vocabulary}/grouping`

Permissions:
- Custom permission: `administer taxonomy term config groups` — required to access the module’s admin routes and operations.

Dependencies:
- Drupal core taxonomy and field modules.

Testing and quality:
- Follow project guidelines to run phpunit, phpstan, and phpcs before merging changes.


## Maintainers
- Daniel Johnson (daniel.j) — Maintainer
  - Drupal.org profile: https://www.drupal.org/u/danielj



## Developer usage: Lookup service
Developers can fetch the taxonomy group configuration for a taxonomy term using the provided service taxonomy_term_config_groups.group_lookup.

Examples:
- Get a group for a term entity:

```php
/** @var \Drupal\taxonomy_term_config_groups\Service\TaxonomyGroupLookup $lookup */
$lookup = \Drupal::service('taxonomy_term_config_groups.group_lookup');
$group = $lookup->getGroupForTerm($term); // TermInterface
if ($group) {
  // Read fields like $group->get('field_terms'), or custom fields on the bundle.
}
```

- Get a group for a term ID (TID):

```php
$group = \Drupal::service('taxonomy_term_config_groups.group_lookup')->getGroupForTerm(123);
```

- Get all groups for a vocabulary:

```php
$groups = \Drupal::service('taxonomy_term_config_groups.group_lookup')->getGroupsForVocabulary('my_vocab');
```

Notes:
- If grouping is disabled for the vocabulary, the service returns NULL (for a term) or an empty array (for a vocabulary).
- When multiple groups reference the same term (should not normally occur), the first match by ascending entity ID is returned for determinism.
