/* eslint no-console: "off", func-names: "off", no-plusplus: "off", valid-jsdoc: "off", no-empty: ["error", { "allowEmptyCatch": true }], no-use-before-define: ["error", { "functions": false, "classes": true, "variables": true }] */
(function (Drupal) {
  // Ensure a global, mutable debug controller exists. This allows turning
  // logging on/off per area: 'init', 'render', 'interactions'.
  function ensureDebugConfig(settings) {
    if (window.icicles_debug) {
      return;
    }
    const defaults = { init: true, render: false, interactions: true };
    const fromSettings =
      settings &&
      settings.taxonomyTermConfigGroups &&
      settings.taxonomyTermConfigGroups.debug
        ? settings.taxonomyTermConfigGroups.debug
        : {};
    const flags = { ...defaults, ...(fromSettings || {}) };
    const api = {
      isEnabled(area) {
        return !!flags[area];
      },
      set(area, enabled) {
        flags[area] = !!enabled;
        return !!flags[area];
      },
      enableAll() {
        Object.keys(defaults).forEach(function (k) {
          flags[k] = true;
        });
        return api.getAll();
      },
      disableAll() {
        Object.keys(defaults).forEach(function (k) {
          flags[k] = false;
        });
        return api.getAll();
      },
      getAll() {
        return { ...flags };
      },
    };
    Object.defineProperty(window, 'icicles_debug', {
      value: api,
      writable: false,
      configurable: false,
      enumerable: true,
    });
  }

  // Local log helper that respects icicles_debug flags for info-level logs.
  function debugLog(area, ...args) {
    try {
      if (
        window.icicles_debug &&
        window.icicles_debug.isEnabled(area) &&
        typeof console !== 'undefined' &&
        console.info
      ) {
        console.info(`[icicles_manager][${area}]`, ...args);
      }
    } catch (e) {}
  }

  function isObject(o) {
    return o !== null && typeof o === 'object';
  }

  function deepFreeze(obj) {
    if (!isObject(obj)) {
      return obj;
    }
    Object.getOwnPropertyNames(obj).forEach(function (prop) {
      const value = obj[prop];
      if (isObject(value)) {
        deepFreeze(value);
      }
    });
    return Object.freeze(obj);
  }

  function setGlobalOnce(name, value) {
    if (Object.prototype.hasOwnProperty.call(window, name)) {
      return;
    }
    Object.defineProperty(window, name, {
      value,
      writable: false,
      configurable: false,
      enumerable: true,
    });
  }

  Drupal.behaviors.icicles_manager = {
    attach: function attach(context, settings) {
      // Ensure debug flags are initialized from drupalSettings (optional overrides).
      ensureDebugConfig(settings);
      // Only act once per page load. Behaviors can re-run; guard by checking
      // if the global has already been set.
      if (window.icicles_manager) {
        // Even if the immutable global already exists, we still want to ensure
        // the mutable transfer store is available once per page.
        ensureTransferStore(settings);
        ensureInteractionHandlers();
        return;
      }

      const data =
        settings && settings.taxonomyTermConfigGroups
          ? settings.taxonomyTermConfigGroups
          : null;
      if (!data) {
        // Still ensure the transfer store exists, even if no data payload.
        ensureTransferStore(settings);
        return;
      }

      let tree = null;
      if (data && typeof data.tree !== 'undefined') {
        tree = data.tree;
      } else if (data && data.terms) {
        tree = data.terms;
      }
      const payload = {
        vocabulary: data.vocabulary || null,
        tree,
      };

      // Compute a stable rank map (id => rank index) from the immutable tree.
      payload.rank = buildImmutableRankObject(payload.tree);

      // Compute a global, immutable color mapping (id => css color string).
      payload.colors = buildImmutableColorsMap(payload.tree);

      // Freeze deeply and expose as a global immutable variable.
      const frozen = deepFreeze(payload);
      setGlobalOnce('icicles_manager', frozen);

      // Also ensure the mutable transfer store is available.
      ensureTransferStore(settings);
      ensureInteractionHandlers();
    },
  };

  /**
   * Ensure a mutable transfer store is exposed globally as window.icicles_transfer.
   * This store is intended to temporarily hold term tree data being transferred
   * between icicles. It provides basic utilities to add/remove terms by id.
   *
   * The global reference is write-protected (non-writable) to avoid replacement,
   * but the object itself and its internal state are mutable.
   */
  function ensureTransferStore() {
    if (window.icicles_transfer) {
      return;
    }

    // Local forest state inside a closure.
    let forest = [];

    // Utilities to manipulate term forests: { id:number, name:string, children:Array }.
    function cloneTreeNode(node) {
      return {
        id: Number(node && node.id != null ? node.id : 0),
        name: String(node && node.name != null ? node.name : ''),
        children: Array.isArray(node && node.children)
          ? node.children.map(cloneTreeNode)
          : [],
      };
    }

    function toForest(treeOrForest) {
      if (!treeOrForest) return [];
      if (Array.isArray(treeOrForest)) return treeOrForest.map(cloneTreeNode);
      if (typeof treeOrForest === 'object')
        return [cloneTreeNode(treeOrForest)];
      return [];
    }

    function indexById(nodes) {
      const idx = new Map();
      (nodes || []).forEach(function (n) {
        idx.set(Number(n.id), n);
      });
      return idx;
    }

    function mergeForests(baseForest, addForest) {
      const baseIndex = indexById(baseForest);
      (addForest || []).forEach(function (addNode) {
        const id = Number(addNode.id);
        const existing = baseIndex.get(id);
        if (existing) {
          if (addNode.name) existing.name = String(addNode.name);
          if (!Array.isArray(existing.children)) existing.children = [];
          const addChildren = Array.isArray(addNode.children)
            ? addNode.children
            : [];
          mergeForests(existing.children, addChildren);
        } else {
          baseForest.push(cloneTreeNode(addNode));
        }
      });
      return baseForest;
    }

    function removeByIds(baseForest, removeForest) {
      const ids = new Set();
      (function collect(nodes) {
        (nodes || []).forEach(function (n) {
          ids.add(Number(n.id));
          if (Array.isArray(n.children)) collect(n.children);
        });
      })(removeForest || []);

      function filter(nodes) {
        return (nodes || [])
          .filter(function (n) {
            return !ids.has(Number(n.id));
          })
          .map(function (n) {
            return {
              id: Number(n.id),
              name: String(n.name || ''),
              children: filter(Array.isArray(n.children) ? n.children : []),
            };
          });
      }

      return filter(baseForest || []);
    }

    // Tracks the id of the icicle that most recently added terms into the
    // transfer buffer. This is optional metadata and may be null.
    let sourceId = null;
    // Tracks the default icicle id (string) to receive orphaned terms.
    let defaultIcicleId = null;
    // Pending orphan queues keyed by a stable icicle key (e.g., group UUID).
    // Each value is a forest array to be transferred to default if the icicle
    // does not reappear after an AJAX refresh.
    const pendingOrphans = new Map();

    const api = {
      // Returns a deep copy of the current transfer forest.
      getTerms() {
        return toForest(forest);
      },
      // Returns a deep copy of the current transfer forest and clears the buffer.
      // Does not modify the remembered sourceId.
      takeTerms() {
        const out = toForest(forest);
        forest = [];
        return out;
      },
      // Deep-merge provided tree/forest into the transfer forest.
      // Optionally pass the id of the source icicle as the second argument.
      addTerms(tree, fromIcicleId) {
        const add = toForest(tree);
        mergeForests(forest, add);
        if (typeof fromIcicleId !== 'undefined') {
          sourceId = fromIcicleId == null ? null : String(fromIcicleId);
        }
        return this.getTerms();
      },
      // Remove any nodes (by id, including descendants) present in given tree/forest.
      removeTerms(tree) {
        const rem = toForest(tree);
        forest = removeByIds(forest, rem);
        return this.getTerms();
      },
      // Clear the transfer buffer (does not alter the remembered source id).
      clear() {
        forest = [];
      },
      // Queue orphan terms for a specific stable icicle key.
      queuePendingOrphans(key, tree) {
        try {
          if (!key) return false;
          const k = String(key);
          const existing = pendingOrphans.get(k) || [];
          const add = toForest(tree);
          const merged = mergeForests(existing.slice(), add);
          pendingOrphans.set(k, merged);
          if (typeof console !== 'undefined' && console.info) {
            try {
              debugLog('init', 'Queued pending orphans for key', k, {
                count: (function cn(a) {
                  let c = 0;
                  (a || []).forEach(function (n) {
                    c++;
                    if (Array.isArray(n.children)) c += cn(n.children);
                  });
                  return c;
                })(add),
              });
            } catch (e) {}
          }
          return true;
        } catch (e) {
          return false;
        }
      },
      // Finalize any pending orphans: if an icicle with the same key exists on
      // the page after re-attach, discard its queued orphans (it was not removed);
      // otherwise, transfer its queued terms into the default icicle.
      finalizePendingOrphans() {
        try {
          if (!pendingOrphans || pendingOrphans.size === 0) return;
          const registry =
            Drupal && Drupal.taxonomyTermIcicles
              ? Drupal.taxonomyTermIcicles
              : {};
          // Build set of keys currently present in registry instances.
          const presentKeys = new Set();
          try {
            Object.keys(registry || {}).forEach(function (id) {
              const inst = registry[id];
              const key = inst && inst.key ? String(inst.key) : '';
              if (key) presentKeys.add(key);
            });
          } catch (e) {}
          // Determine default destination instance.
          const defaultId =
            typeof this.getDefaultIcicleId === 'function'
              ? this.getDefaultIcicleId()
              : null;
          const destInst =
            defaultId && registry && registry[defaultId]
              ? registry[defaultId]
              : null;
          // Iterate pending keys and either discard or transfer.
          Array.from(pendingOrphans.keys()).forEach(function (k) {
            const forestForKey = pendingOrphans.get(k) || [];
            if (presentKeys.has(k)) {
              // Icicle still exists; discard queued orphans.
              if (typeof console !== 'undefined' && console.info) {
                try {
                  debugLog(
                    'init',
                    'Discarding pending orphans for key (icicle still present):',
                    k,
                  );
                } catch (e) {}
              }
              pendingOrphans.delete(k);
              return;
            }
            // No icicle present for this key: transfer to default if possible.
            if (
              destInst &&
              typeof destInst.addTerms === 'function' &&
              Array.isArray(forestForKey) &&
              forestForKey.length
            ) {
              try {
                if (typeof console !== 'undefined' && console.info) {
                  try {
                    debugLog(
                      'init',
                      'Finalizing pending orphans for key',
                      k,
                      '→ default icicle id',
                      String(defaultId),
                    );
                  } catch (e) {}
                }
                destInst.addTerms(forestForKey);
              } catch (e) {}
            } else if (typeof console !== 'undefined' && console.warn) {
              console.warn(
                '[icicles_manager] No valid default icicle to receive pending orphans for key:',
                k,
              );
            }
            pendingOrphans.delete(k);
          });
        } catch (e) {}
      },
      // Getter/Setter for the last source icicle id.
      getSourceId() {
        return sourceId;
      },
      setSourceId(id) {
        sourceId = id == null ? null : String(id);
        return sourceId;
      },
      // Getter/Setter for the default icicle id.
      getDefaultIcicleId() {
        return defaultIcicleId;
      },
      setDefaultIcicleId(id) {
        defaultIcicleId = id == null ? null : String(id);
        return defaultIcicleId;
      },
    };

    setGlobalOnce('icicles_transfer', api);
  }

  /**
   * Bind once-only interaction handlers to coordinate icicle term transfers.
   * - On termsIcicle:nodeHoldStart: remove subtree from source icicle and add to transfer buffer with source id.
   * - On mouseup: return any buffered terms back to the source icicle, then clear buffer and reset source id.
   */
  function ensureInteractionHandlers() {
    if (window.__iciclesInteractionHandlersBound) {
      return;
    }
    window.__iciclesInteractionHandlersBound = true;

    // Utility to count total nodes in a forest (includes all descendants).
    function countNodes(nodes) {
      let c = 0;
      (function walk(arr) {
        (arr || []).forEach(function (n) {
          c++;
          if (Array.isArray(n && n.children)) walk(n.children);
        });
      })(nodes || []);
      return c;
    }

    // Simple floating "ghost" element that follows the mouse while transferring.
    let ghostEl = null;
    let mouseMoveBound = false;
    function ensureGhost() {
      if (ghostEl) return;
      ghostEl = document.createElement('div');
      ghostEl.className = 'terms-icicle__ghost';
      ghostEl.setAttribute('aria-hidden', 'true');
      ghostEl.style.position = 'fixed';
      ghostEl.style.pointerEvents = 'none';
      ghostEl.style.zIndex = '2147483647';
      ghostEl.style.left = '0px';
      ghostEl.style.top = '0px';
      document.body.appendChild(ghostEl);
    }
    function updateGhostContent(forest) {
      if (!ghostEl) return;
      const count = countNodes(forest || []);
      const names = [];
      (forest || []).forEach(function (n) {
        if (n && (n.name || n.id)) names.push(String(n.name || n.id));
      });
      const title = names.length ? names.join(', ') : '';
      ghostEl.innerHTML = `<strong>${count}</strong> term${
        count === 1 ? '' : 's'
      }${title ? ` — ${title}` : ''}`;
    }
    function onMouseMove(ev) {
      if (!ghostEl) return;
      const x = ev && typeof ev.clientX === 'number' ? ev.clientX : 0;
      const y = ev && typeof ev.clientY === 'number' ? ev.clientY : 0;
      // Offset so the cursor is not obscured.
      ghostEl.style.left = `${x + 12}px`;
      ghostEl.style.top = `${y + 20}px`;
    }
    function bindMouseMove() {
      if (mouseMoveBound) return;
      document.addEventListener('mousemove', onMouseMove, true);
      mouseMoveBound = true;
    }
    function removeGhost() {
      if (mouseMoveBound) {
        document.removeEventListener('mousemove', onMouseMove, true);
        mouseMoveBound = false;
      }
      if (ghostEl && ghostEl.parentNode) {
        ghostEl.parentNode.removeChild(ghostEl);
      }
      ghostEl = null;
    }

    // When a node hold starts, move that subtree into the transfer buffer and remove from the source icicle.
    document.addEventListener(
      'termsIcicle:nodeHoldStart',
      function (e) {
        try {
          const detail = e && e.detail ? e.detail : null;
          if (!detail) return;
          const icicleId = detail.icicleId;
          const tree = detail.tree;
          if (!icicleId || !tree || !Array.isArray(tree)) return;
          const registry =
            Drupal && Drupal.taxonomyTermIcicles
              ? Drupal.taxonomyTermIcicles
              : {};
          const inst = registry[icicleId];
          if (!inst || typeof inst.removeTerms !== 'function') return;

          const num = countNodes(tree);
          if (typeof console !== 'undefined' && console.info) {
            try {
              debugLog(
                'interactions',
                'transfer start (mousedown): moving',
                num,
                'term(s) from icicle',
                String(icicleId),
                'to icicles_transfer',
              );
            } catch (logErr) {}
          }

          // Remove from the source icicle first.
          inst.removeTerms(tree);
          // Add to the transfer buffer with source id.
          if (
            window.icicles_transfer &&
            typeof window.icicles_transfer.addTerms === 'function'
          ) {
            window.icicles_transfer.addTerms(tree, icicleId);
          }
          // Show and update ghost to follow the mouse.
          ensureGhost();
          updateGhostContent(
            window.icicles_transfer &&
              typeof window.icicles_transfer.getTerms === 'function'
              ? window.icicles_transfer.getTerms()
              : tree,
          );
          bindMouseMove();
        } catch (err) {
          if (typeof console !== 'undefined' && console.error) {
            console.error(
              '[icicles_manager] Error handling nodeHoldStart:',
              err,
            );
          }
        }
      },
      false,
    );

    // On mouseup anywhere, determine drop target and transfer buffered terms accordingly.
    document.addEventListener(
      'mouseup',
      function (event) {
        try {
          const transfer = window.icicles_transfer;
          if (!transfer) return;
          const srcId =
            typeof transfer.getSourceId === 'function'
              ? transfer.getSourceId()
              : null;
          const terms =
            typeof transfer.takeTerms === 'function'
              ? transfer.takeTerms()
              : [];
          const num = countNodes(terms);

          if (!Array.isArray(terms) || !terms.length) {
            if (typeof console !== 'undefined' && console.info) {
              try {
                debugLog(
                  'interactions',
                  'transfer end (mouseup): no terms to transfer',
                  { sourceId: srcId, count: num },
                );
              } catch (logErr) {}
            }
            // Still clear and reset below.
          } else {
            // Detect the icicle under the mouse pointer (also matches the empty placeholder inside it).
            const clientX =
              event && typeof event.clientX === 'number' ? event.clientX : null;
            const clientY =
              event && typeof event.clientY === 'number' ? event.clientY : null;
            let hoveredIcicleId = null;
            if (
              clientX !== null &&
              clientY !== null &&
              document.elementFromPoint
            ) {
              const el = document.elementFromPoint(clientX, clientY);
              if (el && typeof el.closest === 'function') {
                const wrapper = el.closest('.terms-icicle');
                if (wrapper && wrapper.id) {
                  hoveredIcicleId = String(wrapper.id);
                }
              }
            }

            const registry =
              Drupal && Drupal.taxonomyTermIcicles
                ? Drupal.taxonomyTermIcicles
                : {};
            const destId = hoveredIcicleId || srcId; // Fallback to source if not hovering any icicle.

            if (destId) {
              if (typeof console !== 'undefined' && console.info) {
                try {
                  debugLog(
                    'interactions',
                    'transfer end (mouseup): moving',
                    num,
                    'term(s) from icicles_transfer to icicle',
                    String(destId),
                    hoveredIcicleId ? '(hover target)' : '(source fallback)',
                  );
                } catch (logErr) {}
              }
              const inst = registry[destId];
              if (inst && typeof inst.addTerms === 'function') {
                inst.addTerms(terms);
              } else if (typeof console !== 'undefined' && console.warn) {
                console.warn(
                  '[icicles_manager] Drop target icicle instance not found for id:',
                  destId,
                );
              }
            } else if (typeof console !== 'undefined' && console.info) {
              try {
                debugLog(
                  'interactions',
                  'transfer end (mouseup): could not resolve destination icicle; discarding transfer terms',
                  { count: num },
                );
              } catch (logErr) {}
            }
          }

          // Ensure buffer is empty and source id cleared regardless.
          if (typeof transfer.clear === 'function') transfer.clear();
          if (typeof transfer.setSourceId === 'function')
            transfer.setSourceId(null);
          // Always remove the global dragging class on mouseup (safety net).
          if (document && document.body && document.body.classList) {
            document.body.classList.remove('terms-icicle--dragging');
          }
          // Remove the ghost now that transfer is complete/cancelled.
          removeGhost();
        } catch (err) {
          if (typeof console !== 'undefined' && console.error) {
            console.error('[icicles_manager] Error handling mouseup:', err);
          }
          // Safety: ensure ghost removed on errors as well.
          removeGhost();
        }
      },
      false,
    );
  }

  // Build a rank map (plain object id => rank) using preorder traversal to preserve
  // the original sibling order from the immutable reference tree.
  function buildImmutableRankObject(referenceForest) {
    const rank = Object.create(null);
    let i = 0;
    (function walk(nodes) {
      (nodes || []).forEach(function (n) {
        const idNum = Number(n && n.id);
        if (!Object.prototype.hasOwnProperty.call(rank, idNum)) {
          rank[idNum] = i++;
        }
        if (Array.isArray(n && n.children)) walk(n.children);
      });
    })(referenceForest || []);
    return rank;
  }

  // Build an immutable color map (id => color string) based on the immutable tree.
  // Coloring strategy:
  // - Default: assign distinct colors to top-level nodes (depth 1 in the forest) and
  //   propagate that color to their descendants.
  // - If the forest has exactly one top-level node, assign neutral color to that top
  //   node and assign distinct colors to its children (depth 2), propagating to their
  //   descendants. This mirrors the special-case used in the renderer previously.
  function buildImmutableColorsMap(referenceForest) {
    // Normalize to forest (array of nodes)
    let forest = [];
    if (Array.isArray(referenceForest)) {
      forest = referenceForest;
    } else if (referenceForest) {
      forest = [referenceForest];
    }

    function countTop(nodes) {
      return (Array.isArray(nodes) ? nodes.length : 0) || 0;
    }
    const hasSingleTop = countTop(forest) === 1;

    // Determine categories set at the coloring level.
    let categories = [];
    if (!hasSingleTop) {
      categories = forest.slice();
    } else {
      const onlyTop = forest[0] || null;
      categories =
        onlyTop && Array.isArray(onlyTop.children)
          ? onlyTop.children.slice()
          : [];
    }

    // Build a palette of distinct, visually spaced HSL colors.
    const n = Math.max(1, categories.length);
    function hsl(i, total) {
      const hue = (i * (360 / Math.max(1, total))) % 360;
      const sat = 65; // percent
      const light = 55; // percent
      return `hsl(${hue}, ${sat}%, ${light}%)`;
    }

    const categoryColorById = Object.create(null);
    for (let c = 0; c < categories.length; c++) {
      const cat = categories[c];
      if (!cat) continue;
      const id = Number(cat.id);
      categoryColorById[id] = hsl(c, n);
    }

    const NEUTRAL = '#ccc';
    const colors = Object.create(null);

    // Walk the forest, assigning colors based on the strategy above.
    function walk(nodes, depth, inheritedColor) {
      (nodes || []).forEach(function (node) {
        const idNum = Number(node && node.id);
        let ownColor = inheritedColor;
        if (hasSingleTop) {
          // Single top: depth 1 = neutral; depth 2 = category colors; deeper inherit.
          if (depth === 1) {
            ownColor = NEUTRAL;
          } else if (depth === 2) {
            ownColor = categoryColorById[idNum] || NEUTRAL;
          }
        } else if (depth === 1) {
          // Depth 1 nodes get their category color; descendants inherit.
          ownColor = categoryColorById[idNum] || NEUTRAL;
        }
        colors[idNum] = ownColor || NEUTRAL;
        if (Array.isArray(node && node.children)) {
          walk(node.children, depth + 1, colors[idNum]);
        }
      });
    }

    walk(forest, 1, NEUTRAL);

    return colors;
  }
})(Drupal, drupalSettings);
