<?php

namespace Drupal\taxonomy_term_config_groups\Entity;

use Drupal\Component\Transliteration\PhpTransliteration;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;

/**
 * Defines the Taxonomy Group content entity.
 *
 * @ContentEntityType(
 *   id = "taxonomy_group",
 *   label = @Translation("Taxonomy group"),
 *   label_collection = @Translation("Taxonomy groups"),
 *   bundle_label = @Translation("Taxonomy group type"),
 *   handlers = {
 *     "view_builder" = "Drupal\\Core\\Entity\\EntityViewBuilder",
 *     "views_data" = "Drupal\\views\\EntityViewsData",
 *     "access" = "Drupal\\Core\\Entity\\EntityAccessControlHandler"
 *   },
 *   base_table = "taxonomy_group",
 *   translatable = FALSE,
 *   admin_permission = "administer taxonomy",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "bundle" = "type",
 *     "label" = "label"
 *   },
 *   bundle_entity_type = "taxonomy_group_type",
 *   field_ui_base_route = "entity.taxonomy_group_type.edit_form",
 * )
 */
class TaxonomyGroup extends ContentEntityBase {

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('ID'))
      ->setReadOnly(TRUE);

    $fields['uuid'] = BaseFieldDefinition::create('uuid')
      ->setLabel(t('UUID'))
      ->setReadOnly(TRUE);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Label'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 255,
      ]);

    // Machine name automatically generated from the label for new entities.
    $fields['machine_name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Machine name'))
      ->setDescription(t('Machine-readable name generated from the label.'))
      ->setRequired(FALSE)
      ->setSettings([
        'max_length' => 128,
        'is_ascii' => TRUE,
        'case_sensitive' => FALSE,
      ])
      // Not editable via default forms; managed automatically.
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'));

    // Base field to store taxonomy terms for every group across all bundles.
    $fields['field_terms'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Terms'))
      ->setDescription(t('Taxonomy terms that belong to this group.'))
      ->setRequired(FALSE)
      ->setTranslatable(FALSE)
      ->setRevisionable(FALSE)
      ->setCardinality(-1)
      ->setSetting('target_type', 'taxonomy_term')
      // Do not expose in Field UI or default forms/views.
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    return $fields;
  }

  /**
   * Generate a machine name from a human label.
   *
   * Uses Drupal's transliteration component and standard cleanup rules:
   * - Transliterate to ASCII using underscores as word separators.
   * - Lowercase all characters.
   * - Replace any remaining disallowed characters with single underscores.
   * - Trim leading/trailing underscores and collapse repeats.
   */
  protected static function generateMachineName(string $label): string {
    $label = trim($label);
    if ($label === '') {
      return '';
    }
    $trans = new PhpTransliteration();
    $text = $trans->transliterate($label, 'en', '_');
    $text = mb_strtolower($text);
    // Allow only a-z, 0-9 and underscore.
    $text = preg_replace('/[^a-z0-9_]+/', '_', $text) ?? '';
    // Collapse multiple underscores and trim.
    $text = preg_replace('/_+/', '_', $text) ?? '';
    $text = trim($text, '_');
    // Fallback if becomes empty.
    return $text !== '' ? $text : 'group';
  }

  /**
   * Ensure machine_name is set on creation and unique within bundle.
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    // Ensure the label does not exceed the storage limit (255 chars).
    $label_value = (string) $this->get('label')->value;
    // Use mb_* if available to be multibyte-safe.
    $length_fn = function_exists('mb_strlen') ? 'mb_strlen' : 'strlen';
    $substr_fn = function_exists('mb_substr') ? 'mb_substr' : 'substr';
    if ($length_fn($label_value) > 255) {
      $this->set('label', (string) $substr_fn($label_value, 0, 255));
    }

    // Only generate on create or when currently empty.
    $machine = $this->get('machine_name')->getString();
    if ($this->isNew() && $machine === '') {
      $label = (string) $this->label();
      $base = static::generateMachineName($label);

      // Enforce uniqueness within the same bundle by appending a numeric suffix.
      $i = 1;
      $bundle = $this->bundle();
      $max_len = 128;

      // Helper to truncate keeping suffix space.
      $makeCandidate = static function (string $base, int $suffix) use ($max_len): string {
        $suffix_str = $suffix > 1 ? '_' . $suffix : '';
        $max_base = $max_len - strlen($suffix_str);
        if ($max_base < 1) {
          $max_base = 1;
        }
        $truncated = substr($base, 0, $max_base);
        return rtrim($truncated, '_') . $suffix_str;
      };

      // Start with a truncated base (no suffix) to respect max length.
      $candidate = $makeCandidate($base, 1);

      // Iterate until no collision.
      while (TRUE) {
        // Load by properties avoids access checks and is sufficient for uniqueness.
        $existing = $storage->loadByProperties([
          'type' => $bundle,
          'machine_name' => $candidate,
        ]);
        if (empty($existing)) {
          break;
        }
        $i++;
        $candidate = $makeCandidate($base, $i);
      }

      $this->set('machine_name', $candidate);
    }
  }

}
