<?php

namespace Drupal\taxonomy_term_config_groups\Form;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\taxonomy_term_config_groups\Entity\TaxonomyGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for configuring grouping of taxonomy terms within a vocabulary.
 */
class TaxonomyTermGroupingForm extends FormBase {

  /**
   * The current route match service.
   */
  protected RouteMatchInterface $currentRouteMatch;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The UUID service.
   */
  protected UuidInterface $uuid;

  /**
   * Build a nested tree of terms for a vocabulary in linear time.
   *
   * @return array<int, array{id:int,name:string,children:array<int, array<string, mixed>>}>
   */
  protected function buildTermsTree(string $vid): array {
    $items = $this->entityTypeManager->getStorage('taxonomy_term')->loadTree($vid, 0, NULL, FALSE);
    // First pass: create node map keyed by tid.
    $nodes = [];
    $parents = [];
    foreach ($items as $it) {
      $tid = (int) ($it->tid ?? 0);
      if ($tid <= 0) { continue; }
      $nodes[$tid] = ['id' => $tid, 'name' => (string) ($it->name ?? ''), 'children' => []];
      $p = 0;
      if (isset($it->parents) && is_array($it->parents)) {
        $first = reset($it->parents);
        if (is_scalar($first)) { $p = (int) $first; }
      }
      $parents[$tid] = $p;
    }
    // Second pass: link children to parents.
    $roots = [];
    foreach ($nodes as $tid => &$node) {
      $p = $parents[$tid] ?? 0;
      if ($p === 0 || !isset($nodes[$p])) {
        $roots[] = &$node;
      }
      else {
        $nodes[$p]['children'][] = &$node;
      }
    }
    // Return a copy without references.
    return array_map(static function ($n) { return $n; }, $roots);
  }

  /**
   * Collect unique TIDs assigned in taxonomy_group entities.
   *
   * @param array<int, \Drupal\taxonomy_term_config_groups\Entity\TaxonomyGroup> $entities
   * @return list<int>
   */
  protected function collectAssignedTids(array $entities): array {
    $tids = [];
    foreach ($entities as $entity) {
      if ($entity->hasField('field_terms')) {
        foreach ((array) $entity->get('field_terms')->getValue() as $item) {
          if (is_array($item) && isset($item['target_id']) && is_scalar($item['target_id'])) {
            $tid = (int) $item['target_id'];
            if ($tid > 0) { $tids[$tid] = TRUE; }
          }
        }
      }
    }
    $out = array_map('intval', array_keys($tids));
    sort($out, SORT_NUMERIC);
    return $out;
  }

  /**
   * Build the default left icicle forest from all ids minus assigned ids.
   *
   * @param list<int> $all
   * @param list<int> $assigned
   * @return list<array{id:int,name:string,children:list<array<string, mixed>>}>
   */
  protected function buildLeftIcicleDefault(array $all, array $assigned): array {
    $assigned_map = array_fill_keys(array_map('intval', $assigned), TRUE);
    $forest = [];
    foreach ($all as $tid) {
      $tid = (int) $tid;
      if ($tid > 0 && !isset($assigned_map[$tid])) {
        $forest[] = ['id' => $tid, 'name' => '', 'children' => []];
      }
    }
    return $forest;
  }

  /**
   * Load existing groups for a bundle, keyed by UUID.
   *
   * @return array<string, \Drupal\taxonomy_term_config_groups\Entity\TaxonomyGroup>
   */
  protected function loadExistingGroups(string $bundle_id): array {
    $storage = $this->entityTypeManager->getStorage('taxonomy_group');
    $entities = $storage->loadByProperties(['type' => $bundle_id]);
    /** @var array<string, TaxonomyGroup> $by */
    $by = [];
    foreach ($entities as $entity) {
      if ($entity instanceof TaxonomyGroup) {
        $by[(string) $entity->uuid()] = $entity;
      }
    }
    return $by;
  }

  /**
   * Normalize groups array to be keyed by UUID without mutating IDs.
   *
   * @param array<string|int, array<string,mixed>> $groups
   * @return array<string, array<string,mixed>>
   */
  protected function normalizeGroups(array $groups): array {
    $normalized = [];
    foreach ($groups as $key => $group) {
      $uuid = is_string($key) && $key !== '' ? (string) $key : $this->uuid->generate();
      $normalized[$uuid] = $group;
    }
    return $normalized;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = new static();
    $instance->setStringTranslation($container->get('string_translation'));
    $instance->currentRouteMatch = $container->get('current_route_match');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->uuid = $container->get('uuid');
    /** @var LoggerChannelFactoryInterface $loggerFactory */
    $loggerFactory = $container->get('logger.factory');
    $instance->setLoggerFactory($loggerFactory);
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'taxonomy_term_config_groups_grouping_form';
  }

  /**
   * Transform a flat list of terms into a nested tree.
   *
   * @param array<int|string, mixed> $source
   *   Flat list of terms as arrays or objects (loadTree results). Each item may
   *   contain keys/properties: id|tid (int), name (string), parents (int[]).
   * @param array<int, array{id:int,name:string,children:array<int, array<string, mixed>>}> $results
   *   Accumulator for built results (used during recursion).
   * @param array<int, mixed> $index
   *   Internal index mapping parent id to the result node.
   *
   * @return array<int, array<string, mixed>>
   *   Nested tree array of top-level nodes.
   */
  protected function unflattenTerms(array $source, array &$results = [], array &$index = []): array {
    /** @var array<int, array{id:int,name:string,children:array<int, array<string, mixed>>}> $results */
    /** @var array<int, array{id:int,name:string,children:array<int, array<string, mixed>>}> $index */
    $next = [];
    $removedThisPass = 0;

    foreach ($source as $item) {
      $id = 0;
      $name = '';
      $parent = 0;

      if (is_array($item)) {
        $rawId = $item['id'] ?? ($item['tid'] ?? 0);
        if (is_scalar($rawId)) {
          $id = (int) $rawId;
        }
        $rawName = $item['name'] ?? '';
        $name = is_scalar($rawName) ? (string) $rawName : '';
        $parents = isset($item['parents']) && is_array($item['parents']) ? $item['parents'] : [];
        $first = reset($parents);
        $parent = is_scalar($first) ? (int) $first : 0;
      }
      elseif (is_object($item)) {
        // Prefer ->id or ->tid, and ->parents[0].
        $rawId = $item->id ?? ($item->tid ?? 0);
        if (is_scalar($rawId)) {
          $id = (int) $rawId;
        }
        $rawName = $item->name ?? '';
        $name = is_scalar($rawName) ? (string) $rawName : '';

        // The direct parent id is the first element.
        $parents = isset($item->parents) && is_array($item->parents) ? $item->parents : [];
        $first = reset($parents);
        $parent = is_scalar($first) ? (int) $first : 0;
      }

      if ($id <= 0) {
        // Skip invalid items; they are considered removed.
        $removedThisPass++;
        continue;
      }

      /** @var array{id:int,name:string,children:array<int, array<string, mixed>>} $node */
      $node = [
        'id' => $id,
        'name' => $name,
        'children' => [],
      ];

      if ($parent === 0) {
        // Root term: add to results and index.
        $results[] = $node;
        $lastKey = array_key_last($results);
        $index[$id] = &$results[$lastKey];
        $removedThisPass++;
        continue;
      }

      if (isset($index[$parent])) {
        // Parent exists somewhere in results; append as child.
        if (!isset($index[$parent]['children']) || !is_array($index[$parent]['children'])) {
          $index[$parent]['children'] = [];
        }
        $children = &$index[$parent]['children'];
        $children[] = $node;
        $childKey = array_key_last($children);
        $index[$id] = &$children[$childKey];
        $removedThisPass++;
        continue;
      }

      // Parent not yet present; keep for the next recursive pass.
      $next[] = $item;
    }

    if (!empty($next)) {
      if ($removedThisPass === 0) {
        // No progress this pass => likely missing or cyclic parents.
        $unresolved = array_map(static function ($t) {
          if (is_array($t)) {
            $val = $t['id'] ?? ($t['tid'] ?? '?');
            return is_scalar($val) ? (string) $val : '?';
          }
          if (is_object($t)) {
            $val = $t->id ?? ($t->tid ?? '?');
            return is_scalar($val) ? (string) $val : '?';
          }
          return '?';
        }, $next);
        throw new \RuntimeException('Unable to build taxonomy term tree due to unresolved parent references for term IDs: ' . implode(', ', $unresolved));
      }
      // Recurse with the remaining items.
      return $this->unflattenTerms($next, $results, $index);
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   *
   * @return array<string, mixed>
   *   The built form render array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Get vocabulary and derived bundle id.
    $vocabulary = $this->getVocabularyFromRoute();
    $vid = $vocabulary ? $vocabulary->id() : NULL;
    $vocab_label = $vocabulary ? $vocabulary->label() : '';
    $bundle_id = $vid ? $this->getBundleId((string) $vid) : NULL;

    // Build a nested tree structure for the terms: { name, children: [...] }.
    $terms_tree = [];
    /** @var array<int, true> $all_term_ids */
    $all_term_ids = [];
    if ($vid) {
      try {
        $terms_tree = $this->buildTermsTree((string) $vid);
        // Collect flat list of all term IDs.
        foreach ($terms_tree as $root) {
          $stack = [$root];
          while ($stack) {
            $n = array_pop($stack);
            if (isset($n['id']) && is_scalar($n['id'])) { $all_term_ids[(int) $n['id']] = TRUE; }
            if (!empty($n['children']) && is_array($n['children'])) {
              foreach ($n['children'] as $c) { if (is_array($c)) { $stack[] = $c; } }
            }
          }
        }
      }
      catch (\Throwable $e) {
        $terms_tree = ['error' => $e->getMessage()];
      }
    }

    $form['#tree'] = TRUE;

    // Set a contextual page title including the vocabulary name.
    $form['#title'] = $vocab_label ? $this->t('Term grouping: %label', ['%label' => $vocab_label]) : $this->t('Term grouping');

    // Instructions: how to use the grouping UI.
    $form['instructions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--info']],
      '#markup' => $this->t('Use the "Unassigned terms" tree on the left to drag terms into "Groups" on the right. To remove a term from a group, drag it back to Unassigned. Dragging a parent term moves all of its children with it.'),
    ];

    // Attach assets and settings.
    if (!isset($form['#attached']) || !is_array($form['#attached'])) { $form['#attached'] = []; }
    if (!isset($form['#attached']['drupalSettings']) || !is_array($form['#attached']['drupalSettings'])) { $form['#attached']['drupalSettings'] = []; }
    if (!isset($form['#attached']['library']) || !is_array($form['#attached']['library'])) { $form['#attached']['library'] = []; }
    $form['#attached']['drupalSettings']['taxonomyTermConfigGroups'] = [
      'vocabulary' => [
        'id' => $vid ? (string) $vid : '',
        'label' => (string) $vocab_label,
      ],
      'tree' => $terms_tree,
    ];
    $form['#attached']['library'][] = 'taxonomy_term_config_groups/grouping_form';

    // Two-column layout wrapper with an id so AJAX can replace both sides.
    $form['columns'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'taxonomy-term-config-groups__columns',
        'class' => ['taxonomy-term-config-groups__columns'],
      ],
    ];

    // Left column with the full icicle of all terms.
    $form['columns']['left'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['taxonomy-term-config-groups__col', 'taxonomy-term-config-groups__col--left'],
      ],
    ];
    // Section header to clarify purpose of the left column.
    $form['columns']['left']['header'] = [
      '#type' => 'html_tag',
      '#tag' => 'h3',
      '#value' => $this->t('Unassigned terms'),
      '#attributes' => [
        'class' => ['taxonomy-term-config-groups__section-title', 'taxonomy-term-config-groups__section-title--left'],
      ],
    ];
    $form['columns']['left']['terms_icicle'] = [
      '#type' => 'terms_icicle',
      '#init_full' => TRUE,
      '#is_default' => TRUE,
    ];

    // Right column: groups wrapper only (no controls yet).
    $form['columns']['right'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['taxonomy-term-config-groups__col', 'taxonomy-term-config-groups__col--right'],
      ],
    ];
    // Section header to clarify purpose of the right column.
    $form['columns']['right']['header'] = [
      '#type' => 'html_tag',
      '#tag' => 'h3',
      '#value' => $this->t('Groups'),
      '#attributes' => [
        'class' => ['taxonomy-term-config-groups__section-title', 'taxonomy-term-config-groups__section-title--right'],
      ],
    ];

    // Initialize groups on first build and normalize the in-form structure.
    $groups = $this->initializeGroupsOnFirstBuild($form_state, $vid ? (string) $vid : NULL, $bundle_id ?: NULL);
    $groups = $this->normalizeGroups($groups);
    $form_state->set('groups', $groups);

    // Compute and assign the left icicle default to exclude terms already assigned.
    $assigned_raw = $form_state->get('assigned_tids');
    $assigned_from_state = [];
    if (is_array($assigned_raw)) {
      foreach ($assigned_raw as $v) {
        if (is_int($v) || (is_string($v) && ctype_digit($v))) {
          $assigned_from_state[] = (int) $v;
        }
      }
    }
    $left_forest = $this->buildLeftIcicleDefault(
      array_keys($all_term_ids),
      $assigned_from_state
    );
    // Set the left icicle defaults and disable init_full so the default is respected.
    $form['columns']['left']['terms_icicle']['#default_value'] = $left_forest;
    $form['columns']['left']['terms_icicle']['#init_full'] = FALSE;

    $form['columns']['right']['groups_wrapper'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'taxonomy-term-config-groups__groups-wrapper',
        'class' => ['taxonomy-term-config-groups__groups-wrapper'],
      ],
    ];

    if (count($groups) === 0) {
      // Show an informational message box when there are no groups yet.
      $form['columns']['right']['groups_wrapper']['placeholder'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['messages', 'messages--status', 'taxonomy-term-config-groups__empty-message'],
        ],
        'content' => [
          '#markup' => $this->t('No groupings yet. Create a new group to get started.'),
        ],
      ];
    }
    else {
      $form['columns']['right']['groups_wrapper']['groups'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['taxonomy-term-config-groups__groups']],
      ];
      /** @var array<string, array<string, mixed>> $groups */
      foreach ($groups as $uuid => $group) {
        if ($uuid === '') {
          continue;
        }
        $name_value = '';
        if (isset($group['name']) && is_scalar($group['name']) && $group['name'] !== '') {
          $name_value = (string) $group['name'];
        }
        elseif (isset($group['title']) && is_scalar($group['title'])) {
          $name_value = (string) $group['title'];
        }
        $this->buildGroupFieldset($form, $form_state, $uuid, $group, (string) ($bundle_id ?? ''));
      }
    }

    // Add grouping button (AJAX) – adds an empty group to form_state and rebuilds.
    if (!isset($form['columns']) || !is_array($form['columns'])) { $form['columns'] = []; }
    if (!isset($form['columns']['right']) || !is_array($form['columns']['right'])) { $form['columns']['right'] = []; }
    if (!isset($form['columns']['right']['groups_wrapper']) || !is_array($form['columns']['right']['groups_wrapper'])) { $form['columns']['right']['groups_wrapper'] = []; }
    $groups_wrapper = &$form['columns']['right']['groups_wrapper'];
    /** @var array<string,mixed> $groups_wrapper */
    $groups_wrapper['add_group'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add grouping'),
      '#submit' => ['::addGroupSubmit'],
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::groupsAjaxCallback',
        'wrapper' => 'taxonomy-term-config-groups__columns',
      ],
      '#attributes' => ['class' => ['button--add-grouping']],
    ];

    // Actions: primary submit to save group definitions.
    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save groupings'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * Ajax callback to rebuild only the groups wrapper.
   *
   * @param array<string, mixed> $form
   *   The full form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The groups wrapper render array.
   */
  public function groupsAjaxCallback(array &$form, FormStateInterface $form_state): array {
    // Rebuild the entire columns container so the left icicle reflects current selections.
    $columns = isset($form['columns']) && is_array($form['columns']) ? $form['columns'] : NULL;
    return is_array($columns) ? $columns : $form;
  }

  /**
   * Submit handler for adding a new empty grouping to form_state.
   */
  /**
   * Submit handler for adding a new empty grouping to form_state.
   *
   * @param array<string, mixed> $form
   *   The form render array.
   */
  public function addGroupSubmit(array &$form, FormStateInterface $form_state): void {
    $groups = $form_state->get('groups');
    if (!is_array($groups)) {
      $groups = [];
    }
    // Create a minimal empty group keyed by a UUID for stability. Do not set
    // the group's 'id' value; it is a separate business field which may be blank.
    $uuid = $this->uuid->generate();
    $groups[(string) $uuid] = [
      'name' => '',
      'terms_icicle' => [],
    ];
    $form_state->set('groups', $groups);
    $form_state->setRebuild();
  }

  /**
   * Build the fieldset UI for one group.
   *
   * @param array<mixed> $form
   * @param array<string,mixed> $group
   */
  protected function buildGroupFieldset(array &$form, FormStateInterface $form_state, string $uuid, array $group, string $bundle_id): void {
    // Ensure nested structures exist to satisfy static analysis and prevent notices.
    if (!isset($form['columns']) || !is_array($form['columns'])) { $form['columns'] = []; }
    if (!isset($form['columns']['right']) || !is_array($form['columns']['right'])) { $form['columns']['right'] = []; }
    if (!isset($form['columns']['right']['groups_wrapper']) || !is_array($form['columns']['right']['groups_wrapper'])) { $form['columns']['right']['groups_wrapper'] = []; }
    if (!isset($form['columns']['right']['groups_wrapper']['groups']) || !is_array($form['columns']['right']['groups_wrapper']['groups'])) { $form['columns']['right']['groups_wrapper']['groups'] = []; }

    $name_value = '';
    if (isset($group['name']) && is_scalar($group['name']) && $group['name'] !== '') {
      $name_value = (string) $group['name'];
    }
    elseif (isset($group['title']) && is_scalar($group['title'])) {
      $name_value = (string) $group['title'];
    }

    $form['columns']['right']['groups_wrapper']['groups'][$uuid] = [
      '#type' => 'fieldset',
      '#attributes' => ['class' => ['taxonomy-term-config-groups__group']],
    ];

    $form['columns']['right']['groups_wrapper']['groups'][$uuid]['header'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['taxonomy-term-config-groups__group-header']],
    ];
    $form['columns']['right']['groups_wrapper']['groups'][$uuid]['header']['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name'),
      '#default_value' => $name_value,
      '#required' => TRUE,
      '#size' => 40,
      '#maxlength' => 255,
    ];
    $form['columns']['right']['groups_wrapper']['groups'][$uuid]['header']['id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ID'),
      '#default_value' => (isset($group['id']) && is_scalar($group['id'])) ? (string) $group['id'] : '',
      '#size' => 30,
      '#maxlength' => 128,
      '#disabled' => TRUE,
    ];

    $form['columns']['right']['groups_wrapper']['groups'][$uuid]['terms_icicle'] = [
      '#type' => 'terms_icicle',
      '#init_full' => FALSE,
      '#default_value' => isset($group['terms_icicle']) ? $group['terms_icicle'] : '',
      '#icicle_key' => (string) $uuid,
    ];

    $form['columns']['right']['groups_wrapper']['groups'][$uuid]['details'] = [
      '#type' => 'details',
      '#title' => $this->t('Group configuration'),
      '#open' => FALSE,
    ];
    $form['columns']['right']['groups_wrapper']['groups'][$uuid]['details']['fields'] = [
      '#type' => 'container',
      '#parents' => ['columns', 'right', 'groups_wrapper', 'groups', $uuid, 'details', 'fields'],
    ];

    // Render dynamic fields via EntityFormDisplay, excluding label and field_terms.
    try {
      $storage = $this->entityTypeManager->getStorage('taxonomy_group');
      $entity = NULL;
      if ($bundle_id !== '') {
        $loaded = $storage->loadByProperties(['uuid' => (string) $uuid, 'type' => $bundle_id]);
        if (!empty($loaded)) {
          $entity = reset($loaded);
        }
      }
      if (!$entity) {
        $entity = $storage->create(['type' => $bundle_id, 'uuid' => (string) $uuid]);
      }
      if ($entity instanceof \Drupal\Core\Entity\FieldableEntityInterface) {
        if ($name_value !== '') { $entity->set('label', $name_value); }
        $display = EntityFormDisplay::collectRenderDisplay($entity, 'default');
        if ($display->getComponent('label')) { $display->removeComponent('label'); }
        if ($display->getComponent('field_terms')) { $display->removeComponent('field_terms'); }
        $subform = &$form['columns']['right']['groups_wrapper']['groups'][$uuid]['details']['fields'];
        $subform_state = SubformState::createForSubform($subform, $form, $form_state);
        $display->buildForm($entity, $subform, $subform_state);
      }
    }
    catch (\Throwable $e) {
      $this->logger('taxonomy_term_config_groups')->warning('Dynamic fields failed for group @uuid: @msg', [
        '@uuid' => (string) $uuid,
        '@msg' => $e->getMessage(),
      ]);
    }

    $form['columns']['right']['groups_wrapper']['groups'][$uuid]['remove'] = [
      '#type' => 'submit',
      '#value' => $this->t('Remove group'),
      '#name' => 'remove_group__' . $uuid,
      '#submit' => ['::removeGroupSubmit'],
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::groupsAjaxCallback',
        'wrapper' => 'taxonomy-term-config-groups__columns',
      ],
      '#attributes' => ['class' => ['button--remove-grouping']],
    ];
  }

  /**
   * Get the subform container for dynamic fields of a group, if present.
   *
   * @param array<string,mixed> $form
   * @return array<string,mixed>|null
   */
  protected function &getGroupFieldsSubform(array &$form, string $uuid): array|null {
    $null = NULL;
    if (!isset($form['columns']) || !is_array($form['columns'])) { return $null; }
    $columns = &$form['columns'];
    if (!isset($columns['right']) || !is_array($columns['right'])) { return $null; }
    $right = &$columns['right'];
    if (!isset($right['groups_wrapper']) || !is_array($right['groups_wrapper'])) { return $null; }
    $wrapper = &$right['groups_wrapper'];
    if (!isset($wrapper['groups']) || !is_array($wrapper['groups'])) { return $null; }
    $groups = &$wrapper['groups'];
    if (!isset($groups[$uuid]) || !is_array($groups[$uuid])) { return $null; }
    $g = &$groups[$uuid];
    if (!isset($g['details']) || !is_array($g['details'])) { return $null; }
    $details = &$g['details'];
    if (!isset($details['fields']) || !is_array($details['fields'])) { return $null; }
    $fields = &$details['fields'];
    /** @var array<string,mixed> $fields */
    return $fields;
  }

  /**
   * Apply dynamic field form values from the group subform onto the entity.
   *
   * @param array<string,mixed> $form
   */
  protected function applyDynamicFields(\Drupal\Core\Entity\FieldableEntityInterface $entity, string $uuid, array &$form, FormStateInterface $form_state): void {
    $subform = &$this->getGroupFieldsSubform($form, $uuid);
    if (is_array($subform)) {
      $display = EntityFormDisplay::collectRenderDisplay($entity, 'default');
      if ($display->getComponent('label')) { $display->removeComponent('label'); }
      if ($display->getComponent('field_terms')) { $display->removeComponent('field_terms'); }
      $display->extractFormValues($entity, $subform, $form_state);
    }
  }

  /**
   * Submit handler for removing a grouping from form_state by UUID.
   */
  /**
   * Submit handler for removing a grouping from form_state by UUID.
   *
   * @param array<string, mixed> $form
   *   The form render array.
   */
  public function removeGroupSubmit(array &$form, FormStateInterface $form_state): void {
    $trigger = $form_state->getTriggeringElement();
    $name = isset($trigger['#name']) ? (string) $trigger['#name'] : '';
    $uuid = NULL;
    if (preg_match('/^remove_group__(.+)$/', $name, $m)) {
      $uuid = $m[1];
    }

    $groups = $form_state->get('groups');
    if (!is_array($groups)) {
      $groups = [];
    }

    if ($uuid !== NULL && isset($groups[$uuid])) {
      unset($groups[$uuid]);
    }

    $form_state->set('groups', $groups);
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string,mixed> $form The form render array.
   * @param-out array<string,mixed> $form
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Determine vocabulary and corresponding taxonomy_group bundle id.
    $vocabulary = $this->getVocabularyFromRoute();
    $vid = $vocabulary ? $vocabulary->id() : NULL;
    if (!$vid) {
      $this->messenger()->addError($this->t('Unable to determine vocabulary for saving.'));
      return;
    }
    $bundle_id = $this->getBundleId((string) $vid);

    // Extract submitted groups from the form values. These live under
    // columns/right/groups_wrapper/groups and are keyed by UUID.
    $values_groups = (array) $form_state->getValue(['columns', 'right', 'groups_wrapper', 'groups']) ?: [];
    /** @var array<string, array<string, mixed>> $values_groups */

    // Build desired set: uuid => [label, tids[]]. Ignore empty names.
    $desired = [];
    foreach ($values_groups as $uuid => $row) {
      if ($uuid === '') { continue; }
      $header = (array) ($row['header'] ?? []);
      $name = (isset($header['name']) && is_scalar($header['name'])) ? trim((string) $header['name']) : '';
      if ($name === '') { continue; }
      $ti = $row['terms_icicle'] ?? null;
      if (!is_array($ti) && !is_string($ti)) { $ti = null; }
      $tids = $this->parseIcicleIds($ti);
      $desired[$uuid] = ['label' => $name, 'tids' => $tids];
    }

    $result = $this->saveDesiredGroups($form, $form_state, $bundle_id, $desired);

    $this->messenger()->addStatus($this->t('Saved groupings. @created created, @updated updated, @deleted deleted.', [
      '@created' => $result['created'],
      '@updated' => $result['updated'],
      '@deleted' => $result['deleted'],
    ]));

    $form_state->setRedirect('taxonomy_term_config_groups.grouping_form', [
      'taxonomy_vocabulary' => (string) $vid,
    ]);
  }

  /**
   * Fetch the vocabulary entity from the current route.
   */
  protected function getVocabularyFromRoute(): ?\Drupal\taxonomy\VocabularyInterface {
    $vocabulary = $this->currentRouteMatch->getParameter('taxonomy_vocabulary');
    return $vocabulary instanceof \Drupal\taxonomy\VocabularyInterface ? $vocabulary : NULL;
  }

  /**
   * Initialize groups state on first build and return current groups array.
   *
   * @param string|null $vid
   * @param string|null $bundle_id
   * @return array<string, array<string,mixed>>
   */
  protected function initializeGroupsOnFirstBuild(FormStateInterface $form_state, ?string $vid, ?string $bundle_id): array {
    $groups = $form_state->get('groups');
    $initialized = (bool) $form_state->get('groups_initialized');
    if ($initialized && is_array($groups)) {
      /** @var array<int|string, array<string,mixed>> $groups */
      return $this->normalizeGroups($groups);
    }
    $groups = [];
    if ($vid) {
      $bid = $bundle_id ?: ('vocab_' . $vid);
      $loaded = $this->loadExistingGroups($bid);
      $assigned = $this->collectAssignedTids(array_values($loaded));
      $form_state->set('assigned_tids', $assigned);
      foreach ($loaded as $uuid => $entity) {
        $label = (string) $entity->label();
        $machine = $entity->hasField('machine_name') ? $entity->get('machine_name')->getString() : '';
        $tids = [];
        if ($entity->hasField('field_terms')) {
          foreach ((array) $entity->get('field_terms')->getValue() as $item) {
            if (is_array($item) && isset($item['target_id']) && is_scalar($item['target_id'])) {
              $tid = (int) $item['target_id'];
              if ($tid > 0) { $tids[] = $tid; }
            }
          }
        }
        $forest = array_map(static fn (int $tid) => ['id' => $tid, 'name' => '', 'children' => []], $tids);
        $groups[$uuid] = [
          'name' => $label,
          'id' => $machine,
          'terms_icicle' => $forest,
        ];
      }
    }
    $form_state->set('groups', $groups);
    $form_state->set('groups_initialized', TRUE);
    return $groups;
  }

  /**
   * Derive the taxonomy_group bundle id from a vocabulary id.
   */
  protected function getBundleId(string $vid): string {
    try {
      return (string) \taxonomy_term_config_groups_bundle_id_from_vid($vid);
    }
    catch (\Throwable $e) {
      return 'vocab_' . $vid;
    }
  }

  /**
   * Ensure valid groups data before submit.
   *
   * @param array<mixed> $form
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    $values_groups = (array) $form_state->getValue(['columns', 'right', 'groups_wrapper', 'groups']) ?: [];
    $names = [];
    $term_to_group = [];
    foreach ($values_groups as $uuid => $row) {
      if (!is_string($uuid) || $uuid === '') { continue; }
      if (!is_array($row)) { continue; }
      $header = (array) ($row['header'] ?? []);
      $name = (isset($header['name']) && is_scalar($header['name'])) ? trim((string) $header['name']) : '';
      if ($name === '') { continue; }
      if (isset($names[strtolower($name)])) {
        $form_state->setError($form['columns']['right']['groups_wrapper']['groups'][$uuid]['header']['name'], $this->t('Group name "@name" is duplicated.', ['@name' => $name]));
      }
      else {
        $names[strtolower($name)] = $uuid;
      }
      // Duplicate term detection across groups.
      $ti = $row['terms_icicle'] ?? null;
      if (!is_array($ti) && !is_string($ti)) { $ti = null; }
      $tids = $this->parseIcicleIds($ti);
      foreach ($tids as $tid) {
        if (isset($term_to_group[$tid]) && $term_to_group[$tid] !== $uuid) {
          $form_state->setError($form['columns']['right']['groups_wrapper']['groups'][$uuid]['terms_icicle'], $this->t('Term @tid is assigned to multiple groups.', ['@tid' => $tid]));
        }
        else {
          $term_to_group[$tid] = $uuid;
        }
      }
    }
  }

  /**
   * Save desired group state to storage and return change counters.
   *
   * @param array<string,mixed> $form
   * @param array<string, array{label:string,tids:array<int>}> $desired
   * @return array{created:int,updated:int,deleted:int}
   */
  protected function saveDesiredGroups(array &$form, FormStateInterface $form_state, string $bundle_id, array $desired): array {
    $storage = $this->entityTypeManager->getStorage('taxonomy_group');
    $by_uuid = $this->loadExistingGroups($bundle_id);

    $created = 0; $updated = 0; $deleted = 0;

    // Delete entities not desired anymore.
    foreach ($by_uuid as $uuid => $entity) {
      if (!isset($desired[$uuid])) {
        $storage->delete([$entity]);
        unset($by_uuid[$uuid]);
        $deleted++;
      }
    }

    foreach ($desired as $uuid => $info) {
      $label = (string) $info['label'];
      $new_tids = array_values(array_map(static fn ($v): int => (int) $v, $info['tids']));

      if (isset($by_uuid[$uuid])) {
        /** @var \Drupal\Core\Entity\FieldableEntityInterface $entity */
        $entity = $by_uuid[$uuid];
        $label_changed = ((string) $entity->label() !== $label);
        $existing_tids = [];
        if ($entity->hasField('field_terms')) {
          $values = (array) $entity->get('field_terms')->getValue();
          foreach ($values as $item) {
            if (is_array($item) && isset($item['target_id']) && is_scalar($item['target_id'])) {
              $existing_tids[] = (int) $item['target_id'];
            }
          }
        }
        sort($existing_tids, SORT_NUMERIC);
        $compare_new = array_values(array_unique($new_tids));
        sort($compare_new, SORT_NUMERIC);
        $terms_changed = ($existing_tids !== $compare_new);

        if ($label_changed) { $entity->set('label', $label); }
        if ($terms_changed) { $entity->set('field_terms', array_map(static fn ($tid) => ['target_id' => (int) $tid], $new_tids)); }

        $this->applyDynamicFields($entity, (string) $uuid, $form, $form_state);

        $entity->save();
        $updated++;
      }
      else {
        /** @var \Drupal\Core\Entity\FieldableEntityInterface $entity */
        $entity = $storage->create(['type' => $bundle_id, 'uuid' => (string) $uuid, 'label' => $label]);
        if (!empty($new_tids)) {
          $entity->set('field_terms', array_map(static fn ($tid) => ['target_id' => (int) $tid], $new_tids));
        }
        $this->applyDynamicFields($entity, (string) $uuid, $form, $form_state);
        $entity->save();
        $created++;
      }
    }

    return ['created' => $created, 'updated' => $updated, 'deleted' => $deleted];
  }

  /**
   * Derive the left icicle AJAX refresh wrapper id.
   * Placeholder for future sub-builder refactors (attachPageAssets/buildColumns/etc.).
   */
  /**
   * Placeholder to attach page assets.
   *
   * @param array<string,mixed> $form
   * @param array<int, array<string,mixed>> $terms_tree
   */
  protected function attachPageAssets(array &$form, ?string $vid, string $vocab_label, array $terms_tree): void {}

  /**
   * Placeholder to build columns wrapper.
   *
   * @param array<string,mixed> $form
   */
  protected function buildColumns(array &$form): void {}

  /**
   * Placeholder to build left column.
   *
   * @param array<string,mixed> $form
   */
  protected function buildLeftColumn(array &$form): void {}

  /**
   * Placeholder to build right column.
   *
   * @param array<string,mixed> $form
   */
  protected function buildRightColumn(array &$form): void {}

  /**
   * Parse the value from a terms_icicle element and return unique TIDs.
   *
   * @param string|array<mixed>|null $value
   * @return list<int>
   */
  protected function parseIcicleIds(string|array|null $value): array {
    $forest = is_array($value)
      ? $value
      : (is_string($value) ? (json_decode(trim($value), TRUE) ?: []) : []);
    if (is_array($forest) && $forest !== [] && array_keys($forest) !== range(0, count($forest) - 1)) {
      $forest = [$forest];
    }
    $ids = [];
    $stack = is_array($forest) ? $forest : [];
    while ($stack) {
      $node = array_pop($stack);
      if (!is_array($node)) { continue; }
      if (isset($node['id']) && is_scalar($node['id'])) { $ids[] = (int) $node['id']; }
      elseif (isset($node['tid']) && is_scalar($node['tid'])) { $ids[] = (int) $node['tid']; }
      if (!empty($node['children']) && is_array($node['children'])) {
        foreach ($node['children'] as $child) { $stack[] = $child; }
      }
    }
    $ids = array_values(array_unique(array_map(static fn ($v): int => (int) $v, $ids)));
    $ids = array_values(array_filter($ids, static fn (int $v): bool => $v > 0));
    sort($ids, SORT_NUMERIC);
    return $ids;
  }

}
